package jp.ill.photon.util.mail;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.mail.EmailException;
import org.apache.commons.mail.SimpleEmail;

import jp.ill.photon.util.UtilTools;

/**
 * 汎用SMTPメール送信クラス
 *
 * @author h_tanaka
 *
 */
public class SmtpMailSender {

	/** メール送信サーバアドレス */
	private String smtpHost;

	/** メール送信サーバポート */
	private int smtpPort;

	/** メール送信認証用ユーザー名 */
	private String smtpUserName;

	/** メール送信認証用パスワード */
	private String smtpPassword;

	/**
	 * 認証付SMTPメール送信オブジェクト生成
	 *
	 * @param smtpHost
	 * @param smtpPort
	 * @param smtpUserName
	 * @param smtpPassword
	 */
	public SmtpMailSender(String smtpHost, int smtpPort, String smtpUserName,
			String smtpPassword) {
		this.smtpHost = smtpHost;
		this.smtpPort = smtpPort;
		this.smtpUserName = smtpUserName;
		this.smtpPassword = smtpPassword;
	}

	/**
	 * 認証無しSMTPメール送信オブジェクト生成
	 *
	 * @param smtpHost
	 * @param smtpPort
	 */
	public SmtpMailSender(String smtpHost, int smtpPort) {
		this.smtpHost = smtpHost;
		this.smtpPort = smtpPort;
		this.smtpUserName = "";
		this.smtpPassword = "";
	}

	/**
	 * 簡易メール送信（1送信先）
	 *
	 * @param toAddress
	 * @param toAddressName
	 * @param subject
	 * @param contents
	 * @param fromAddress
	 * @param fromAddressName
	 * @param replyAddress
	 * @return
	 * @throws EmailException
	 */
	public boolean sendMail(String toAddress,
							String toAddressName,
							String subject,
							String contents,
							String fromAddress,
							String fromAddressName,
							String replyAddress)
			throws EmailException {
		Map<String, String> toAddresses = new HashMap<>();
		toAddresses.put(toAddress, toAddressName);
		return sendMail(toAddresses, subject, contents, fromAddress,
				fromAddressName, replyAddress);
	}

	/**
	 * 簡易メール送信（複数送信先）
	 *
	 * @param toAddresses
	 * @param subject
	 * @param contents
	 * @param fromAddress
	 * @param fromAddressName
	 * @param replyAddress
	 * @return
	 * @throws EmailException
	 */
	public boolean sendMail(Map<String, String> toAddresses,
							String subject,
							String contents,
							String fromAddress,
							String fromAddressName,
							String replyAddress)
			throws EmailException {
		Map<String, String> ccAddresses = null;
		Map<String, String> bccAddresses = null;
		String charSet = "UTF8";
		return sendMail(toAddresses, ccAddresses, bccAddresses, subject,
				contents, charSet, fromAddress, fromAddressName, replyAddress);
	}

	/**
	 * フル設定メール送信
	 *
	 * @param toAddresses
	 * @param ccAddresses
	 * @param bccAddresses
	 * @param subject
	 * @param contents
	 * @param charSet
	 * @param fromAddress
	 * @param fromAddressName
	 * @param replyAddress
	 * @return
	 * @throws EmailException
	 */
	public boolean sendMail(Map<String, String> toAddresses,
							Map<String, String> ccAddresses,
							Map<String, String> bccAddresses,
							String subject,
							String contents,
							String charSet,
							String fromAddress,
							String fromAddressName,
							String replyAddress)
			throws EmailException {

		boolean result = false;

		SimpleEmail email = new SimpleEmail();

		// SMTPユーザー名とパスワードが設定されていたら認証
		if (!UtilTools.isEmpty(getSmtpUserName())
				&& !UtilTools.isEmpty(getSmtpPassword())) {
			email.setAuthentication(getSmtpUserName(), getSmtpPassword());
		}

		email.setHostName(getSmtpHost());
		email.setSmtpPort(getSmtpPort());
		email.setCharset(charSet);
		email.setFrom(fromAddress, fromAddressName);
		email.addReplyTo(replyAddress);
		email.setSubject(subject);
		email.setMsg(contents);

		// 送信先をセット
		for (Entry<String, String> entry : toAddresses.entrySet()) {
			email.addTo(entry.getKey(), entry.getValue(), charSet);
		}

		// CC送信先をセット
		if (ccAddresses != null && ccAddresses.size() > 0) {
			for (Entry<String, String> entry : ccAddresses.entrySet()) {
				if (entry.getKey().length() > 0) {
					email.addCc(entry.getKey(), entry.getValue(), charSet);
				}
			}
		}

		// BCC送信先をセット
		if (bccAddresses != null && bccAddresses.size() > 0) {
			for (Entry<String, String> entry : bccAddresses.entrySet()) {
				if (entry.getKey().length() > 0) {
					email.addBcc(entry.getKey(), entry.getValue(), charSet);
				}
			}
		}

		email.send();

		result = true;

		return result;
	}

	public String getSmtpHost() {
		return smtpHost;
	}

	public void setSmtpHost(String smtpHost) {
		this.smtpHost = smtpHost;
	}

	public int getSmtpPort() {
		return smtpPort;
	}

	public void setSmtpPort(int smtpPort) {
		this.smtpPort = smtpPort;
	}

	public String getSmtpUserName() {
		return smtpUserName;
	}

	public void setSmtpUserName(String smtpUserName) {
		this.smtpUserName = smtpUserName;
	}

	public String getSmtpPassword() {
		return smtpPassword;
	}

	public void setSmtpPassword(String smtpPassword) {
		this.smtpPassword = smtpPassword;
	}
}
