package jp.ill.photon.util.mail;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringFormatUtil;
import jp.ill.photon.util.StringUtil;

public class MailUtil {

	/**
	 * メール送信用アドレスを取得する
	 *
	 * @param addressMap アドレスマップ
	 * @param context ActionDto
	 * @return メール送信用アドレス
	 */
	@SuppressWarnings("rawtypes")
	public static Map<String, String> getAddressFormat(	Map<String, Object> addressMap,
														ModuleContext context) {
		if (addressMap == null || addressMap.size() == 0) {
			return new HashMap<String, String>();
		}
		Map<String, String> rtnMap = new HashMap<String, String>();
		List<String> addressList = new ArrayList<String>();
		for (Map.Entry<String, Object> param : addressMap.entrySet()) {
			Map content = (Map) param.getValue();
			String address = StringUtil.defaultString(ParamUtil
					.getParamValueByType(String.valueOf(content.get("type")),
							content.get("val"), context.getDto()));
			if (address != null) {
				addressList.add(address);
			}
		}
		rtnMap = addressListToMap(addressList);

		return rtnMap;
	}

	/**
	 * メール送信用アドレスを取得する
	 *
	 * @param addressMap アドレスマップ
	 * @param context ActionDto
	 * @return メール送信用アドレス
	 * 
	 */
	public static Map<String, String> getAddressFormat(Map<String, Object> addressMap) {
		if (addressMap == null || addressMap.size() == 0) {
			return new HashMap<String, String>();
		}
		Map<String, String> rtnMap = new HashMap<String, String>();
		List<String> addressList = new ArrayList<String>();
		for (Map.Entry<String, Object> param : addressMap.entrySet()) {
			String address = StringUtil.defaultString(param.getValue());
			if (address != null) {
				addressList.add(address);
			}
		}
		rtnMap = addressListToMap(addressList);

		return rtnMap;
	}

	/**
	 * メールアドレスのリストをMailSender用のMapに変換する
	 * 
	 * @param addressList
	 * @return
	 */
	public static Map<String, String> addressListToMap(List<String> addressList) {
		Map<String, String> addressMap = new LinkedHashMap<String, String>();
		if (addressList == null || addressList.size() == 0) {
			return addressMap;
		}

		for (String address : addressList) {
			addressMap.put(address, null);
		}

		return addressMap;
	}

	/**
	 * メールテンプレートを置換する
	 * 
	 * @param template
	 * @param dataMap
	 * @return
	 */
	public static String processMailTemplate(	String template,
												Map<String, Object> dataMap) {
		return processMailTemplate(template, dataMap,
				new HashMap<String, String>());
	}

	/**
	 * メールテンプレートを置換する
	 * 
	 * @param template
	 * @param dataMap
	 * @return
	 */
	public static String processMailTemplate(	String template,
												Map<String, Object> dataMap,
												Map<String, String> formatMap) {
		String result = new String(template);

		// ループ用ロジック
		Pattern loopPattern = Pattern.compile("\\{%(.+?)%\\}.*?\\{/%\\1%\\}",
				Pattern.MULTILINE | Pattern.DOTALL);
		Pattern loopReplacePattern = Pattern.compile(
				"\\{%(.+?)%\\}(.*?)\\{/%\\1%\\}",
				Pattern.MULTILINE | Pattern.DOTALL);

		Matcher loopMatch = loopPattern.matcher(template);
		while (loopMatch.find()) {
			// リスト名
			String loopName = loopMatch.group(1);
			String loopText = loopMatch.group();
			String changedLoopText = loopMatch.group();

			Map<String, Object> inputMap = new HashMap<>();

			@SuppressWarnings("unchecked")
			List<Map<String, Object>> loopDataList = (List<Map<String, Object>>) dataMap
					.getOrDefault(loopName, new ArrayList<>());
			for (Map<String, Object> detail : loopDataList) {

				// ループ内変数の変換
				for (Entry<String, Object> dataMapObject : dataMap.entrySet()) {
					// Key名がリスト名で修飾されている場合は、ループ内変数とする
					if (dataMapObject.getKey().startsWith(loopName + ".")) {
						inputMap.put(dataMapObject.getKey(), detail
								.getOrDefault(dataMapObject.getValue(), ""));

					} else {
						inputMap.put(dataMapObject.getKey(),
								dataMapObject.getValue());

					}
				}

				// リストブロック内の変数置換
				String tmpLoopText = processMailTemplateReplaceSection(
						changedLoopText, inputMap, formatMap);

				// リストブロックタグを除去
				changedLoopText = loopReplacePattern.matcher(tmpLoopText)
						.replaceFirst("$2");
				changedLoopText = changedLoopText + loopText;
			}
			changedLoopText = changedLoopText.replace(loopText, "");
			result = result.replace(loopText, changedLoopText);

		}

		// リスト外要素を置換
		result = processMailTemplateReplaceSection(result, dataMap, formatMap);

		return result;
	}

	/**
	 * メールテンプレートを置換する
	 * 
	 * @param template
	 * @param dataMap
	 * @return
	 */
	public static String processMailTemplateReplaceSection(	String template,
															Map<String, Object> dataMap,
															Map<String, String> formatMap) {
		String result = new String(template);
		// セクションの表示・非表示を判定し、置換（セクションタグの削除、セクション接頭辞の削除）
		Pattern sectionPattern = Pattern.compile("\\{#(.+?)#\\}.*?\\{/#\\1#\\}",
				Pattern.MULTILINE | Pattern.DOTALL);
		Pattern sectionReplacePattern = Pattern.compile(
				"\\{#(.+?)#\\}(.*?)\\{/#\\1#\\}",
				Pattern.MULTILINE | Pattern.DOTALL);
		boolean hasMatch = false;

		while (true) {
			Pattern sectionInnerTagPattern = null;
			String sectionName = null;
			String sectionInnerTagName = null;
			String sectionText = null;
			String sectionInnerTagText = null;
			Matcher sectionMatch = sectionPattern.matcher(result);
			hasMatch = false;

			Matcher sectionInnerTagMatch = null;
			boolean sectionHasValue = false;
			while (sectionMatch.find()) {
				hasMatch = true;
				sectionName = sectionMatch.group(1);
				sectionText = sectionMatch.group();
				sectionInnerTagPattern = Pattern
						.compile("\\{\\{" + sectionName + "\\.(.*?)\\}\\}");
				sectionInnerTagMatch = sectionInnerTagPattern
						.matcher(sectionText);
				sectionHasValue = false;
				while (sectionInnerTagMatch.find()) {
					sectionInnerTagName = sectionInnerTagMatch.group(1);
					sectionInnerTagText = sectionInnerTagMatch.group();
					if (dataMap.getOrDefault(sectionInnerTagName, "") != null
							&& !"".equals(dataMap
									.getOrDefault(sectionInnerTagName, ""))) {
						sectionHasValue = true;
						break;
					}
				}

				if (sectionHasValue) {
					sectionInnerTagMatch.reset();
					String replacedSectionText = new String(sectionText);
					while (sectionInnerTagMatch.find()) {
						sectionInnerTagName = sectionInnerTagMatch.group(1);
						replacedSectionText = replacedSectionText.replace(
								sectionInnerTagMatch.group(),
								"{{" + sectionInnerTagName + "}}");
					}
					replacedSectionText = sectionReplacePattern
							.matcher(replacedSectionText).replaceFirst("$2");
					result = result.replace(sectionText, replacedSectionText);
				} else {
					result = result.replace(sectionText, "");
				}
			}

			if (!hasMatch) {
				break;
			}
		}

		// 各変数の置換
		Pattern tagPattern = Pattern.compile("\\{\\{(.*?)\\}\\}");
		Matcher tagMatch = tagPattern.matcher(result);
		String value = null;
		while (tagMatch.find()) {
			value = StringUtil.defaultString(dataMap.getOrDefault(
					tagMatch.group(1), "[" + tagMatch.group(1) + "]"), "");
			value = StringFormatUtil.formatByType(
					formatMap.getOrDefault(tagMatch.group(1), ""), value);
			result = result.replace(tagMatch.group(), value);
		}

		return result;
	}
}
