package jp.ill.photon.util;

import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.sql.Date;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;

import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.model.CharacterConst.TextConvert;
import jp.ill.photon.module.ModuleContext;

public class StringUtil {

	/**
	 * 文字列がnullの場合、空文字に変換する
	 *
	 * @param str 文字列
	 * @return 変換後の文字列
	 */
	public static String nullToBlank(String str) {
		if (str == null) {
			return "";
		}
		return str;
	}

	/**
	 * リストを指定の区切り文字で文字列結合する.
	 *
	 * @param targetList 対象リスト
	 * @param divideChar 区切り文字
	 * @return String 文字列
	 */
	public static String join(ArrayList<String> targetList, String divideChar) {
		StringBuffer returnValue = new StringBuffer();
		for (String elementValue : targetList) {
			if (returnValue.length() > 0) {
				returnValue.append(divideChar);
			}
			returnValue.append(elementValue);
		}
		return returnValue.toString();
	}

	/**
	 * 文字列をバイト数で分割<br>
	 * 文字コード指定のエラーの場合は要素数1(空文字)の文字列配列を返します。<br>
	 * 分割する文字列に1byteより大きい文字が含まれる場合に、
	 * 分割するバイト数に1を指定しないように注意してください。(1文字ずつの分割になります)<br>
	 *
	 * @param str 分割する文字列
	 * @param len 分割するバイト数
	 * @param charset 文字コード
	 * @return 分割後の文字列配列
	 */
	public static String[] lengthSplitBytes(String str,
											int len,
											String charset) {
		String[] result = { "" };
		if (str == null)
			return result;
		if (len <= 0)
			return result;
		try {
			// 分割後の文字列
			StringBuffer temp = new StringBuffer();
			LinkedList<String> list = new LinkedList<String>();

			// 取得した文字までのバイト数
			int nowLength = 0;
			int charLength = 0;
			String charString = null;

			for (int i = 0; i < str.length(); i++) {
				// 1文字取得
				charString = Character.toString(str.charAt(i));
				// 1文字のバイト数
				charLength = charString.getBytes(charset).length;
				// バイト数を加算
				nowLength += charLength;
				// 1単位分のバイト数を超えているかチェック
				if (nowLength > len) {
					// 1単位のバイト数を超えた場合
					// 今までの文字列をリストに格納
					if (!"".equals(temp.toString())) {
						list.add(temp.toString());
					}
					// 文字列を初期化
					temp = new StringBuffer();
					// 現在バイト数を初期化
					nowLength = charLength;
				}
				// 文字列を設定
				temp.append(charString);
			}
			// 今までの文字列をリストに格納
			list.add(temp.toString());

			// リストから配列を作成
			result = list.toArray(new String[list.size()]);
		} catch (Exception e) {
		}
		// 結果を返します
		return result;
	}

	/* Aladdin-EC 3rdリリース #6718 START */
	/**
	 * 半角→全角
	 *
	 */
	public static String toZenkaku(String val) {

		StringBuffer ret = new StringBuffer();
		int len = val.length();

		for (int i = 0; i < len; i++) {

			char c = val.charAt(i);

			if (TextConvert.HALF_NUM.indexOf(c) >= 0
					|| TextConvert.HALF_ALPHA_UPPER.indexOf(c) >= 0
					|| TextConvert.HALF_ALPHA_LOWER.indexOf(c) >= 0) {

				// 半角数字、アルファベットのとき
				ret.append((char) (c + ('Ａ' - 'A')));

			} else if (TextConvert.HALF_KANA.indexOf(c) >= 0) {

				// 濁点、半濁点付きだった場合
				if (len > (i + 1) && TextConvert.HALF_KANA_KIGOU
						.indexOf(val.charAt(i + 1)) >= 0) {

					String dakuonStr = TextConvert.HALF_KANA_DAKUON_TABLE.get(
							new String(new char[] { c, val.charAt(i + 1) }));
					if (dakuonStr != null) {
						// 濁点、半濁点付きのカナが取得できた場合は、次の次の文字へ
						ret.append(dakuonStr);
						i++;
						continue;
					}

				}

				// 通常のカナ
				ret.append(TextConvert.HALF_KANA_TABLE
						.get(new String(new char[] { c })));

			} else if (TextConvert.HALF_KIGOU.indexOf(c) >= 0) {

				// 半角記号
				ret.append(TextConvert.HALF_KIGOU_TABLE
						.get(new String(new char[] { c })));

			} else {

				// その他
				// 変換せずにそのまま返す
				ret.append(c);

			}

		}

		return ret.toString();

	}

	/**
	 * 0埋め処理.
	 *
	 * @param str
	 * @param length
	 * @return
	 */
	public static String zeroPadding(String str, int length) {

		if (str == null || str.length() == 0 || str.length() > length) {
			return str;
		}

		StringBuffer sb = new StringBuffer();
		for (int i = str.length(); i < length; i++) {
			sb.append("0");
		}
		sb.append(str);
		return sb.toString();
	}

	/**
	 * 指定文字で文字埋め処理
	 *
	 * @param baseStr
	 * @param padStr
	 * @param byteLength
	 * @return
	 */
	public static String leftPadding(	String baseStr,
										String padStr,
										int byteLength) {

		if (baseStr == null || padStr == null || padStr.length() == 0) {
			return baseStr;
		}

		int baseByteLength = 0;
		int padByteLength = 0;
		try {
			baseByteLength = baseStr.getBytes("windows-31j").length;
			padByteLength = padStr.getBytes("windows-31j").length;
		} catch (UnsupportedEncodingException e) {
		}

		if (baseByteLength > byteLength) {
			return baseStr;
		}

		StringBuffer sb = new StringBuffer();
		for (int i = baseByteLength; i < byteLength; i += padByteLength) {
			sb.append(padStr);
		}

		sb.append(baseStr);
		return sb.toString();
	}

	/**
	 * 数値文字列変換<br>
	 * フォーマット変換によりカンマを入れた数値の文字列のカンマを除去します。
	 *
	 * @param str
	 * @return
	 */
	public static String toNumString(String str) {
		if (StringUtils.isEmpty(str)) {
			return null;
		}
		str = str.replaceAll(",", "");
		return str;
	}

	/**
	 * 文字列をInteger型に変換
	 *
	 * @param obj オブジェクト
	 * @param defaultDcm デフォルト値
	 * @return Integer(数値以外:デフォルト値)
	 */
	public static Integer toInteger(Object obj, Integer defaultValue) {
		if (obj == null) {
			return defaultValue;
		}
		try {
			return Integer.parseInt((String) obj);
		} catch (Exception e) {
			return defaultValue;
		}
	}

	/**
	 * 文字列をDouble型に変換
	 *
	 * @param obj オブジェクト
	 * @param defaultDcm デフォルト値
	 * @return Integer(数値以外:デフォルト値)
	 */
	public static Double toDouble(Object obj, Double defaultValue) {
		if (obj == null) {
			return defaultValue;
		}
		try {
			return Double.parseDouble((String) obj);
		} catch (Exception e) {
			return defaultValue;
		}
	}

	/**
	 * 文字列をDouble型に変換
	 *
	 * @param obj オブジェクト
	 * @return Double(数値以外:デフォルト値)
	 */
	public static Double toDouble(Object obj) {
		if (obj == null) {
			return new Double(0.0);
		}
		try {
			return Double.parseDouble((String) obj);
		} catch (Exception e) {
			return new Double(0.0);
		}
	}

	/**
	 * 文字列をInteger型に変換
	 *
	 * @param obj オブジェクト
	 * @return Integer(数値以外:0)
	 */
	public static Integer toInteger(Object obj) {
		return StringUtil.toInteger(obj, 0);
	}

	/**
	 * 文字列をBigDecimal型に変換
	 *
	 * @param obj オブジェクト
	 * @param defaultDcm デフォルト値
	 * @return BigDecimal(数値以外:デフォルト値)
	 */
	public static BigDecimal toBigDecimal(Object obj, BigDecimal defaultValue) {
		if (obj == null) {
			return defaultValue;
		}
		try {
			return new BigDecimal((String) obj);
		} catch (Exception e) {
			return defaultValue;
		}
	}

	/**
	 * 文字列をBigDecimal型に変換
	 *
	 * @param str 文字列
	 * @param defaultDcm デフォルト値
	 * @return BigDecimal(数値以外:デフォルト値)
	 */
	public static BigDecimal toBigDecimal(String str, BigDecimal defaultDcm) {
		if (StringUtils.isEmpty(str)) {
			return defaultDcm;
		}
		try {
			return new BigDecimal(str);

		} catch (Exception e) {
			return defaultDcm;
		}
	}

	/**
	 * オブジェクトが null の場合に null を返します。
	 *
	 * @param object オブジェクト
	 * @return
	 */
	public static String defaultString(Object object) {
		return defaultString(object, null);
	}

	/**
	 * オブジェクトが null の場合にデフォルト値を返します。
	 *
	 * @param object オブジェクト
	 * @param defaultStr デフォルト値
	 * @return
	 */
	public static String defaultString(Object object, String defaultStr) {
		if (object != null) {
			return object.toString();
		}
		return defaultStr;
	}

	/**
	 * 正規表現メタ文字のエスケープ<br>
	 * "|"を"\|"などに変換します。<br>
	 *
	 * @param str 文字列
	 * @return 変換後の文字列
	 */
	public static String escapeRegexpMeta(String str) {
		String result = str;
		try {
			String regex = "\\\\";
			String replace = "\\\\\\\\";
			result = result.replaceAll(regex, replace);

			String en2 = "\\";
			result = result.replace("*", en2 + "*");
			result = result.replace("+", en2 + "+");
			result = result.replace(".", en2 + ".");
			result = result.replace("?", en2 + "?");
			result = result.replace("{", en2 + "{");
			result = result.replace("}", en2 + "}");
			result = result.replace("(", en2 + "(");
			result = result.replace(")", en2 + ")");
			result = result.replace("[", en2 + "[");
			result = result.replace("]", en2 + "]");
			result = result.replace("^", en2 + "^");
			result = result.replace("$", en2 + "$");
			result = result.replace("-", en2 + "-");
			result = result.replace("|", en2 + "|");
		} catch (Exception e) {
		}
		return result;
	}

	/**
	 * 1文字目を小文字に変換する。
	 *
	 * @param str String
	 * @return 変換した文字列
	 */
	public static String toLowerCaseInitial(String str) {
		if (StringUtils.isEmpty(str)) {
			return str;
		} else {
			return str.substring(0, 1).toLowerCase().concat(str.substring(1));
		}
	}

	/**
	 * 1文字目を大文字に変換する。
	 *
	 * @param str String
	 * @return 変換した文字列
	 */
	public static String toUpperCaseInitial(String str) {
		if (StringUtils.isEmpty(str)) {
			return str;
		} else {
			return str.substring(0, 1).toUpperCase().concat(str.substring(1));
		}
	}

	/**
	 * 文字列を先頭が小文字のキャメルケースに変換する。
	 *
	 * LOWER_CAMEL_CASE => lowerCamelCase
	 *
	 * @param str String
	 * @return 変換した文字列
	 */
	public static String toCamelCase(String str) {
		String lowerStr = str.toLowerCase();
		StringBuilder sb = new StringBuilder();
		String[] terms = toLowerCaseInitial(lowerStr).split("_");
		sb.append(terms[0]);
		for (int i = 1, len = terms.length; i < len; i++) {
			sb.append(toUpperCaseInitial(terms[i]));
		}
		return sb.toString();
	}

	/**
	 * DB列名をURLパラメータに変換<br>
	 * スネークケース表記をローワーキャメルケース表記へ変換します。
	 *
	 * @param targetStr 変換文字列
	 * @return sb 変換した文字列
	 */
	public static String snakeToCamel(String targetStr) {
		Pattern p = Pattern.compile("_([a-z])");
		Matcher m = p.matcher(targetStr.toLowerCase());

		StringBuffer sb = new StringBuffer(targetStr.length());
		while (m.find()) {
			m.appendReplacement(sb, m.group(1).toUpperCase());
		}
		m.appendTail(sb);
		return sb.toString();
	}

	/**
	 * URLパラメータ名をDB列名に変換<br>
	 * ローワーキャメルケース表記をスネークケース表記へ変換します。
	 *
	 * @param targetStr 変換文字列
	 * @return sb 変換した文字列
	 */
	public static String camelToSnake(String targetStr) {
		StringBuilder sb = new StringBuilder();
		char[] buf = toLowerCaseInitial(targetStr).toCharArray();
		sb.append(buf[0]);
		for (int i = 1; i < buf.length; i++) {
			if ('A' <= buf[i] && buf[i] <= 'Z') {
				sb.append('_');
				sb.append(String.valueOf(buf[i]));
			} else {
				sb.append(String.valueOf(buf[i]));
			}
		}
		return sb.toString().toUpperCase();
	}

	/***
	 *
	 * 引き渡された値をJSONとして扱えるようにエスケープする
	 *
	 * @param val 入力値
	 *
	 * @return エスケープされた文字列
	 *
	 */
	public static String escapeJsonStr(String val) {
		Map<String, String> needToConvert = new HashMap<String, String>() {
			{
				put("\"", "\\\"");
				put("\\", "\\\\");
				put("/", "\\/");
				put("\n", "\\n");
				put("\r", "\\r");
				put("\t", "\\t");
			}
		};
		StringBuffer result = new StringBuffer();
		for (int i = 0; i < val.length(); i++) {
			String key = new String(new char[] { val.charAt(i) });
			if (needToConvert.containsKey(key)) {
				result.append(needToConvert.get(key));
			} else {
				result.append(val.charAt(i));
			}
		}
		return result.toString();
	}

	/**
	 * 特定の区切り文字で区切られたテキストをリストに変換して返す
	 *
	 * @param separator
	 * @param src
	 * @return
	 */
	public static List<String> csvTextToList(String separator, String src) {
		List<String> dest = new ArrayList<>();

		if (src == null || src.length() == 0) {
			return dest;
		}

		if (separator == null || separator.length() == 0) {
			dest.add(src);
			return dest;
		}

		String[] splitted = src.split(separator);

		return Arrays.asList(splitted);
	}

	public static String excludeUTF8Bom(String str) {
		if (str == null) {
			return null;
		}

		if (str == "") {
			return str;
		}

		char c = str.charAt(0);
		if (Integer.toHexString(c).equals("feff")) {
			StringBuilder sb = new StringBuilder();
			for (int i = 1; i < str.length(); i++) {
				sb.append(str.charAt(i));
			}
			return sb.toString();
		} else {
			return str;
		}
	}

	/**
	 * 変換文字列の特定文字を置換文字で変換した文字列を返す
	 *
	 * @param templateStr 変換文字列
	 * @param transferContent 特定文字と置換文字
	 * @param context ActionDto
	 * @return
	 *
	 */
	public static String getDecodString(String templateStr,
										Map<String, Object> transferContent,
										Map<String, Object> map) {
		ActionDto dto = new ActionDto();
		dto.setDataMap(map);
		return getDecodString(templateStr, transferContent, dto);
	}

	/**
	 * 変換文字列の特定文字を置換文字で変換した文字列を返す
	 *
	 * @param templateStr 変換文字列
	 * @param transferContent 特定文字と置換文字
	 * @param context ActionDto
	 * @return
	 *
	 */
	public static String getDecodString(String templateStr,
										Map<String, Object> transferContent,
										ModuleContext context) {
		return getDecodString(templateStr, transferContent, context.getDto());
	}

	/**
	 * 変換文字列の特定文字を置換文字で変換した文字列を返す
	 *
	 * @param templateStr 変換文字列
	 * @param transferContent 特定文字と置換文字
	 * @param context ActionDto
	 * @return
	 *
	 */
	@SuppressWarnings("rawtypes")
	public static String getDecodString(String templateStr,
										Map<String, Object> transferContent,
										ActionDto dto) {
		if (CheckUtil.isEmpty(templateStr)) {
			// 変換文字列がない場合は処理終了
			return templateStr;
		}
		String rtnString = templateStr;

		// 特定文字と置換文字 取得
		Map<String, Object> transferParams = new HashMap<String, Object>();
		for (Map.Entry<String, Object> param : transferContent.entrySet()) {
			Map content = (Map) param.getValue();
			transferParams.put(param.getKey(),
					ParamUtil.getParamValueByType(
							String.valueOf(content.get("type")),
							content.get("val"), dto));
		}

		// 変換後文字列作成
		for (Map.Entry<String, Object> entry : transferParams.entrySet()) {
			if (!CheckUtil.isEmpty(String.valueOf(entry.getValue()))) {
				rtnString = rtnString.replaceAll(
						Matcher.quoteReplacement(entry.getKey()),
						nullStringToBlank(String.valueOf(entry.getValue())));
			}
		}

		return rtnString;
	}

	/**
	 * 文字列がnullまたはnull文字列の場合、空文字に変換する
	 *
	 * @param str 文字列
	 * @return 変換後の文字列
	 */
	public static String nullStringToBlank(Object str) {
		if (str == null || "null".equals(String.valueOf(str))) {
			return "";
		}
		return String.valueOf(str);
	}

	/**
	 * 文字列(yyyy/MM/dd)をjava.sql.Dateに変換
	 *
	 * @param str 文字列
	 * @return 変換後の文字列(変換できなかった場合はnull)
	 */
	public static Date toDate(Object str) {
		if (str == null || "null".equals(String.valueOf(str))) {
			return null;
		}

		java.sql.Date sqlDate = null;
		try {
			sqlDate = new java.sql.Date(new SimpleDateFormat("yyyy/MM/dd")
					.parse(String.valueOf(str)).getTime());
		} catch (ParseException e) {
		}

		return sqlDate;
	}
}
