package jp.ill.photon.util;

import java.sql.SQLException;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.postgresql.jdbc4.Jdbc4Array;
import org.postgresql.util.PGobject;

import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.model.Banner;
import jp.ill.photon.model.News;
import jp.ill.photon.model.Publishing;

/**
 * DomaSQLで使用するための関数群
 *
 * @author h_tanaka
 *
 */
public class SQLUtil {

	/**
	 * Doma SQL用に文字列をエスケープして返す
	 * 
	 * @param src
	 * @return
	 */
	public static String escapeSqlString(String src) {
		if (src == null || src.length() == 0) {
			return src;
		}

		// Domaの埋め込み文字列内で使えない文字は"chr(XX)"にしてしまう
		char[] bannedChars = { '/', '\'', '\r', '\n', '\t', '"', '-', ';' };
		for (int i = 0; i < bannedChars.length; i++) {
			src = src.replace(String.valueOf(bannedChars[i]),
					"$$ || chr(" + (int) bannedChars[i] + ") || $$");
		}

		// PostgreSQL予約済み記号はエスケープする
		src = src.replace("\\", "\\\\");
		src = src.replace("%", "\\%");
		src = src.replace("_", "\\_");
		src = src.replace("*", "\\*");

		return src;
	}

	public static String infix(Object src) {
		StringBuffer sb = new StringBuffer();
		sb.append("%");
		sb.append(src);
		sb.append("%");

		return sb.toString();
	}

	public static boolean isInstanceOf(Object src, String type) {
		Class<?> clazz = null;
		try {
			clazz = Class.forName(type);
		} catch (ClassNotFoundException e) {
		}

		if (clazz == null) {
			return false;
		}

		return clazz.isInstance(src);
	}

	public static boolean isIn(Object src, String dataInputTypes) {

		if (dataInputTypes == null || dataInputTypes.length() == 0) {
			return false;
		}

		String[] dataInputTypeList = dataInputTypes.split(",");
		if (Arrays.asList(dataInputTypeList).contains(src)) {
			return true;
		}

		return false;
	}

	/** 現在日時を取得 */
	public static String getNowDatetime() {
		String nowDatetime = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS")
				.format(new Timestamp(System.currentTimeMillis()));
		return nowDatetime;
	}

	/** ログイン中の管理者ユーザー名を取得 */
	public static String getLoginAdminUserName(ActionDto dto) {
		Map<String, Object> authUser = autoCast(
				dto.getDataMap().get("auth_user"));
		Map<String, Object> account = autoCast(authUser.get("account"));
		String loginAdminUserName = autoCast(account.get("admin_user_name"));
		return loginAdminUserName;
	}

	/** Object型を自動キャスト */
	@SuppressWarnings("unchecked")
	public static <T> T autoCast(Object obj) {
		T castObj = (T) obj;
		return castObj;
	}

	public static Map<String, Object> toSubMap(String val) {
		Map<String, Object> subMap = new LinkedHashMap<String, Object>();
		subMap.put("type", "static");
		subMap.put("val", val);
		return subMap;
	}

	/** システム設定テーブル更新SQLパラメータ作成 */
	public static HashMap<String, Object> createUpdateSystemSettingParams(	String tenantId,
																			String id,
																			String note,
																			String nowDatetime,
																			String loginUserName) {
		HashMap<String, Object> paramMap = new HashMap<String, Object>();
		paramMap.put("tenantId", toSubMap(StringUtil.defaultString(tenantId)));
		paramMap.put("id", toSubMap(StringUtil.defaultString(id)));
		paramMap.put("note", toSubMap(StringUtil.defaultString(note)));
		paramMap.put("updateUserName",
				toSubMap(StringUtil.defaultString(loginUserName)));
		paramMap.put("updateDatetime",
				toSubMap(StringUtil.defaultString(nowDatetime)));
		paramMap.put("updateDiv", toSubMap("1"));

		return paramMap;
	}

	/** お知らせID新規採番SQLパラメータ作成 */
	public static HashMap<String, Object> createNewsIdNextValParams(String tenantId) {
		HashMap<String, Object> paramMap = new HashMap<String, Object>();
		paramMap.put("tenantId", toSubMap(tenantId));
		return paramMap;
	}

	/** お知らせマスタ登録/更新SQLパラメータ作成 */
	public static HashMap<String, Object> createNewsInfoParams(	String tenantId,
																News news,
																boolean isNewRegist) {
		HashMap<String, Object> paramMap = new HashMap<String, Object>();
		paramMap.put("tenantId", toSubMap(StringUtil.defaultString(tenantId)));
		paramMap.put("newsId",
				toSubMap(StringUtil.defaultString(news.getNewsId())));
		paramMap.put("title",
				toSubMap(StringUtil.defaultString(news.getTitle())));
		paramMap.put("newsType",
				toSubMap(StringUtil.defaultString(news.getNewsType())));
		paramMap.put("newsImage",
				toSubMap(StringUtil.defaultString(news.getNewsImage())));
		paramMap.put("contentFormat",
				toSubMap(StringUtil.defaultString(news.getContentFormat())));
		paramMap.put("detail",
				toSubMap(StringUtil.defaultString(news.getDetail())));
		paramMap.put("startDate",
				toSubMap(StringUtil.defaultString(news.getStartDate())));
		paramMap.put("endDate",
				toSubMap(StringUtil.defaultString(news.getEndDate())));
		paramMap.put("dispFlg",
				toSubMap(StringUtil.defaultString(news.getDispFlg())));
		paramMap.put("sortNo",
				toSubMap(StringUtil.defaultString(news.getSortNo())));
		paramMap.put("dispDiv",
				toSubMap(StringUtil.defaultString(news.getDispDiv())));
		paramMap.put("userCategory1DispDiv", toSubMap(
				StringUtil.defaultString(news.getUserCategory1DispDiv())));
		paramMap.put("userCategory2DispDiv", toSubMap(
				StringUtil.defaultString(news.getUserCategory2DispDiv())));
		paramMap.put("userCategory3DispDiv", toSubMap(
				StringUtil.defaultString(news.getUserCategory3DispDiv())));
		paramMap.put("userDispDiv",
				toSubMap(StringUtil.defaultString(news.getUserDispDiv())));
		paramMap.put("metaDescription",
				toSubMap(StringUtil.defaultString(news.getMetaDescription())));
		paramMap.put("metaKeywords",
				toSubMap(StringUtil.defaultString(news.getmetaKeywords())));
		paramMap.put("updateUserName",
				toSubMap(StringUtil.defaultString(news.getUpdateUserName())));
		paramMap.put("updateDatetime",
				toSubMap(StringUtil.defaultString(news.getUpdateDatetime())));
		paramMap.put("updateDiv",
				toSubMap(StringUtil.defaultString(news.getUpdateDiv())));

		if (isNewRegist) {
			paramMap.put("insertUserName", toSubMap(
					StringUtil.defaultString(news.getInsertUserName())));
			paramMap.put("insertDatetime", toSubMap(
					StringUtil.defaultString(news.getInsertDatetime())));
			paramMap.put("insertDiv",
					toSubMap(StringUtil.defaultString(news.getInsertDiv())));
		}
		return paramMap;
	}

	/** バナーID新規採番SQLパラメータ作成 */
	public static HashMap<String, Object> createBannerIdNextValParams(String tenantId) {
		HashMap<String, Object> paramMap = new HashMap<String, Object>();
		paramMap.put("tenantId", toSubMap(tenantId));
		return paramMap;
	}

	/** バナーマスタ登録/更新SQLパラメータ作成 */
	public static HashMap<String, Object> createBannerInfoParams(	String tenantId,
																	Banner banner,
																	boolean isNewRegist) {
		HashMap<String, Object> paramMap = new HashMap<String, Object>();
		paramMap.put("tenantId", toSubMap(StringUtil.defaultString(tenantId)));
		paramMap.put("bannerId",
				toSubMap(StringUtil.defaultString(banner.getBannerId())));
		paramMap.put("bannerName",
				toSubMap(StringUtil.defaultString(banner.getBannerName())));
		paramMap.put("bannerImage",
				toSubMap(StringUtil.defaultString(banner.getBannerImage())));
		paramMap.put("bannerAlt",
				toSubMap(StringUtil.defaultString(banner.getBannerAlt())));
		paramMap.put("bannerUrl",
				toSubMap(StringUtil.defaultString(banner.getBannerUrl())));
		paramMap.put("startDate",
				toSubMap(StringUtil.defaultString(banner.getStartDate())));
		paramMap.put("endDate",
				toSubMap(StringUtil.defaultString(banner.getEndDate())));
		paramMap.put("dispFlg",
				toSubMap(StringUtil.defaultString(banner.getDispFlg())));
		paramMap.put("sortNo",
				toSubMap(StringUtil.defaultString(banner.getSortNo())));
		paramMap.put("dispDiv",
				toSubMap(StringUtil.defaultString(banner.getDispDiv())));
		paramMap.put("userCategory1DispDiv", toSubMap(
				StringUtil.defaultString(banner.getUserCategory1DispDiv())));
		paramMap.put("userCategory2DispDiv", toSubMap(
				StringUtil.defaultString(banner.getUserCategory2DispDiv())));
		paramMap.put("userCategory3DispDiv", toSubMap(
				StringUtil.defaultString(banner.getUserCategory3DispDiv())));
		paramMap.put("userDispDiv",
				toSubMap(StringUtil.defaultString(banner.getUserDispDiv())));
		paramMap.put("updateUserName",
				toSubMap(StringUtil.defaultString(banner.getUpdateUserName())));
		paramMap.put("updateDatetime",
				toSubMap(StringUtil.defaultString(banner.getUpdateDatetime())));
		paramMap.put("updateDiv",
				toSubMap(StringUtil.defaultString(banner.getUpdateDiv())));

		if (isNewRegist) {
			paramMap.put("insertUserName", toSubMap(
					StringUtil.defaultString(banner.getInsertUserName())));
			paramMap.put("insertDatetime", toSubMap(
					StringUtil.defaultString(banner.getInsertDatetime())));
			paramMap.put("insertDiv",
					toSubMap(StringUtil.defaultString(banner.getInsertDiv())));
		}
		return paramMap;
	}

	/** 公開先テーブル登録SQLパラメータ作成 */
	public static HashMap<String, Object> createInsertPubParams(String tenantId,
																Publishing bannerPub,
																String procDiv) {

		HashMap<String, Object> paramMap = new HashMap<String, Object>();

		if (UtilTools.isEmpty(procDiv)
				|| !("banner".equals(procDiv) || "news".equals(procDiv))) {
			return null;
		} else if ("banner".equals(procDiv)) {
			paramMap.put("bannerId",
					toSubMap(StringUtil.defaultString(bannerPub.getId())));
		} else if ("news".equals(procDiv)) {
			paramMap.put("newsId",
					toSubMap(StringUtil.defaultString(bannerPub.getId())));
		}

		paramMap.put("procDiv", SQLUtil.toSubMap(procDiv));
		paramMap.put("tenantId", toSubMap(StringUtil.defaultString(tenantId)));

		paramMap.put("updateUserName", toSubMap(
				StringUtil.defaultString(bannerPub.getUpdateUserName())));
		paramMap.put("insertUserName", toSubMap(
				StringUtil.defaultString(bannerPub.getInsertUserName())));
		paramMap.put("userCategoryNum", toSubMap(
				StringUtil.defaultString(bannerPub.getUserCategoryNum())));
		paramMap.put("userCategoryNumRen", toSubMap(
				StringUtil.defaultString(bannerPub.getUserCategoryNumRen())));
		paramMap.put("userCategoryCd", toSubMap(
				StringUtil.defaultString(bannerPub.getUserCategoryCd())));
		paramMap.put("userCd",
				toSubMap(StringUtil.defaultString(bannerPub.getUserCd())));

		return paramMap;
	}

	/**
	 *
	 * 値のコンバート
	 *
	 * @param src
	 */
	public static void convertDbValues(List<Map<String, Object>> src) {

		// コンバート対象がないか、レコード内を走査
		if (!CollectionUtils.isEmpty(src)) {
			PGobject pgVal = null;
			Jdbc4Array arVal = null;
			for (Map<String, Object> mp : src) {
				for (Map.Entry<String, Object> e : mp.entrySet()) {
					if (e.getValue() != null) {

						if (PGobject.class.equals(e.getValue().getClass())
								&& ("jsonb"
										.equals(((PGobject) e.getValue())
												.getType().toLowerCase())
										|| "json".equals(
												((PGobject) e.getValue())
														.getType().toLowerCase()))) {
							pgVal = (PGobject) e.getValue();
							if (JsonUtil
									.isAbleToConverToMap(pgVal.getValue())) {
								mp.put(e.getKey(),
										JsonUtil.jsonToMap(pgVal.getValue()));
							} else if (JsonUtil
									.isAbleToConverToList(pgVal.getValue())) {
								mp.put(e.getKey(),
										JsonUtil.jsonToList(pgVal.getValue()));
							}
						}

						if (Jdbc4Array.class.equals(e.getValue().getClass())) {
							arVal = (Jdbc4Array) e.getValue();
							try {
								mp.put(e.getKey(), ArrayUtil
										.asList((Object[]) arVal.getArray()));
							} catch (SQLException e1) {
								mp.put(e.getKey(), new ArrayList<Object>());
							}
						}
					}
				}
			}
		}
	}
}
