package jp.ill.photon.util;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.module.ModuleContext;

public class ParamUtil {

	public static final class TransferTypes {
		public static final String PARAM = "param";
		public static final String PARAM_JOINED = "param_joined";
		public static final String PARAM_FILE = "param_file";
		public static final String DTO = "dto";
		public static final String PARAM_MULTI = "param_multi";
		public static final String PARAM_LIST = "param_list";
		public static final String PARAM_FILE_MULTI = "param_file_multi";
		public static final String STATIC_JSON = "static_json";
		public static final String PARAM_ARRAY = "param_array";
		public static final String PARAM_ARRAY_MULTI = "param_array_multi";
		public static final String PARAM_ARRAY_FILE = "param_array_file";
	}

	/**
	 * アクションのパラメータをコンバートする
	 * パラメータタイプがparam:の場合はdtoのアクションパラメータからtransferValのキーを持つ値を取得する。
	 * パラメータタイプがdto:の場合はdtoのデータマップからtransferValのキーを持つ値を取得する
	 * パラメータタイプがparam_multi:の場合はdtoのデータマップからtransferValのキーを持つ値をリストで取得する
	 * パラメータタイプがparam_file:の場合はdtoのアクションパラメータからtransferValのキーを持つファイルパラメータを取得する。
	 * パラメータタイプがparam_file_multi:
	 * の場合はdtoのアクションパラメータからtransferValのキーを持つファイルパラメータのリストを取得する。
	 * それ以外の場合はtransferValを固定値として取得する
	 *
	 * @param transferType
	 * @param transferVal
	 * @param dto
	 * @return
	 */
	public static Object getParamValueByType(	String transferType,
												Object transferVal,
												ActionDto dto) {
		if (transferType.equals(TransferTypes.PARAM)) {
			if (StringUtil.defaultString(transferVal, "").isEmpty()) {
				return dto.getParams().getSingleValueParams();
			} else {
				return dto.getParams()
						.get(StringUtil.defaultString(transferVal, ""));
			}
		} else if (transferType.equals(TransferTypes.PARAM_JOINED)) {
			if (StringUtil.defaultString(transferVal, "").isEmpty()) {
				return dto.getParams().getJoinedValueParams();
			} else {
				return dto.getParams()
						.getJoined(StringUtil.defaultString(transferVal, ""));
			}
		} else if (transferType.equals(TransferTypes.DTO)) {
			return dto.get(StringUtil.defaultString(transferVal, ""));
		} else if (transferType.equals(TransferTypes.PARAM_MULTI)) {
			return dto.getParams()
					.getValues(StringUtil.defaultString(transferVal, ""));
		} else if (transferType.equals(TransferTypes.PARAM_ARRAY)) {
			return dto.getParams()
					.getArrayValues(StringUtil.defaultString(transferVal, ""));
		} else if (transferType.equals(TransferTypes.PARAM_ARRAY_MULTI)) {
			return dto.getParams().getArrayMultiValues(
					StringUtil.defaultString(transferVal, ""));
		} else if (transferType.equals(TransferTypes.PARAM_ARRAY_FILE)) {
			return dto.getParams().getArrayFileValue(
					StringUtil.defaultString(transferVal, ""));
		} else if (transferType.equals(TransferTypes.PARAM_LIST)) {
			return dto.getParams()
					.getListValues(StringUtil.defaultString(transferVal, ""));
		} else if (transferType.equals(TransferTypes.STATIC_JSON)) {
			return transferVal;
		} else if (transferType.equals(TransferTypes.PARAM_FILE)) {
			return dto.getParams()
					.getFileValue(StringUtil.defaultString(transferVal, ""));
		} else if (transferType.equals(TransferTypes.PARAM_FILE_MULTI)) {
			return dto.getParams().getFileParamValues(
					StringUtil.defaultString(transferVal, ""));
		} else {
			return transferVal;
		}
	}

	@SuppressWarnings("unchecked")
	public static Map<String, Object> getParamValueByType(	Map<String, Object> src,
															ModuleContext context) {
		Map<String, Object> params = new LinkedHashMap<String, Object>();
		if (src != null) {
			for (Map.Entry<String, Object> param : src.entrySet()) {
				if (param.getValue() instanceof Map) {
					Map<String, Object> paramInfo = (Map<String, Object>) param
							.getValue();
					if (paramInfo != null
							&& paramInfo.containsKey("transfer_type")
							&& !((String) paramInfo.get("transfer_type"))
									.equals(TransferTypes.STATIC_JSON)) {
						Object paramValue = ParamUtil.getParamValueByType(
								(String) paramInfo.get("transfer_type"),
								paramInfo.get("transfer_val"),
								context.getDto());
						params.put(param.getKey(), paramValue);
					} else {
						params.put(param.getKey(),
								getParamValueByType(paramInfo, context));
					}
				} else {
					if (param != null
							&& ((ActionDto) param).containsKey("transfer_type")
							&& !((String) ((ActionDto) param)
									.get("transfer_type")).equals(
											TransferTypes.STATIC_JSON)) {
						Object paramValue = ParamUtil.getParamValueByType(
								(String) ((ActionDto) param)
										.get("transfer_type"),
								((ActionDto) param).get("transfer_val"),
								context.getDto());
						params.put(param.getKey(), paramValue);
					}
				}
			}
		}
		return params;
	}

	/**
	 * "type"に指定されているタイプのオブジェクトを取得
	 *
	 * @param src
	 * @param dto
	 * @return String
	 *
	 */
	public static String getParamStrValueByType(Object src, ActionDto dto) {
		return (String) getParamObjectValueByType(src, dto);
	}

	/**
	 * "type"に指定されているタイプのオブジェクトを取得
	 *
	 * @param src
	 * @param dto
	 * @return Object
	 *
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	public static Object getParamObjectValueByType(Object src, ActionDto dto) {
		if (src == null) {
			return null;
		}
		Map<String, Object> mp = (Map) src;
		return getParamValueByType((String) mp.get("type"), mp.get("val"), dto);
	}

	/**
	 * キャメルケースからスネークケースに変換
	 *
	 * @param targetStr
	 * @return
	 */
	public static String camelToSnake(String targetStr) {
		String convertedStr = targetStr
				.replaceAll("([A-Z]+)([A-Z][a-z])", "$1_$2")
				.replaceAll("([a-z])([A-Z])", "$1_$2");
		return convertedStr.toLowerCase();
	}

	/**
	 * スネークケースからキャメルケースに変換
	 *
	 * @param targetStr
	 * @return
	 */
	public static String snakeToCamel(String targetStr) {
		Pattern p = Pattern.compile("_([a-z])");
		Matcher m = p.matcher(targetStr.toLowerCase());

		StringBuffer sb = new StringBuffer(targetStr.length());
		while (m.find()) {
			m.appendReplacement(sb, m.group(1).toUpperCase());
		}
		m.appendTail(sb);
		return sb.toString();
	}

	/**
	 * フィールド名からgetterメソッド名を生成
	 *
	 * @param fieldName
	 * @return
	 */
	public static String getGetterName(String fieldName) {
		return String.format("get%s%s",
				Character.toUpperCase(fieldName.charAt(0)),
				fieldName.substring(1));
	}

	/**
	 * フィールド名からsetterメソッド名を生成
	 *
	 * @param fieldName
	 * @return
	 */
	public static String getSetterName(String fieldName) {
		return String.format("set%s%s",
				Character.toUpperCase(fieldName.charAt(0)),
				fieldName.substring(1));
	}
}
