package jp.ill.photon.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import jp.ill.photon.util.StringUtil;

public class FormUtil {

	/**
	 * FormFieldを変換設定リストを元に変換する
	 * 
	 * @param formFieldData
	 * @param formFieldSettings
	 * @param convertSettings
	 * @return
	 */
	public static Map<String, Map<String, Object>> convertForm(	Map<String, Object> formFieldData,
																Map<String, Object> formFieldSettings,
																List<Map<String, Object>> convertSettings) {

		return convert(formFieldData, formFieldSettings, convertSettings,
				"form_fields", "output_fields");
	}

	/**
	 * output_field -> form_fieldでconvertFormと逆方向のコンバートを行う
	 * 
	 * @param formFieldData
	 * @param formFieldSettings
	 * @param convertSettings
	 * @return
	 */
	public static Map<String, Map<String, Object>> convertFormBackwards(Map<String, Object> formFieldData,
																		Map<String, Object> formFieldSettings,
																		List<Map<String, Object>> convertSettings) {

		return convert(formFieldData, formFieldSettings, convertSettings,
				"output_fields", "form_fields");
	}

	/**
	 * 共通出力生成処理
	 * 
	 * @param formFieldData
	 * @param formFieldSettings
	 * @param convertSettings
	 * @param inputSettingField
	 * @param outputSettingField
	 * @return
	 */
	protected static Map<String, Map<String, Object>> convert(	Map<String, Object> formFieldData,
																Map<String, Object> formFieldSettings,
																List<Map<String, Object>> convertSettings,
																String inputSettingField,
																String outputSettingField) {

		Map<String, Map<String, Object>> convertedData = new LinkedHashMap<>();

		Map<String, String> convertedResult = null;
		List<String> fieldCdList = null;
		List<String> outputFieldCdList = null;
		List<String> convertedFieldCdList = new ArrayList<>();
		String joinedValue = null;
		for (Map<String, Object> setting : convertSettings) {
			outputFieldCdList = getInputFieldCdList(setting,
					outputSettingField);
			if (outputFieldCdList == null || outputFieldCdList.size() == 0) {
				continue;
			}
			convertedFieldCdList.addAll(outputFieldCdList);

			fieldCdList = getInputFieldCdList(setting, inputSettingField);

			joinedValue = getJoinedValueFromFormFieldCdList(fieldCdList,
					formFieldData);

			convertedResult = convertValue(joinedValue, setting);
			joinedValue = convertedResult.get("value");

			convertedData.putAll(createOutputData(joinedValue, setting,
					formFieldSettings, convertedResult.get("cutback"),
					outputSettingField));
		}

		List<String> formFieldKeys = formFieldSettings.keySet().stream()
				.map(s -> s.replaceAll(".*\\.", ""))
				.collect(Collectors.toList());

		Map<String, Object> outputFieldMap = null;
		for (Map.Entry<String, Object> entry : formFieldData.entrySet()) {
			if (!convertedFieldCdList.contains(entry.getKey())
					&& formFieldKeys.contains(entry.getKey())) {
				outputFieldMap = new HashMap<>();
				outputFieldMap.put("group", "");
				outputFieldMap.put("value", entry.getValue());
				outputFieldMap.put("errors", new ArrayList<String>());
				convertedData.put(entry.getKey(), outputFieldMap);
			}
		}

		return convertedData;
	}

	/**
	 * フォームフィールドコードリスト取得
	 * 
	 * @param convertSetting
	 * @return
	 */
	protected static List<String> getInputFieldCdList(	Map<String, Object> convertSetting,
														String inputSettingField) {
		List<String> fieldCdList = new ArrayList<>();

		for (Map.Entry<String, String> formField : ((Map<String, String>) convertSetting
				.get(inputSettingField)).entrySet()) {
			fieldCdList.add(formField.getValue());
		}

		return fieldCdList;
	}

	/**
	 * 入力データ作成
	 * 
	 * @param fieldCdList
	 * @param formFieldData
	 * @return
	 */
	protected static String getJoinedValueFromFormFieldCdList(	List<String> fieldCdList,
																Map<String, Object> formFieldData) {
		StringBuffer value = new StringBuffer();
		for (String fieldCd : fieldCdList) {
			if (formFieldData.containsKey(fieldCd)) {
				value.append(String.valueOf(formFieldData.get(fieldCd)));
			}
		}

		return value.toString();
	}

	/**
	 * 値変換処理
	 * 
	 * @param value
	 * @param convertSetting
	 * @return
	 */
	protected static Map<String, String> convertValue(	String value,
														Map<String, Object> convertSetting) {

		String cutback = "0";

		Map<String, Object> convertMap = (Map<String, Object>) convertSetting
				.getOrDefault("convert_settings", null);
		if (convertMap != null) {
			if (convertMap.containsKey("default_value")) {
				if (StringUtils.isEmpty(value)) {
					value = StringUtil.defaultString(convertMap.get("default_value"), "");
				}
			}
			if (convertMap.containsKey("fill_text")
					&& convertMap.containsKey("fill_digits")) {
                String fillDigits = StringUtil.defaultString(
                        convertMap.getOrDefault("fill_digits", "0"), "0");
                try {
                    value = StringUtil.leftPadding(value,
                            StringUtil.defaultString(convertMap.get("fill_text"), ""),
                            Integer.parseInt(fillDigits));
                } catch (NumberFormatException e) {
                }
			}

			if (convertMap.getOrDefault("cutback_by_length", "0").equals("1")) {
				cutback = "1";
			}

			if (convertMap.containsKey("convert_str_map")) {
				Map<String, String> convertStrMap = (Map<String, String>) convertMap
						.get("convert_str_map");
				if (convertStrMap.containsKey(value)) {
					value = StringUtil.defaultString(convertStrMap.get(value), "");
				}
			}
		}

		Map<String, String> result = new HashMap<>();
		result.put("cutback", cutback);
		result.put("value", value);

		return result;
	}

	/**
	 * 出力データを作成
	 * 
	 * @param value
	 * @param convertSetting
	 * @param formFieldSettings
	 * @param cutback
	 * @return
	 */
	protected static Map<String, Map<String, Object>> createOutputData(	String value,
																		Map<String, Object> convertSetting,
																		Map<String, Object> formFieldSettings,
																		String cutback,
																		String outputSettingField) {
		Map<String, Map<String, Object>> convertedData = new HashMap<>();
		String outputValue = null;
		Map<String, Object> outputFieldSetting = null;
		int maxByteLength = 0;
		String[] splittedValue = null;
		Map<String, Object> outputFieldMap = null;
		List<String> errors = null;
		String dataGroup = StringUtil.defaultString(
				convertSetting.getOrDefault("data_group", ""), "");
		for (Map.Entry<String, String> outputField : ((Map<String, String>) convertSetting
				.get(outputSettingField)).entrySet()) {

			errors = new ArrayList<>();
			outputValue = value;
			if (formFieldSettings
					.containsKey(dataGroup + "." + outputField.getValue())) {
				outputFieldSetting = (Map<String, Object>) formFieldSettings
						.get(dataGroup + "." + outputField.getValue());
				if (outputFieldSetting.containsKey("max_byte_length")) {
					maxByteLength = Integer.parseInt(StringUtil.defaultString(
							outputFieldSetting.get("max_byte_length"), "0"));
					if (maxByteLength == 0) {
						maxByteLength = Integer.MAX_VALUE;
					}
					splittedValue = StringUtil.lengthSplitBytes(value,
							maxByteLength, "windows-31j");
					outputValue = splittedValue[0];
					value = String.join("", ArrayUtil.rest(splittedValue));
					if (cutback.equals("0") && value.length() > 0) {
						errors.add("max_byte_length_error");
					}
				}
			}

			outputFieldMap = new HashMap<>();
			outputFieldMap.put("group", dataGroup);
			outputFieldMap.put("value", outputValue);
			outputFieldMap.put("errors", errors);
            if (dataGroup != "") {
                convertedData.put(dataGroup + "." + outputField.getValue(), outputFieldMap);
            } else {
                convertedData.put(outputField.getValue(), outputFieldMap);
            }
		}

		return convertedData;
	}
}
