package jp.ill.photon.util;

import java.io.File;
import java.io.FileOutputStream;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.FileUtils;

/**
 * ファイル処理クラス.
 *
 * @author
 */
public class FileUtil {

	/**
	 * ファイルの移動
	 *
	 * @param file 移動対象のファイル
	 * @param path 移動先のファイル名を含めたフルパス
	 */
	public void moveFile(File file, String path) throws Exception {

		// 移動先のディレクトリを取得
		File newfile = new File(path);
		// ファイルをコピーする
		FileUtils.copyFile(file, newfile, true);

		// ファイルを削除する
		FileUtils.deleteQuietly(file);
	}

	/**
	 * ファイルの移動
	 *
	 * @param fromPath 移動対象のファイルパス
	 * @param toPath 移動先のファイル名を含めたフルパス
	 */
	public void moveFile(String fromPath, String toPath) throws Exception {

		// 移動元のディレクトリを取得
		File fromFilePath = new File(fromPath);
		// 移動先のディレクトリを取得
		File toFilePath = new File(toPath);
		// ファイルをコピーする
		FileUtils.copyFile(fromFilePath, toFilePath, true);

		// ファイルを削除する
		FileUtils.deleteQuietly(fromFilePath);
	}

	/**
	 * ファイルの移動
	 *
	 * @param targetFile 移動対象のファイル
	 * @param movePath 移動先のパス
	 * @param newFileName 移動後のファイル名
	 * @return newFile 移動後のファイル
	 */
	public File moveFile(File targetFile,
								String movePath,
								String newFileName) throws Exception {

		// 移動先のディレクトリがなければ作成
		File moveDir = new File(movePath);
		if (!moveDir.exists()) {
			boolean mkDirResult = moveDir.mkdirs();
			if (!mkDirResult) {
				return null;
			}
		}

		// 移動先のパスを生成
		StringBuffer newPath = new StringBuffer();
		newPath.append(movePath);
		newPath.append(File.separator);
		newPath.append(newFileName);

		// 移動先ファイル
		File newFile = new File(newPath.toString());

		// ファイル移動
		// boolean moveResult = targetFile.renameTo(newFile);
		boolean moveResult = true;
		try {
			// ファイルをコピーする
			FileUtils.copyFile(targetFile, newFile, true);
			// ファイルを削除する
			FileUtils.deleteQuietly(targetFile);
		} catch (Exception e) {
			moveResult = false;
		}

		if (!moveResult) {
			return null;
		}

		return newFile;
	}

	/**
	 * ファイルの削除
	 *
	 * @param path 移動先のパス
	 * @param name ファイル名
	 */
	public void delFile(String path, String name) throws Exception {

		// 移動先のディレクトリを取得
		File file = new File(path + File.separator + name);

		// ファイルを削除する
		FileUtils.deleteQuietly(file);
	}

	/**
	 * ファイル名・ディレクトリ名の変更
	 *
	 * @param path 変更前パス
	 * @param path 変更後パス
	 * @return 結果
	 */
	public boolean renamePath(String fromPath, String toPath) throws Exception {

		// 移動元パス
		File fromPathFile = new File(fromPath);

		// ファイル名・ディレクトリ名を変更
		return fromPathFile.renameTo(new File(toPath));
	}

	/**
	 * ディレクトリの削除
	 *
	 * @param path 削除ディレクトリパス
	 * @param name ファイル名
	 */
	public void delDirectory(String path) throws Exception {

		// 移動先のディレクトリを取得
		File dir = new File(path);

		// ディレクトリを削除する
		FileUtils.deleteDirectory(dir);
	}

	/**
	 * ファイル名から拡張子を返します。
	 *
	 * @param fileName ファイル名
	 * @return ファイルの拡張子
	 */
	public String getSuffix(String fileName) {
		if (fileName == null) {
			return null;
		}
		int point = fileName.lastIndexOf(".");
		if (point != -1) {
			return fileName.substring(point + 1);
		}
		return fileName;
	}

	/**
	 * ファイルの出力(Base64デコード)
	 *
	 * @param targetFile Base64エンコード
	 * @param movePath 作成先のパス
	 * @param newFileName 作成ファイル名
	 * @return newFile 作成ファイル
	 */
	public static File outputFileBase64(String targetFile,
										String movePath,
										String newFileName) {
		// 移動先のディレクトリがなければ作成
		File moveDir = new File(movePath);
		if (!moveDir.exists()) {
			boolean mkDirResult = moveDir.mkdirs();
			if (!mkDirResult) {
				return null;
			}
		}

		// 移動先のパスを生成
		StringBuffer newPath = new StringBuffer();
		newPath.append(movePath);
		if (!movePath.endsWith(File.separator)) {
			newPath.append(File.separator);
		}
		newPath.append(newFileName);

		// 移動先ファイル
		File newFile = new File(newPath.toString());

		// ファイル移動
		boolean result = true;
		try {
			// Base64デコード
			byte[] inByte = Base64.decodeBase64(targetFile);

			// ファイル書き出し
			FileOutputStream fo = new FileOutputStream(newFile);
			fo.write(inByte);
			fo.close();

		} catch (Exception e) {
			result = false;
		}

		if (!result) {
			return null;
		}
		return newFile;
	}

	/**
	 * Base64エンコードのみ取得
	 *
	 * @param fileStr ファイルBase64
	 * @return Base64エンコード
	 */
	public static String getFileBase64(String fileStr) {
		if (fileStr == null) {
			return null;
		}
		String regex = "data:image/(gif|jpeg|png);base64,";
		return fileStr.replaceFirst(regex, "");
	}

	/***
	 *
	 * attachmentテーブルに登録するファイル名を取得
	 *
	 * @param guid
	 * @param editTableCd
	 * @param colCd
	 * @param suffix
	 * @return
	 */
	public static String getAttachmentFileName(String guid, String editTableCd, String colCd, String suffix) {
		return String.format("%s_%s_%s", guid, editTableCd, colCd) + "." + suffix;
	}

}
