package jp.ill.photon.util;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;

/**
 * 日付ユーティリティクラス
 *
 * @author s_imai
 */
public class DateUtil {

	/**
	 * 現在の日付を返す
	 *
	 * @return Calendar 現在日付
	 */
	public static Calendar getCalendar() {
		return Calendar.getInstance();
	}

	/**
	 * 現在の日付を返す(YYYY-MM-DD HH24:MI:SS.000000000)
	 *
	 * @return String YYYY-MM-DD HH24:MI:SS.000000000
	 */
	public static String getTime() {
		return getTime(Calendar.getInstance());
	}

	/**
	 * 現在の日付を返す(YYYY-MM-DD HH24:MI:SS)
	 *
	 * @return String YYYY-MM-DD HH24:MI:SS
	 */
	public Object getSendTime() {
		return getSendTime(Calendar.getInstance(), "-", 3);
	}

	/**
	 * 引数のカレンダーに引数の日数を足した日付を返却する
	 *
	 * @param cal 加算対象の日付
	 * @param plus 加算日数
	 * @return Calendar 日を足した日付
	 */
	public static Calendar plusDay(Calendar cal, int plus) {
		// 日付の加算
		cal.add(Calendar.DATE, +plus);

		return cal;

	}

	/**
	 * 引数のカレンダーに引数の日数を引いた日付を返却する
	 *
	 * @param cal 加算対象の日付
	 * @param minus 減算日数
	 * @return Calendar 日を引いた日付
	 */
	public static Calendar minusDay(Calendar cal, int minus) {
		// 日付の減算
		cal.add(Calendar.DATE, -minus);

		return cal;

	}

	/**
	 * 引数のカレンダーに引数の月を足した日付を返却する
	 *
	 * @param cal 加算対象の日付
	 * @param plus 加算月数
	 * @return Calendar 月を足した日付
	 */
	public static Calendar plusMonth(Calendar cal, int plus) {
		// 月の加算
		cal.add(Calendar.MONTH, plus);

		return cal;

	}

	/**
	 * 引数のカレンダーに引数の月を引いた日付を返却する
	 *
	 * @param cal 加算対象の日付
	 * @param minus 減算月数
	 * @return Calendar 月を引いた日付
	 */
	public static Calendar minusMonth(Calendar cal, int minus) {
		// 月の減算
		cal.add(Calendar.MONTH, -minus);

		return cal;

	}

	/**
	 * 引数のカレンダーに引数の年を足した日付を返却する
	 *
	 * @param cal 加算対象の日付
	 * @param plus 加算年数
	 * @return Calendar 年を足した日付
	 */
	public static Calendar plusYear(Calendar cal, int plus) {
		// 日付の加算
		cal.add(Calendar.YEAR, plus);

		return cal;

	}

	/**
	 * 引数のカレンダーに引数の年を引いた日付を返却する
	 *
	 * @param cal 加算対象の日付
	 * @param minus 減算年数
	 * @return Calendar 年を引いた日付
	 */
	public static Calendar minusYear(Calendar cal, int minus) {
		// 日付の減算
		cal.add(Calendar.YEAR, -minus);

		return cal;

	}

	/**
	 * 引数のカレンダーを指定のフォーマットで返す（YYYY-MM-DD HH24:MI:SS.000000000）
	 *
	 * @return String YYYY-MM-DD HH24:MI:SS.000000000
	 */
	public static String getTime(Calendar cal) {

		int year = cal.get(Calendar.YEAR); // (2)現在の年を取得
		int month = cal.get(Calendar.MONTH) + 1; // (3)現在の月を取得
		int day = cal.get(Calendar.DATE); // (4)現在の日を取得
		int hour = cal.get(Calendar.HOUR_OF_DAY); // (5)現在の時を取得
		int minute = cal.get(Calendar.MINUTE); // (6)現在の分を取得
		int second = cal.get(Calendar.SECOND); // (7)現在の秒を取得

		return year + "-" + month + "-" + day + " " + hour + ":" + minute + ":"
				+ second + ".000000000";
	}

	/**
	 * 引数のカレンダーを指定のフォーマットで返す（YYYY-MM-DD）
	 *
	 * @return String YYYY-MM-DD
	 */
	public static String formatDate(String str) {
		return getSendTime(str, "/", 1);
	}

	/**
	 * 引数のカレンダーを指定のフォーマットで返す（YYYY-MM-DD HH24:MI）
	 *
	 * @return String YYYY-MM-DD HH24:MI
	 */
	public static String formatDateTimeToMinute(String str) {
		return getSendTime(str, "/", 2);
	}

	/**
	 * 引数のカレンダーを指定のフォーマットで返す（YYYY-MM-DD HH24:MI:SS）
	 *
	 * @return String YYYY-MM-DD HH24:MI:SS
	 */
	public static String formatDateTimeToSecond(String str) {
		return getSendTime(str, "/", 3);
	}

	/**
	 * 引数のカレンダーを指定のフォーマットで返す（YYYY-MM-DD HH24:MI:SS）
	 * 
	 * @return
	 *
	 * @return String YYYY-MM-DD HH24:MI:SS
	 */
	private static String getSendTime(	String str,
										String dateSeparator,
										int type) {
		if (str == null) {
			return "";
		}
		if (str == "") {
			return "";
		}

		Calendar cal = DateUtil.toCalendar(str);
		return getSendTime(cal, dateSeparator, type);
	}

	/**
	 * 引数のカレンダーを指定のフォーマットで返す（YYYY-MM-DD HH24:MI:SS）
	 *
	 * @return String YYYY-MM-DD HH24:MI:SS
	 */
	private static String getSendTime(	Calendar cal,
										String dateSeparator,
										int type) {
		if (cal == null) {
			return "";
		}
		int year = cal.get(Calendar.YEAR); // (2)現在の年を取得
		int month = cal.get(Calendar.MONTH) + 1; // (3)現在の月を取得
		int day = cal.get(Calendar.DATE); // (4)現在の日を取得
		int hour = cal.get(Calendar.HOUR_OF_DAY); // (5)現在の時を取得
		int minute = cal.get(Calendar.MINUTE); // (6)現在の分を取得
		int second = cal.get(Calendar.SECOND); // (7)現在の秒を取得

		StringBuffer returnDate = new StringBuffer();

		if (type == 1) {
			returnDate.append(year);
			returnDate.append(dateSeparator);
			returnDate.append((String.format("%1$02d", month)));
			returnDate.append(dateSeparator);
			returnDate.append((String.format("%1$02d", day)));
		}
		if (type == 2) {
			returnDate.append(year);
			returnDate.append(dateSeparator);
			returnDate.append((String.format("%1$02d", month)));
			returnDate.append(dateSeparator);
			returnDate.append((String.format("%1$02d", day)));
			returnDate.append(" ");
			returnDate.append((String.format("%1$02d", hour)));
			returnDate.append(":");
			returnDate.append((String.format("%1$02d", minute)));
		}
		if (type == 3) {
			returnDate.append(year);
			returnDate.append(dateSeparator);
			returnDate.append((String.format("%1$02d", month)));
			returnDate.append(dateSeparator);
			returnDate.append((String.format("%1$02d", day)));
			returnDate.append(" ");
			returnDate.append((String.format("%1$02d", hour)));
			returnDate.append(":");
			returnDate.append((String.format("%1$02d", minute)));
			returnDate.append(":");
			returnDate.append((String.format("%1$02d", second)));
		}

		return returnDate.toString();
	}

	/**
	 * 期間日数計算
	 *
	 * @param ys （開始年）
	 * @param ms （開始月）
	 * @param ds （開始日）
	 * @param ye （終了年）
	 * @param me （終了月）
	 * @param de （終了日）
	 * @return int term（開始日～終了日間の日数）
	 */
	public static int termCalc(int ys, int ms, int ds, int ye, int me, int de) {

		int term = 0;

		Calendar objAfter = Calendar.getInstance();
		Calendar objBefore = Calendar.getInstance();

		// Calenderクラスでは1月は0、2月は1となっているため、取得月-1する。
		objAfter.set(ye, me - 1, de);
		objBefore.set(ys, ms - 1, ds);

		long timeDiff = objAfter.getTimeInMillis()
				- objBefore.getTimeInMillis();
		long dayDif = timeDiff / 1000 / 60 / 60 / 24;

		term = (int) (dayDif);

		return term;
	}

	/**
	 * 終了日計算 現在日時に日数を足した年月日を返す。
	 *
	 * @param int term（日数）
	 * @return String YYYY/MM/DD
	 */
	public static String endDateCalc(int term) {

		Calendar objPresent = Calendar.getInstance();

		int adDay = term;
		objPresent.add(Calendar.DATE, adDay);

		int yee = objPresent.get(Calendar.YEAR);
		int mee = objPresent.get(Calendar.MONTH) + 1;
		int dee = objPresent.get(Calendar.DATE);

		String yes = yee + "";
		String mes = mee + "";
		String des = dee + "";

		if (mes.length() == 1) {
			mes = "0" + mes;
		}
		if (des.length() == 1) {
			des = "0" + des;
		}

		return yes + "/" + mes + "/" + des + "";

	}

	/**
	 * 現在の日付を返す
	 * 
	 * @return
	 *
	 * @return String YYYY-MM-DD HH24:MI:SS.000000000
	 */
	public String getCurrentTime() {

		Calendar cal1 = Calendar.getInstance(); // (1)オブジェクトの生成

		int year = cal1.get(Calendar.YEAR); // (2)現在の年を取得
		int month = cal1.get(Calendar.MONTH) + 1; // (3)現在の月を取得
		int day = cal1.get(Calendar.DATE); // (4)現在の日を取得
		int hour = cal1.get(Calendar.HOUR_OF_DAY); // (5)現在の時を取得
		int minute = cal1.get(Calendar.MINUTE); // (6)現在の分を取得
		int second = cal1.get(Calendar.SECOND); // (7)現在の秒を取得

		return year + "-" + month + "-" + day + " " + hour + ":" + minute + ":"
				+ second + ".000000000";
	}

	/**
	 * 現在の日付を返す
	 *
	 * @return String String YYYY-MM-DD
	 */
	public String getCurrentDate() {

		Calendar cal1 = Calendar.getInstance(); // (1)オブジェクトの生成

		int year = cal1.get(Calendar.YEAR); // (2)現在の年を取得
		int month = cal1.get(Calendar.MONTH) + 1; // (3)現在の月を取得
		int day = cal1.get(Calendar.DATE); // (4)現在の日を取得

		return year + "-" + month + "-" + day;
	}

	/**
	 * 指定された日付・時刻文字列を、Calendarクラスに変換 変換可能な形式 yyyy/MM/dd yy/MM/dd yyyy-MM-dd
	 * yy-MM-dd yyyyMMdd 上記に以下の時間フィールドが組み合わされた状態でも有効 HH:mm HH:mm:ss HH:mm:ss.SSS
	 *
	 * @param strDate 日付・時刻文字列
	 * @return 変換後のCalendarクラス
	 */
	public static Calendar toCalendar(String strDate) {
		strDate = format(strDate);
		Calendar cal = Calendar.getInstance();

		if (strDate != null) {
			cal.setLenient(false);

			int yyyy = Integer.parseInt(strDate.substring(0, 4));
			int MM = Integer.parseInt(strDate.substring(5, 7));
			int dd = Integer.parseInt(strDate.substring(8, 10));
			int HH = cal.get(Calendar.HOUR_OF_DAY);
			int mm = cal.get(Calendar.MINUTE);
			int ss = cal.get(Calendar.SECOND);
			int SSS = cal.get(Calendar.MILLISECOND);
			cal.clear();
			cal.set(yyyy, MM - 1, dd);
			int len = strDate.length();
			switch (len) {
			case 10:
				break;
			case 16: // yyyy/MM/dd HH:mm
				HH = Integer.parseInt(strDate.substring(11, 13));
				if (HH == 24) {
					HH = 0;
					cal.add(Calendar.DAY_OF_MONTH, 1);
				}
				mm = Integer.parseInt(strDate.substring(14, 16));
				cal.set(Calendar.HOUR_OF_DAY, HH);
				cal.set(Calendar.MINUTE, mm);
				break;
			case 19: // yyyy/MM/dd HH:mm:ss
				HH = Integer.parseInt(strDate.substring(11, 13));
				if (HH == 24) {
					HH = 0;
					cal.add(Calendar.DAY_OF_MONTH, 1);
				}
				mm = Integer.parseInt(strDate.substring(14, 16));
				ss = Integer.parseInt(strDate.substring(17, 19));
				cal.set(Calendar.HOUR_OF_DAY, HH);
				cal.set(Calendar.MINUTE, mm);
				cal.set(Calendar.SECOND, ss);
				break;
			case 23: // yyyy/MM/dd HH:mm:ss.SSS
				HH = Integer.parseInt(strDate.substring(11, 13));
				if (HH == 24) {
					HH = 0;
					cal.add(Calendar.DAY_OF_MONTH, 1);
				}
				mm = Integer.parseInt(strDate.substring(14, 16));
				ss = Integer.parseInt(strDate.substring(17, 19));
				SSS = Integer.parseInt(strDate.substring(20, 23));
				cal.set(Calendar.HOUR_OF_DAY, HH);
				cal.set(Calendar.MINUTE, mm);
				cal.set(Calendar.SECOND, ss);
				cal.set(Calendar.MILLISECOND, SSS);
				break;
			default:
				cal = null;
			}
		} else {
			cal = null;
		}
		return cal;
	}

	/**
	 * 様々な日付、時刻文字列をデフォルトの日付・時刻フォーマット へ変換します。 デフォルトの日付フォーマットは以下になります。
	 * 日付だけの場合：yyyy/MM/dd 日付+時刻の場合：yyyy/MM/dd HH:mm:ss.SSS
	 *
	 * @param str 変換対象の文字列
	 * @return デフォルトの日付・時刻フォーマット
	 */
	public static String format(String str) {
		if (str == null || str.trim().length() < 8) {
			return null;
		}
		str = str.trim();
		String yyyy = null;
		String MM = null;
		String dd = null;
		String HH = null;
		String mm = null;
		String ss = null;
		String SSS = null;
		// "-" or "/" が無い場合
		if (str.indexOf("/") == -1 && str.indexOf("-") == -1) {
			if (str.length() == 8) {
				yyyy = str.substring(0, 4);
				MM = str.substring(4, 6);
				dd = str.substring(6, 8);
				return yyyy + "/" + MM + "/" + dd;
			}
			yyyy = str.substring(0, 4);
			MM = str.substring(4, 6);
			dd = str.substring(6, 8);
			HH = str.substring(9, 11);
			mm = str.substring(12, 14);
			ss = str.substring(15, 17);
			return yyyy + "/" + MM + "/" + dd + " " + HH + ":" + mm + ":" + ss;
		}

		String[] splitDate = str.split("[_/\\-:\\.\\s]");

		StringBuffer result = new StringBuffer();
		boolean errorFlg = false;
		for (int i = 0; i < splitDate.length; i++) {
			String temp = splitDate[i];
			switch (i) {
			case 0:// 年の部分
				yyyy = fillString(str, temp, "L", "20", 4);
				if (yyyy != null) {
					result.append(yyyy);
				} else {
					errorFlg = true;
				}
				break;
			case 1:// 月の部分
				MM = fillString(str, temp, "L", "0", 2);
				if (MM != null && MM.length() > 0) {
					result.append("/" + MM);
				} else {
					errorFlg = true;
				}
				break;
			case 2:// 日の部分
				dd = fillString(str, temp, "L", "0", 2);
				if (dd != null) {
					result.append("/" + dd);
				} else {
					errorFlg = true;
				}
				break;
			case 3:// 時間の部分
				HH = fillString(str, temp, "L", "0", 2);
				if (HH != null) {
					result.append(" " + HH);
				} else {
					errorFlg = true;
				}
				break;
			case 4:// 分の部分
				mm = fillString(str, temp, "L", "0", 2);
				if (mm != null) {
					result.append(":" + mm);
				} else {
					errorFlg = true;
				}
				break;
			case 5:// 秒の部分
				ss = fillString(str, temp, "L", "0", 2);
				if (ss != null) {
					result.append(":" + ss);
				} else {
					errorFlg = true;
				}
				break;
			case 6:// ミリ秒の部分
				SSS = fillString(str, temp, "R", "0", 3);
				if (SSS != null) {
					result.append("." + SSS);
				} else {
					errorFlg = true;
				}
				break;
			}
			if (errorFlg == true) {
				return null;
			}
		}
		return result.toString();
	}

	private static String fillString(	String strDate,
										String str,
										String position,
										String addStr,
										int len) {
		if (str.length() > len) {
			return null;
		}
		return fillString(str, position, len, addStr);
	}

	/**
	 * 文字列[str]に対して、補充する文字列[addStr]を [position]の位置に[len]に満たすまで挿入します。
	 * ※[str]がnullや空リテラルの場合でも[addStr]を [len]に満たすまで挿入した結果を返します。
	 *
	 * @param str 対象文字列
	 * @param position 前に挿入 ⇒ L or l 後に挿入 ⇒ R or r
	 * @param len 補充するまでの桁数
	 * @param addStr 挿入する文字列
	 * @return 変換後の文字列。
	 */
	private static String fillString(	String str,
										String position,
										int len,
										String addStr) {
		if (addStr == null || addStr.length() == 0) {
			return null;
		}
		if (str == null) {
			str = "";
		}
		StringBuffer buffer = new StringBuffer(str);
		while (len > buffer.length()) {
			if (position.equalsIgnoreCase("l")) {
				int sum = buffer.length() + addStr.length();
				if (sum > len) {
					addStr = addStr.substring(0, addStr.length() - (sum - len));
					buffer.insert(0, addStr);
				} else {
					buffer.insert(0, addStr);
				}
			} else {
				buffer.append(addStr);
			}
		}
		if (buffer.length() == len) {
			return buffer.toString();
		}
		return buffer.toString().substring(0, len);
	}

	/**
	 * 日付文字列"yyyy-MM-dd","yyyy-MM-dd HH:mm:ss.SS"を"yyyy/MM/dd"に変換します
	 *
	 * @param date 日付文字列
	 * @return 変換後の日付文字列
	 */
	public static String formatString(String date) {
		if (StringUtils.isEmpty(date)) {
			return "";
		}
		String year = date.substring(0, 4);
		String month = date.substring(5, 7);
		String day = date.substring(8, 10);

		return year + "/" + month + "/" + day;
	}

	/**
	 * 日付文字列"yyyy-MM-dd HH:mm:ss.SS"を"yyyy/MM/dd HH:mm:ss"に変換します
	 *
	 * @param date 日付文字列
	 * @return 変換後の日付文字列
	 */
	public static String formatStringTimeSecond(String date) {
		String result = "";
		try {
			String year = date.substring(0, 4);
			String month = date.substring(5, 7);
			String day = date.substring(8, 10);
			String hour = date.substring(11, 13);
			String minute = date.substring(14, 16);
			String second = date.substring(17, 19);
			result = year + "/" + month + "/" + day + " " + hour + ":" + minute
					+ ":" + second;
		} catch (Exception e) {
		}
		return result;
	}

	/**
	 * 日付のフォーマットが正しいかのチェック(YYYY/MM/DD)
	 *
	 * @param date 日付文字
	 * @return 正しい/誤り(true/false)
	 */
	public static boolean dateCheck(String date) {
		try {
			DateUtils.parseDateStrictly(date, new String[] { "yyyy/MM/dd" });

			return true;
		} catch (ParseException e) {
			return false;
		}
	}

	/**
	 * タイムスタンプ用ストリング取得.
	 *
	 * @return 現在時刻をYYYYMMDDHHMMSSS形式にしたもの
	 */
	public static String getYYYYMMDDHHMIMS() {
		Date nowDate = new Date();
		SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmssSSS");
		return sdf.format(nowDate);
	}

	// CSVメモインポート処理 START
	/**
	 * 様々な日付、時刻文字列をjava.sql.Timestampへ変換します。 デフォルトの日付フォーマットは以下になります。
	 * 日付だけの場合：yyyy/MM/dd 日付+時刻の場合：yyyy/MM/dd HH:mm:ss.SSS
	 *
	 * @param strDate 日付、時刻文字列
	 * @return java.sql.Timestampオブジェクト
	 */
	public static java.sql.Timestamp toSqlTimestamp(String strDate) {
		if (StringUtils.isEmpty(strDate)) {
			return null;
		}
		Calendar cal = toCalendar(strDate);
		return new java.sql.Timestamp(cal.getTimeInMillis());
	}

	// CSVメモインポート処理 END

	// CSVメモエクスポート処理 START
	/**
	 * Java.sql.Dateをフォーマットした日付文字列で返します.
	 *
	 * @param date java.sql.Dateオブジェクト
	 * @param format フォーマット文字列（SimpleDateFormatに依存）
	 * @return 日付文字列
	 */
	public static String formatDate(java.sql.Date date, String format) {
		if (date == null) {
			return null;
		}
		SimpleDateFormat sd = new SimpleDateFormat(format);
		return sd.format(date);
	}

	/**
	 * Java.sql.Timestampをフォーマットした日時文字列で返します.
	 *
	 * @param date java.sql.Timestampオブジェクト
	 * @param format フォーマット文字列（SimpleDateFormatに依存）
	 * @return 日時文字列
	 */
	public static String formatDate(java.sql.Timestamp timestamp,
									String format) {
		if (timestamp == null) {
			return null;
		}
		SimpleDateFormat sd = new SimpleDateFormat(format);
		return sd.format(timestamp);
	}
	// CSVメモエクスポート処理 END

	// Aladdin-EC ３次開発_02 20141204 ADD-S
	public static String formatDate(String dateStr,
									String inFormat,
									String outFormat) {

		DateFormat inDf = new SimpleDateFormat(inFormat);
		ParsePosition pos = new ParsePosition(0);
		Date date = inDf.parse(dateStr, pos);

		// 入力形式が不正の場合
		if (pos.getErrorIndex() != -1) {
			return null;
		}

		DateFormat outDf = new SimpleDateFormat(outFormat);
		return outDf.format(date);
	}
	// Aladdin-EC ３次開発_02 20141204 ADD-E

	/**
	 * Long値から日付(yyyy/MM/dd HH:mm:ss)を生成する
	 *
	 * @return String yyyy/MM/dd HH:mm:ss
	 */
	public static String formatDate(Long time) {
		String rtn = "";
		try {
			rtn = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss").format(time);
		} catch (Exception e) {
		}

		return rtn;
	}
}
