package jp.ill.photon.util;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

/**
 * 日付フォーマットチェック用ユーティリティクラス.
 */
public class DateFormatCheckUtil {

	/* 日付フォーマット区分 */
	public static class DateFormatDiv {
		/** 日付 */
		public static final String DATE_NM = "日付";
		public static final String DATE_FORMAT = "yyyy/MM/dd";
		/** 日付 */
		public static final String DATE8_NM = "日付区切なし";
		public static final String DATE8_FORMAT = "yyyyMMdd";
		/** 日時 */
		public static final String DATETIME_NM = "日時";
		public static final String DATETIME_FORMAT = "yyyy/MM/dd HH:mm";
		/** 日時秒 */
		public static final String DATETIME_HMS_NM = "年月日時分秒";
		public static final String DATETIME_HMS_FORMAT = "yyyy/MM/dd HH:mm:ss";
		/** 時刻 */
		public static final String TIME_NM = "時刻";
		public static final String TIME_FORMAT = "HH:mm";
		/** 時分秒 */
		public static final String TIME_HMS_NM = "時分秒";
		public static final String TIME_HMS_FORMAT = "HH:mm:ss";
	}

	/**
	 * 共通
	 * ------------------------------------------------------------------------------------------------------
	 */
	/**
	 * 汎用日付フォーマットチェック
	 *
	 * @param dateFormatDiv 日付フォーマット区分
	 * @param value チェック対象値
	 * @return チェック結果(true: OK, false: NG)
	 */
	public static boolean dateFormatCheck(String dateFormatDiv, String value) {

		// チェック対象値が設定されていない場合
		if (UtilTools.isEmpty(value)) {
			return false;
		}

		// 日付フォーマット区分が設定されていない場合
		if (UtilTools.isEmpty(dateFormatDiv)) {
			return false;
		}

		SimpleDateFormat formatter = null;

		/** 日付フォーマットチェック */
		if (DateFormatDiv.DATE_NM.equals(dateFormatDiv)
				|| DateFormatDiv.DATE_FORMAT.equals(dateFormatDiv)) {

			formatter = new SimpleDateFormat(DateFormatDiv.DATE_FORMAT);

			/** 区切りなし日付フォーマットチェック */
		} else if (DateFormatDiv.DATE8_NM.equals(dateFormatDiv)
				|| DateFormatDiv.DATE8_FORMAT.equals(dateFormatDiv)) {

			formatter = new SimpleDateFormat(DateFormatDiv.DATE8_FORMAT);

			/** 日時フォーマットチェック */
		} else if (DateFormatDiv.DATETIME_NM.equals(dateFormatDiv)
				|| DateFormatDiv.DATETIME_FORMAT.equals(dateFormatDiv)) {

			formatter = new SimpleDateFormat(DateFormatDiv.DATETIME_FORMAT);

			/** 日時秒フォーマットチェック */
		} else if (DateFormatDiv.DATETIME_HMS_NM.equals(dateFormatDiv)
				|| DateFormatDiv.DATETIME_HMS_FORMAT.equals(dateFormatDiv)) {

			formatter = new SimpleDateFormat(DateFormatDiv.DATETIME_HMS_FORMAT);

			/** 時刻フォーマットチェック */
		} else if (DateFormatDiv.TIME_NM.equals(dateFormatDiv)
				|| DateFormatDiv.TIME_FORMAT.equals(dateFormatDiv)) {

			formatter = new SimpleDateFormat(DateFormatDiv.TIME_FORMAT);

			/** 時分秒フォーマットチェック */
		} else if (DateFormatDiv.TIME_HMS_NM.equals(dateFormatDiv)
				|| DateFormatDiv.TIME_HMS_FORMAT.equals(dateFormatDiv)) {

			formatter = new SimpleDateFormat(DateFormatDiv.TIME_HMS_FORMAT);
		} else {
			// 無効な日付フォーマット区分
			return false;
		}

		try {
			Date result = formatter.parse(value);
			formatter.setLenient(false);

			// フォーマットを元に戻して引数と比較
			String reverse = formatter.format(result);
			if (value.equals(reverse)) {
				return true;
			} else {
				// 24時台を有効とする
				String checkValue = value.replaceAll("24:([0-9]+)", "00:$1");
				Calendar cal = Calendar.getInstance();
				cal.setTime(result);
				cal.add(Calendar.DAY_OF_MONTH, -1);
				reverse = formatter.format(cal.getTime());
				return checkValue.equals(reverse);
			}

		} catch (ParseException e) {
			return false;
		}

	}

}
