package jp.ill.photon.util;

import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;

/**
 * 汎用的に使うメソッドを集めたクラス 状況次第で取得して使用する.
 *
 * @author s_imai
 */
public class CheckUtil {

	// 文字数チェック用半角文字
	// Aladdin-EC 修正対応_3489 20140603 MOD-S
	// private static final String HALF_STRINGS =
	// " !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~｡｢｣､･ｦｧｨｩｪｫｬｭｮｯｰｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐﾑﾒﾓﾔﾕﾖﾗﾘﾙﾚﾛﾜﾝﾞﾟ";
	public static final String HALF_STRINGS = " !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~｡｢｣､･ｦｧｨｩｪｫｬｭｮｯｰｱｲｳｴｵｶｷｸｹｺｻｼｽｾｿﾀﾁﾂﾃﾄﾅﾆﾇﾈﾉﾊﾋﾌﾍﾎﾏﾐﾑﾒﾓﾔﾕﾖﾗﾘﾙﾚﾛﾜﾝﾞﾟ";

	// Aladdin-EC 修正対応_3489 20140603 MOD-E

	/**
	 * 現在のOSをWindowsかどうか判定する. Windowsシリーズの場合Trueが返る
	 *
	 * @return
	 */
	public boolean isWindows() {
		// システムプロパティからＯＳ名を取得
		String os = System.getProperty("os.name");
		// 操作し易いように小文字に変換
		os = os.toLowerCase();
		// windowsの文字列を検索
		boolean retVal = (os.indexOf("windows") > -1);
		// 終了
		return retVal;
	}

	/**
	 * 空(NULL)文字チェック.
	 *
	 * @param string
	 * @return boolean (true:空文字又はNULL false:空文字、NULL以外)
	 **/
	public static boolean isEmpty(String string) {
		if (StringUtils.isEmpty(string)) {
			return true;
		}
		return false;
	}

	/**
	 * 空(NULL)文字チェック.<br>
	 * 半角スペースや全角スペースだけの場合も空と判断します。
	 *
	 * @param string
	 * @return boolean (true:空文字又はNULL false:空文字、NULL以外)
	 **/
	public static boolean isBlank(String string) {
		if (StringUtils.isBlank(string)) {
			return true;
		}
		return false;
	}

	/**
	 * 空(NULL)リストチェック.
	 *
	 * @param array
	 * @return boolean (true:空リスト又はNULL false:空リストやNULL以外)
	 **/
	public static boolean isArrayNull(ArrayList<String> array) {
		if (array == null) {
			return true;
		}
		return false;
	}

	/**
	 * 空(NULL)データチェック(ArrayListの中に空データ有).
	 *
	 * @param array
	 * @return boolean (true:空データ又はNULL false:空データやNULL以外)
	 **/
	public static boolean isArrayRecordNull(ArrayList<String> array) {

		for (int i = 0; i < array.size(); i++) {
			if (array.get(i) == null || array.get(i).equals("")) {
				return true;
			}
		}

		return false;
	}

	/**
	 * 空(NULL)データチェック(ArrayListの中が空データのみ).
	 *
	 * @param array
	 * @return boolean (true:空データ又はNULL false:空データやNULL以外)
	 **/
	public static boolean isArrayAllNull(ArrayList<String> array) {

		ArrayList<String> nArray = new ArrayList<String>();
		for (int i = 0; i < array.size(); i++) {
			if (array.get(i) == null || array.get(i).equals("")) {
				nArray.add("e");
			}
		}
		if (nArray != null) {
			if (nArray.size() == array.size()) {
				return true;
			}
		}

		return false;
	}

	/**
	 * 日付存在チェック （年、月、日文字列指定）
	 *
	 * @param y 年
	 * @param m 月
	 * @param d 日
	 * @param hh 時
	 * @param mm 分
	 * @param ss 秒
	 * @return boolean (true:有効 false:無効)
	 */
	public static boolean invalidDateTime(	String y,
											String m,
											String d,
											String hh,
											String mm,
											String ss) {
		// 空はNG
		if (isEmpty(y)) {
			return false;
		}
		if (isEmpty(m)) {
			return false;
		}
		if (isEmpty(d)) {
			return false;
		}
		// 数値でなければNG
		int inty = 0;
		int intm = 0;
		int ints = 0;
		int inthh = 0;
		int intmm = 0;
		int intss = 0;
		try {
			inty = Integer.parseInt(y);
			intm = Integer.parseInt(m);
			ints = Integer.parseInt(d);
			inthh = Integer.parseInt(hh);
			intmm = Integer.parseInt(mm);
			intss = Integer.parseInt(ss);
		} catch (NumberFormatException e) {
			return false;
		}
		// 数値タイプの日付チェック
		return invalidDateTime(inty, intm, ints, inthh, intmm, intss);

	}

	/**
	 * 日付存在チェック（数値設定）
	 *
	 * @param y 年
	 * @param m 月
	 * @param d 日
	 * @param hh 時
	 * @param mm 分
	 * @param ss 秒
	 * @return boolean (true:有効 false:無効)
	 */
	public static boolean invalidDateTime(	int y,
											int m,
											int d,
											int hh,
											int mm,
											int ss) {

		Calendar cal = Calendar.getInstance();

		try {
			cal.setLenient(false); // 厳密チェックを行う。
			cal.set(y, m - 1, d, hh, mm, ss); // 引数の日付を設定
			cal.getTime(); // Auto変換をしていないため、暦日でなければ、Exception発生
		} catch (IllegalArgumentException e) {
			return false;
		}

		return true;

	}

	/**
	 * 日付存在チェック （年、月、日文字列指定）.<br />
	 * 厳密チェックを行わない
	 *
	 * @param y 年
	 * @param m 月
	 * @param d 日
	 * @param hh 時
	 * @param mm 分
	 * @param ss 秒
	 * @return boolean (true:有効 false:無効)
	 */
	public static boolean invalidDateTimeNo(String y,
											String m,
											String d,
											String hh,
											String mm,
											String ss) {
		// 空はNG
		if (isEmpty(y)) {
			return false;
		}
		if (isEmpty(m)) {
			return false;
		}
		if (isEmpty(d)) {
			return false;
		}
		// 数値でなければNG
		int inty = 0;
		int intm = 0;
		int ints = 0;
		int inthh = 0;
		int intmm = 0;
		int intss = 0;
		try {
			inty = Integer.parseInt(y);
			intm = Integer.parseInt(m);
			ints = Integer.parseInt(d);
			inthh = Integer.parseInt(hh);
			intmm = Integer.parseInt(mm);
			intss = Integer.parseInt(ss);
		} catch (NumberFormatException e) {
			return false;
		}
		// 数値タイプの日付チェック
		return invalidDateTimeNo(inty, intm, ints, inthh, intmm, intss);

	}

	/**
	 * 日付存在チェック（数値設定）
	 *
	 * @param y 年
	 * @param m 月
	 * @param d 日
	 * @param hh 時
	 * @param mm 分
	 * @param ss 秒
	 * @return boolean (true:有効 false:無効)
	 */
	public static boolean invalidDateTimeNo(int y,
											int m,
											int d,
											int hh,
											int mm,
											int ss) {

		Calendar cal = Calendar.getInstance();

		try {
			cal.setLenient(true); // 厳密チェックを行わない
			cal.set(y, m - 1, d, hh, mm, ss); // 引数の日付を設定
			cal.getTime(); // Auto変換をしていないため、暦日でなければ、Exception発生
		} catch (IllegalArgumentException e) {
			return false;
		}

		return true;

	}

	/**
	 * 日付存在チェック （年、月、日文字列指定）
	 *
	 * @param y 年
	 * @param m 月
	 * @param d 日
	 * @return boolean (true:有効 false:無効)
	 */
	public static boolean invalidDate(String y, String m, String d) {
		// 空はNG
		if (isEmpty(y)) {
			return false;
		}
		if (isEmpty(m)) {
			return false;
		}
		if (isEmpty(d)) {
			return false;
		}
		// 数値でなければNG
		int inty = 0;
		int intm = 0;
		int ints = 0;
		try {
			inty = Integer.parseInt(y);
			intm = Integer.parseInt(m);
			ints = Integer.parseInt(d);
		} catch (NumberFormatException e) {
			return false;
		}
		// 数値タイプの日付チェック
		return invalidDate(inty, intm, ints);

	}

	/**
	 * 日付存在チェック（数値設定）
	 *
	 * @param y 年
	 * @param m 月
	 * @param d 日
	 * @return boolean (true:有効 false:無効)
	 */
	public static boolean invalidDate(int y, int m, int d) {

		Calendar cal = Calendar.getInstance();
		cal.setLenient(false); // 厳密チェックを行う。
		cal.set(y, m - 1, d); // 引数の日付を設定

		try {
			cal.getTime(); // Auto変換をしていないため、暦日でなければ、Exception発生
		} catch (IllegalArgumentException e) {
			return false;
		}

		return true;

	}

	/**
	 * 数値の範囲チェックを行う
	 *
	 * @param num 判定値
	 * @param min 最小値
	 * @param max 最大値
	 * @return 範囲内/範囲外 (true/false)
	 */
	public static boolean isRange(int num, int min, int max) {
		// 最小値よりも小さい場合、もしくは、最大値よりも大きい場合
		if (num < min || num > max) {
			return false;
		}
		return true;
	}

	/**
	 * 数値の範囲チェックを行う
	 *
	 * @param num 判定値
	 * @param min 最小値
	 * @param max 最大値
	 * @return 範囲内/範囲外 (true/false)
	 */
	public static boolean isRange(String num, int min, int max) {
		// 数値ではない場合
		if (!isNo(num)) {
			return false;
		}

		return isRange(Integer.parseInt(num), min, max);
	}

	/**
	 * No文字チェック.
	 *
	 * @param string
	 * @return boolean (true:数値 false:数値以外)
	 **/
	public static boolean isNo(String string) {

		if (string.matches("[[0-9]]*")) {

			return true;
		} else {

			return false;
		}

	}

	/**
	 * メールアドレス書式チェック.
	 *
	 * @param chk_str 文字列
	 * @return boolean (true:メールアドレス false:メールアドレス以外)
	 **/
	public static boolean mailFormatCheck(String chk_str) {
		String ptnStr = "[\\w\\.\\-]+@(?:[\\w\\-]+\\.)+[\\w\\-]+";
		Pattern ptn = Pattern.compile(ptnStr);
		Matcher mc = ptn.matcher(chk_str);
		if (mc.matches()) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * URL書式チェック.
	 *
	 * @param chk_str 文字列
	 * @return boolean (true:URL false:URL以外)
	 **/
	public static boolean urlFormatCheck(String chk_str) {
		String ptnStr = "http(s)?://([\\w-]+\\.)+[\\w-]+(/[\\w- ./?%&=]*)?";
		Pattern ptn = Pattern.compile(ptnStr);
		Matcher mc = ptn.matcher(chk_str);
		if (mc.matches()) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 入力文字列チェック（全角）.
	 *
	 * @param chk_str 文字列
	 * @return boolean (true:全角 false:全角以外)
	 **/
	public static boolean fullCheck(String chk_str) {
		String ptnStr = "[^ -~｡-ﾟ]*";
		Pattern ptn = Pattern.compile(ptnStr);
		Matcher mc = ptn.matcher(chk_str);
		if (mc.matches()) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 入力文字列チェック（全角ひらがな）.
	 *
	 * @param chk_str 文字列
	 * @return boolean (true:全角ひらがな false:全角ひらがな以外)
	 **/
	public static boolean fullHiraganaCheck(String chk_str) {
		String ptnStr = "[ぁ-んー　]*";
		Pattern ptn = Pattern.compile(ptnStr);
		Matcher mc = ptn.matcher(chk_str);
		if (mc.matches()) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 入力文字列チェック（全角カナ）.
	 *
	 * @param chk_str 文字列
	 * @return boolean (true:全角カナ false:全角カナ以外)
	 **/
	public static boolean fullKanaCheck(String chk_str) {
		String ptnStr = "[ァ-ヴー　]*";
		Pattern ptn = Pattern.compile(ptnStr);
		Matcher mc = ptn.matcher(chk_str);
		if (mc.matches()) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 機種依存チェック.
	 *
	 * @param string
	 * @return boolean (true:機種依存文字あり false:機種依存文字なし)
	 * @throws UnsupportedEncodingException
	 * @throws UnsupportedEncodingException
	 **/
	public static boolean isModelDependence(String string) {
		for (int i = string.length() - 1; i >= 0; i--) {
			char c = string.charAt(i);
			String abc = String.valueOf(c);

			String mm = "｡ﾟ№℡\\ⅠⅡⅢⅣⅤⅥⅦⅧⅨⅩⅰⅱⅲⅳⅴⅵⅶⅷⅸⅹ∑√∟∠∥∩∪∫∮∵≒≡⊥⊿①②③④⑤⑥⑦⑧⑨⑩⑪⑫⑬⑭⑮⑯⑰⑱⑲⑳〝〟㈱㈲㈹㊤㊥㊦㊧㊨㌃㌍㌔㌘㌢㌣㌦㌧㌫㌶㌻㍉㍊㍍㍑㍗㍻㍼㍽㍾㎎㎏㎜㎝㎞㎡㏄㏍丨仡仼仼伀伃伹佖侊侒侔侚俉俍俿倞倢偀偂偆偰傔僘僴兊兤冝冾凬刕劜劦劯勀勛匀匇匤卲厓厲叝咊咜咩哿喆坙坥垬埇埈增墲夋奓奛奝奣妤妺孖寀寘寬尞岦岺峵崧嵂嵓嵭嶸嶹巐弡弴彅彧德忞恝悅悊惕惞惲愑愠愰愷憘戓抦揵摠撝擎敎昀昉昕昞昤昮昱昻晗晙晥晳暙暠暲暿曺曻朎杦枻柀栁桄桒棈棏楨榘槢樰橆橫橳橾櫢櫤毖氿汜汯沆泚洄浯涇涖涬淏淲淸淼渧渹渼湜溿澈澵濵瀅瀇瀨瀨炅炫炻焄焏煆煇煜燁燾犱犾猤獷玽珉珒珖珣珵琇琦琩琪琮瑢璉璟甁甯畯皂皛皜皞皦睆砡硎硤硺礰禔禛竑竧竫箞絈絜綠綷緖繒纊罇羡茁荢荿菇菶葈蒴蓜蕓蕙蕫薰蠇裵褜訒訷詹誧誾諟諶譓譿賰賴贒赶軏遧郞鄕鄧釗釚釞釤釥釭釮鈆鈊鈐鈹鈺鈼鉀鉎鉑鉙鉧鉷鉸銈銧鋐鋓鋕鋗鋙鋠鋧鋹鋻鋿錂錝錞錡錥鍈鍗鍰鎤鏆鏞鏸鐱鑅鑈閒隝隯霳霻靃靍靏靑靕顗顥餧馞驎髙髜魲魵鮏鮱鮻鰀鵫鵰鸙黑朗隆﨎﨎﨏塚﨑晴﨓﨔凞猪益礼神祥福靖精羽﨟蘒﨡諸﨣﨤逸都﨧﨨﨩飯飼館鶴";
			if (mm.indexOf(abc) != -1) {
				return true;
			} else {
			}
		}
		return false;
	}

	/**
	 * 文字数制限チェック（最少）.
	 *
	 * @param string
	 * @return boolean (true:制限エラー false:OK)
	 * @throws UnsupportedEncodingException
	 **/
	public static boolean isLengthCheckSJISMin(String string, int i)
			throws UnsupportedEncodingException {
		int j = string.getBytes("SJIS").length;

		if (j < i) {
			return true;
		}
		return false;
	}

	/**
	 * 文字数制限チェック.
	 *
	 * @param string
	 * @return boolean (true:制限エラー false:OK)
	 * @throws UnsupportedEncodingException
	 **/
	public static boolean isLengthCheckSJIS(String string, int i)
			throws UnsupportedEncodingException {
		int j = string.getBytes("SJIS").length;

		if (j > i) {
			return true;
		}
		return false;
	}

	/**
	 * 文字数制限チェック.
	 *
	 * @param string
	 * @return boolean (true:制限エラー false:OK)
	 * @throws UnsupportedEncodingException
	 **/
	public static boolean isLengthCheckWin31J(String string, int i)
			throws UnsupportedEncodingException {
		int j = string.getBytes("Windows-31J").length;

		if (j > i) {
			return true;
		}
		return false;
	}

	/**
	 * 文字数制限チェック<br>
	 * 半角を1全角を2としてチェックを行います。<br>
	 * nullは0文字の文字列として扱います。
	 *
	 * @param string チェックする文字列
	 * @param n チェックする文字数
	 * @return boolean (true:制限エラー false:OK)
	 **/
	public static boolean isLengthCheckWidth(String string, int n) {
		if (isEmpty(string)) {
			string = "";
		}
		int count = 0;
		String s = null;
		for (int i = 0; i < string.length(); i++) {
			s = String.valueOf(string.charAt(i));
			// 半角文字かそれ以外かチェック
			if (HALF_STRINGS.indexOf(s) != -1) {
				// 半角文字
				count += 1;
			} else {
				// 全角文字
				count += 2;
			}
		}
		if (count > n) {
			return true;
		}
		return false;
	}

	/**
	 * 文字数制限チェック<br>
	 * 半角を1全角を2としてチェックを行います。<br>
	 * nullは0文字の文字列として扱います。
	 *
	 * @param string チェックする文字列
	 * @param n チェックする文字数
	 * @return boolean (true:制限エラー false:OK)
	 **/
	public static boolean isLengthCheckWidthMin(String string, int n) {
		if (isEmpty(string)) {
			string = "";
		}
		int count = 0;
		String s = null;
		for (int i = 0; i < string.length(); i++) {
			s = String.valueOf(string.charAt(i));
			// 半角文字かそれ以外かチェック
			if (HALF_STRINGS.indexOf(s) != -1) {
				// 半角文字
				count += 1;
			} else {
				// 全角文字
				count += 2;
			}
		}
		if (count < n) {
			return true;
		}
		return false;
	}

	/**
	 * 画像ファイル拡張子チェック.
	 *
	 * @param chk_str 文字列（画像ファイル名）
	 * @return true -> しない false -> する
	 **/
	public static boolean isImgExtensionCheck(String chk_str) {

		if (chk_str.indexOf(".gif") == -1 && chk_str.indexOf(".jpg") == -1
				&& chk_str.indexOf(".jpeg") == -1
				&& chk_str.indexOf(".png") == -1
				&& chk_str.indexOf(".GIF") == -1
				&& chk_str.indexOf(".JPG") == -1
				&& chk_str.indexOf(".JPEG") == -1
				&& chk_str.indexOf(".PNG") == -1) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 日付比較チェック（前日付と後日付の整合性）
	 *
	 * @param ysi (開始年)
	 * @param msi (開始月)
	 * @param dsi (開始日)
	 * @param yei (終了年)
	 * @param mei (終了月)
	 * @param dei (終了日)
	 * @return boolean (true:NG false:OK)
	 */
	public static boolean beforeDayCheck(	int ysi,
											int yei,
											int msi,
											int mei,
											int dsi,
											int dei) {

		// 開始年月日が終了年月日より後の場合error
		if (ysi > yei) {
			// 西暦比較
			return true;
		} else if (ysi == yei) {
			// 西暦が同じの場合、月比較
			if (msi > mei) {
				return true;
			} else if (msi == mei) {
				// 西暦と月が同じ場合、日比較
				if (dsi > dei) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * 日付比較チェック2（終了日付と本日日付の整合性）
	 *
	 * @param yei (終了年)
	 * @param mei (終了月)
	 * @param dei (終了日)
	 * @return boolean (true:NG false:OK)
	 */
	public static boolean thisDayAfterCheck(int yei, int mei, int dei) {

		// 本日年月日を取得
		Calendar cal1 = Calendar.getInstance();
		int ty = cal1.get(Calendar.YEAR);
		int tm = cal1.get(Calendar.MONTH);
		int td = cal1.get(Calendar.DATE);

		// 終了年月日が本日年月日より前の場合error
		if (ty > yei) {
			// 西暦比較
			return true;
		} else if (ty == yei) {
			// 西暦が同じの場合、月比較
			if (tm > mei) {
				return true;
			} else if (tm == mei) {
				// 西暦と月が同じ場合、日比較
				if (td > dei) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * 半角英数字チェック.
	 *
	 * @param string
	 * @return boolean (true:半角英数字 false:半角英数字以外)
	 **/
	public static boolean isAlphaNumCheck(String string) {

		if (string.matches("^[a-zA-Z0-9]*$")) {
			return true;
		} else {
			return false;
		}

	}

	/**
	 * 半角英数字記号チェック.
	 *
	 * @param string
	 * @return boolean (true:半角英数字記号 false:半角英数字記号以外)
	 **/
	public static boolean isAlphaNumSignCheck(String string) {

		if (string.matches("^[a-zA-Z0-9 -/:-@\\[-\\`\\{-\\~]*$")) {
			return true;
		} else {
			return false;
		}

	}

	// Aladdin-EC ファッション版機能マージ_5685 ADD-S
	/**
	 * 半角英数字記号カナチェック
	 *
	 * @param string
	 * @return boolean (true:半角英数字記号カナ false:半角英数字記号カナ以外)
	 **/
	public static boolean isAlphaNumSignKanaCheck(String string) {
		if (string.matches("^[a-zA-Z0-9 -/:-@\\[-\\`\\{-\\~｡-ﾟ]*$")) {
			return true;
		} else {
			return false;
		}
	}

	// Aladdin-EC ファッション版機能マージ_5685 ADD-E

	/**
	 * 半角の英字,数字,記号のすべてを含むチェック.
	 *
	 * @param string
	 * @return boolean (true:半角の英字,数字,記号のすべてを含む false:半角の英字,数字,記号のすべてを含まない)
	 **/
	public static boolean isAlphaNumSignAllCheck(String string) {
		String alphaRegex = ".*[a-zA-Z]+.*";
		String numRegex = ".*[0-9]+.*";
		String signRegex = ".*[ -/:-@\\[-\\`\\{-\\~]+.*";

		if (isAlphaNumSignCheck(string) && string.matches(alphaRegex)
				&& string.matches(numRegex) && string.matches(signRegex)) {
			return true;
		} else {
			return false;
		}

	}

	/**
	 * 数字チェック.
	 *
	 * @param string
	 * @return boolean (true:数字 false:数字以外)
	 **/
	public static boolean isNumCheck(String string) {

		if (string.matches("^[0-9]*$")) {
			return true;
		} else {
			return false;
		}

	}

	/**
	 * 数値チェック.<br />
	 * マイナスを許容で0埋めNG<br />
	 *
	 * @param string
	 * @return boolean (true:数字 false:数字以外)
	 */
	public static boolean isNumerCheck(String string) {

		if (string.matches("^[-]?[0-9]*$")) {
			return true;
		} else {
			return false;
		}

	}

	/**
	 * ハイフン+数字チェック.
	 *
	 * @param string
	 * @return boolean (true:ハイフン+数字 false:ハイフン+数値以外)
	 **/
	public static boolean isNumHyphen(String string) {

		if (string.matches("^[0-9][0-9-]*-[0-9-]*[0-9]$")) {
			return true;
		} else {
			return false;
		}

	}

	/**
	 * 最大値超えチェック
	 *
	 * @param string 判定値
	 * @param max 最大値
	 * @return boolean (true:許容範囲 false:許容範囲外)
	 */
	public static boolean isAbove(String string, int max) {

		try {
			// 判定値
			BigDecimal a = new BigDecimal(string);
			// 最大値
			BigDecimal b = new BigDecimal(max);

			// 判定値が最大値よりも大きい場合
			if (a.compareTo(b) > 0) {
				return false;
			}

			return true;
		} catch (Exception e) {
			return false;
		}
	}

	/**
	 * 最小値未満チェック
	 *
	 * @param string 判定値
	 * @param min 最小値
	 * @return boolean (true:許容範囲 false:許容範囲外)
	 */
	public static boolean isFollowing(String string, int min) {

		try {
			// 判定値
			BigDecimal a = new BigDecimal(string);
			// 最小値
			BigDecimal b = new BigDecimal(min);

			// 判定値が最大値よりも大きい場合
			if (a.compareTo(b) < 0) {
				return false;
			}

			return true;
		} catch (Exception e) {
			return false;
		}

	}

	/**
	 * 半角英字チェック.
	 *
	 * @param string
	 * @return boolean (true:半角英字 false:半角英字以外)
	 */
	public static boolean isAlphaCheck(String string) {

		if (string.matches("^[a-zA-Z]*$")) {
			return true;
		} else {
			return false;
		}

	}

	/**
	 * 数字、半角英字がいずれも含まれるかチェック
	 *
	 * @param string 文字列
	 * @return boolean (true:含まれる false:含まれない)
	 */
	public static boolean isAlphaNumCheckAll(String string) {

		// チェックパターン
		String checkRegex = "^[0-9a-zA-Z]*$";
		// 数値が含まれるか
		String numRegex = "^.*[0-9]+.*$";
		// 半角英字が含まれるか
		String alphaRegex = "^.*[a-zA-Z]+.*$";

		// 数字、半角英字がいずれも含まれる場合
		if (string.matches(checkRegex) && string.matches(numRegex)
				&& string.matches(alphaRegex)) {
			return true;
		}
		return false;
	}

	/**
	 * 数字、半角英大文字、半角英小文字がいずれも含まれるかチェック
	 *
	 * @param string 文字列
	 * @return boolean (true:含まれる false:含まれない)
	 */
	public static boolean isAlphaSizeNumCheckAll(String string) {

		// チェックパターン
		String checkRegex = "^[0-9a-zA-Z]*$";
		// 数値が含まれるか
		String numRegex = "^.*[0-9]+.*$";
		// 半角大文字英字が含まれるか
		String upperRegex = "^.*[A-Z]+.*$";
		// 半角小文字英字が含まれるか
		String lowerRegex = "^.*[a-z]+.*$";

		// 数字、半角英大文字、半角英小文字がいずれも含まれる場合
		if (string.matches(checkRegex) && string.matches(numRegex)
				&& string.matches(upperRegex) && string.matches(lowerRegex)) {
			return true;
		}
		return false;
	}

	/**
	 * アラジンオフィス連携用の禁則文字チェック
	 *
	 * @param s 対象文字
	 * @return チェック結果(禁則含む:true 禁則含まず:false)
	 */
	public static boolean isNgWord(String s) {
		if (isEmpty(s)) {
			return false;
		}

		if (s.indexOf("'") != -1) {
			return true;
		}

		return false;
	}

	/**
	 * ブラウザがIEであるかどうかの判定を行います。
	 *
	 * @param userAgent ユーザエージェント
	 * @return IEであるかどうか
	 */
	public static boolean isIE(String userAgent) {
		if (isIE11(userAgent)) {
			return true;
		}

		Pattern pattern = Pattern.compile(".*((MSIE)+ [0-9]\\.[0-9]).*");
		Matcher matcher = pattern.matcher(userAgent);
		return matcher.matches();
	}

	/**
	 * ブラウザがIEであるかどうかの判定を行います。（IE11用）
	 *
	 * @param userAgent
	 * @return
	 */
	public static boolean isIE11(String userAgent) {
		// Aladdin-EC ファッション版機能マージ_5972 EDIT-S
		// Pattern pattern = Pattern.compile(".*((Trident/)+[0-9]\\.[0-9]).*");
		Pattern pattern = Pattern
				.compile(".*((Trident/)+[0-9]\\.[0-9]).*((rv:)+[0-9]+\\.[0-9]).*");
		// Aladdin-EC ファッション版機能マージ_5972 EDIT-E
		Matcher matcher = pattern.matcher(userAgent);
		return matcher.matches();
	}

	/**
	 * ブラウザがFirefoxであるかどうかの判定を行います。
	 *
	 * @param userAgent ユーザエージェント
	 * @return Firefoxであるかどうか
	 */
	public static boolean isFirefox(String userAgent) {
		Pattern pattern = Pattern
				.compile(".*((Firefox/)+[1-9]?[0-9]\\.[0-9]\\.?[0-9]?).*");
		Matcher matcher = pattern.matcher(userAgent);
		return matcher.matches();
	}

	/**
	 * ブラウザがOperaであるかどうかの判定を行います。
	 *
	 * @param userAgent ユーザエージェント
	 * @return Opereであるかどうか
	 */
	public static boolean isOpera(String userAgent) {
		Pattern pattern = Pattern
				.compile(".*((Opera)+/? ?[0-9]\\.[0-9][0-9]?).*");
		Matcher matcher = pattern.matcher(userAgent);
		return matcher.matches();

	}

	/**
	 * ブラウザがChromeであるかどうかの判定を行います。
	 *
	 * @param userAgent ユーザエージェント
	 * @return Chromeであるかどうか
	 */
	public static boolean isChrome(String userAgent) {
		Pattern pattern = Pattern.compile(".*((Chrome)+/?[0-9]\\.?[0-9]?).*");
		Matcher matcher = pattern.matcher(userAgent);
		return matcher.matches();
	}

	/**
	 * ブラウザがSafariであるかどうかの判定を行います。
	 *
	 * @param userAgent ユーザエージェント
	 * @return Safariであるかどうか
	 */
	public static boolean isSafari(String userAgent) {
		Pattern pattern = Pattern
				.compile(".*((Version/)+[0-9]\\.?[0-9]?\\.?[0-9]? Safari).*");
		Matcher matcher = pattern.matcher(userAgent);
		return matcher.matches();
	}

	/**
	 * ブラウザがNetscapeであるかどうかの判定を行います。
	 *
	 * @param userAgent ユーザエージェント
	 * @return Netscapeであるかどうか
	 */
	public static boolean isNetscape(String userAgent) {
		Pattern pattern = Pattern
				.compile(".*((Netscape)+[0-9]\\.[0-9][0-9]?).*");
		Matcher matcher = pattern.matcher(userAgent);
		return matcher.matches();
	}


	/**
	 * 浮動小数チェック
	 *
	 * @param str
	 * @return boolean (true:doubleに変換できる false:doubleに変換できない)
	 **/
	public static boolean isDouble(String str) {
		boolean result = false;
		try {
			Double.parseDouble(str);
			result = true;
		} catch (Exception e) {
		}
		return result;
	}

	/**
	 * 半角英数記号チェック(半角スペースは除く).
	 *
	 * @param string
	 * @return boolean (true:半角英数記号 false:半角英数記号以外)
	 */
	public static boolean isAlphaNumSignCheckWithoutSpace(String string) {

		if (string.matches("^[a-zA-Z0-9!-/:-@\\[-\\`\\{-\\~]*$")) {
			return true;
		} else {
			return false;
		}

	}

	/**
	 * 数字、半角英字、半角記号がいずれも含まれるかチェック
	 *
	 * @param string 文字列
	 * @return boolean (true:含まれる false:含まれない)
	 */
	public static boolean isAlphaNumSignCheckAll(String string) {

		// チェックパターン
		String checkRegex = "^[0-9a-zA-Z!-/:-@\\[-\\`\\{-\\~]*$";
		// 数値が含まれるか
		String numRegex = "^.*[0-9]+.*$";
		// 半角英字が含まれるか
		String alphaRegex = "^.*[a-zA-Z]+.*$";
		// 半角記号が含まれるか
		String signRegex = "^.*[!-/:-@\\[-\\`\\{-\\~]+.*$";

		// 数字、半角英字がいずれも含まれる場合
		if (string.matches(checkRegex) && string.matches(numRegex)
				&& string.matches(alphaRegex) && string.matches(signRegex)) {
			return true;
		}
		return false;
	}

	/**
	 * 数字、半角英大文字、半角英小文字がいずれも含まれるかチェック
	 *
	 * @param string 文字列
	 * @return boolean (true:含まれる false:含まれない)
	 */
	public static boolean isAlphaSizeNumSignCheckAll(String string) {

		// チェックパターン
		String checkRegex = "^[0-9a-zA-Z!-/:-@\\[-\\`\\{-\\~]*$";
		// 数値が含まれるか
		String numRegex = "^.*[0-9]+.*$";
		// 半角大文字英字が含まれるか
		String upperRegex = "^.*[A-Z]+.*$";
		// 半角小文字英字が含まれるか
		String lowerRegex = "^.*[a-z]+.*$";
		// 半角記号が含まれるか
		String signRegex = "^.*[!-/:-@\\[-\\`\\{-\\~]+.*$";

		// 数字、半角英大文字、半角英小文字がいずれも含まれる場合
		if (string.matches(checkRegex) && string.matches(numRegex)
				&& string.matches(upperRegex) && string.matches(lowerRegex)
				&& string.matches(signRegex)) {
			return true;
		}
		return false;
	}

	/**
	 * 郵便番号フォーマットチェック
	 *
	 * @param string 文字列
	 * @return boolean (true:郵便番号である false:郵便番号でない)
	 */
	public static boolean isZipFormatCheck(String string) {

		if (isEmpty(string)) {
			return true;
		}
		// チェックパターン
		String checkRegex = "^[0-9\\-]+$";
		// チェックパターンに該当
		if (string.matches(checkRegex)) {
			return true;
		}
		return false;

	}

	/**
	 * 電話番号フォーマットチェック.
	 *
	 * @param string 文字列
	 * @return boolean (true:電話番号である false:電話番号でない)
	 */
	public static boolean isPhoneNumFormatCheck(String string) {

		if (isEmpty(string)) {
			return true;
		}
		// チェックパターン
		String checkRegex = "^[0-9\\-\\+\\s]+$";
		// チェックパターンに該当
		if (string.matches(checkRegex)) {
			return true;
		}
		return false;

	}

	/**
	 * 文字数制限チェック.
	 *
	 * @param string
	 * @return boolean (true:制限エラー false:OK)
	 * @throws UnsupportedEncodingException
	 */
	public static boolean isLengthCheckUtf8(String string, int i)
			throws UnsupportedEncodingException {

		if (isEmpty(string)) {
			return false;
		}

		int j = string.getBytes("UTF-8").length;

		if (j > i) {
			return true;
		}
		return false;
	}

	/**
	 * 文字列が -922337203685477.5808～922337203685477.5807の範囲外であるか確認する.
	 *
	 * @param str 文字列
	 * @return チェック結果(範囲外:true 範囲内:false)
	 */
	public static boolean isDecimalRangeOut(String str) {

		if (isEmpty(str)) {
			return false;
		}

		BigDecimal check = null;
		try {
			check = new BigDecimal(str);
		} catch (Exception e) {

			return true;
		}

		BigDecimal max = new BigDecimal("922337203685477.5807");
		BigDecimal min = new BigDecimal("-922337203685477.5808");

		if (max.compareTo(check) >= 0 && min.compareTo(check) <= 0) {
			return false;
		}
		return true;
	}

	/**
	 * 日付フォーマットをチェックする.
	 *
	 * @param str 日付文字列
	 * @param format フォーマット文字列（SimpleDateFormatに依存）
	 * @return チェック結果(正常:true 異常:false)
	 */
	public static boolean isDateFormat(String str, String format) {
		if (isEmpty(str)) {
			return true;
		}

		SimpleDateFormat formatter = new SimpleDateFormat(format);
		try {
			Date result = formatter.parse(str);
			formatter.setLenient(false);

			// フォーマットを元に戻して引数と比較
			String reverse = formatter.format(result);
			return str.equals(reverse) ? true : false;

		} catch (ParseException e) {
			return false;
		}

	}

	// Rev.12>>> Aladdin-EC ファッション版対応 確認_8 ADD-S
	/**
	 * 通貨型チェック
	 *
	 * @param str 文字列
	 * @return チェック結果(正常:true 異常:false)
	 */
	public static boolean isBigDecimal(String str) {
		boolean result = false;
		try {
			new BigDecimal(str);
			result = true;
		} catch (Exception e) {
		}
		return result;
	}

	// <<<Rev.12 Aladdin-EC ファッション版対応 確認_8 ADD-E
	/**
	 * コレクションがnull、もしくはsizeが0かどうかを調べる。
	 *
	 * @param collection コレクション
	 * @return true ->コレクションがnull、もしくはsizeが0 <br />
	 *         false -> コレクションはnullでもsize0でもない
	 */
	public static <T> boolean isCollectionEmpty(Collection<T> collection) {
		return (collection == null) ? true : collection.isEmpty();
	}

	/**
	 * MAPがnull、もしくはsizeが0かどうかを調べる。
	 *
	 * @param map マップ
	 * @return true ->マップがnull、もしくはsizeが0 <br />
	 *         false -> マップはnullでもsize0でもない
	 */
	public static <K, V> boolean isMapEmpty(Map<K, V> map) {
		return (map == null) ? true : map.isEmpty();
	}

	/**
	 * 文字数制限チェック<br>
	 * 半角を1全角を2としてチェックを行います。<br>
	 * nullは0文字の文字列として扱います。
	 *
	 * @param string チェックする文字列
	 * @param n チェックする文字数
	 * @return boolean (true:制限エラー false:OK)
	 */
	public static boolean isLengthCheckWidthJust(String string, int n) {
		if (isEmpty(string)) {
			string = "";
		}
		int count = 0;
		String s = null;
		for (int i = 0; i < string.length(); i++) {
			s = String.valueOf(string.charAt(i));
			// 半角文字かそれ以外かチェック
			if (HALF_STRINGS.indexOf(s) != -1) {
				// 半角文字
				count += 1;
			} else {
				// 全角文字
				count += 2;
			}
		}
		if (count != n) {
			return true;
		}
		return false;
	}

	/**
	 * 文字数制限チェック<br>
	 * 指定された文字数が、特定の桁数の範囲内稼働かを返します。<br>
	 * 半角を1全角を2としてチェックを行います。<br>
	 * nullは0文字の文字列として扱います。
	 *
	 * @param string チェックする文字列
	 * @param n チェックする文字数(from)
	 * @param n チェックする文字数(to)
	 * @return boolean (true:制限エラー false:OK)
	 */
	public static boolean isLengthCheckWidthRange(	String string,
													int min,
													int max) {
		if (isEmpty(string)) {
			string = "";
		}
		int count = 0;
		String s = null;
		for (int i = 0; i < string.length(); i++) {
			s = String.valueOf(string.charAt(i));
			// 半角文字かそれ以外かチェック
			if (HALF_STRINGS.indexOf(s) != -1) {
				// 半角文字
				count += 1;
			} else {
				// 全角文字
				count += 2;
			}
		}
		if (min > count || count > max) {
			return true;
		}
		return false;
	}

	/**
	 * 数値が等しいか調べる
	 *
	 * @param string 値1
	 * @param string 値2
	 * @return boolean (true:等しい false:等しくない)
	 */
	public static boolean isEqual(String string1, String string2) {

		try {
			// 判定値
			BigDecimal a = new BigDecimal(string1);
			// 最大値
			BigDecimal b = new BigDecimal(string2);

			// 判定値が最大値よりも大きい場合
			if (a.compareTo(b) == 0) {
				return true;
			}

			return false;
		} catch (Exception e) {
			return false;
		}
	}

	/**
	 * ファイル名の拡張子が、拡張子リストに存在しているかチェックする.
	 *
	 * @param chk_str ファイル名
	 * @param extList 拡張子リスト（区切り文字複数指定可）
	 * @param splitter 区切り文字
	 * @return true -> 拡張子リストに存在 false -> 拡張子リストに存在しない
	 */
	public static boolean hasExtensionListCheck(String chk_str,
												String extList,
												String splitter) {

		if (isEmpty(extList)) {
			return true;
		}
		// 拡張子が存在しない場合はエラー
		int point = chk_str.lastIndexOf(".");
		if (point == -1) {
			return false;
		}

		String chkExt = chk_str.substring(point + 1);
		String[] extArray = extList.split(splitter);
		for (String ext : extArray) {
			if (chkExt.equals(ext)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 最大値超えチェック
	 *
	 * @param string 判定値
	 * @param max 最大値
	 * @return boolean (true:許容範囲 false:許容範囲外)
	 */
	public static boolean isAbove(String string, String max) {

		try {
			// 判定値
			BigDecimal a = new BigDecimal(string);
			// 最大値
			BigDecimal b = new BigDecimal(max);

			// 判定値が最大値よりも大きい場合
			if (a.compareTo(b) > 0) {
				return false;
			}

			return true;
		} catch (Exception e) {
			return false;
		}
	}

	/**
	 * 最小値未満チェック
	 *
	 * @param string 判定値
	 * @param min 最小値
	 * @return boolean (true:許容範囲 false:許容範囲外)
	 */
	public static boolean isFollowing(String string, String min) {

		try {
			// 判定値
			BigDecimal a = new BigDecimal(string);
			// 最小値
			BigDecimal b = new BigDecimal(min);

			// 判定値が最小値よりも小さい場合
			if (a.compareTo(b) < 0) {
				return false;
			}

			return true;
		} catch (Exception e) {
			return false;
		}

	}




	/**
	 * コレクションがnull、もしくはsizeが0かどうかを調べる。
	 *
	 * @param value 入力文字列
	 * @return true ->入力文字列は半角カナのみで構成されている <br />
	 *         false ->入力文字列は半角カナのみでは構成されていない
	 */
	public static boolean isHalfKanaOnly(String value) {
		Pattern pattern = Pattern.compile("^[\\uFF65-\\uFF9F\\ｰ]+$");
		Matcher matcher = pattern.matcher(value);
		return matcher.matches();
	}

	/**
	 * SMBC通信時郵便番号フォーマットチェック
	 *
	 * @param string 文字列
	 * @return boolean (true:郵便番号である false:郵便番号でない)
	 */
	public static boolean isSmbcZipFormatCheck(String string) {

		if (isEmpty(string)) {
			return true;
		}
		// チェックパターン
		String patternNoHyphen = "^[0-9]{7}$";
		String patternHasHyphen = "^[0-9]{3}\\-[0-9]{4}$";
		// チェックパターンに該当
		if (string.matches(patternNoHyphen) || string.matches(patternHasHyphen)) {
			return true;
		}
		return false;

	}

	/* アラジンECtoB 3rdバージョンリリース【オープンサイト承認あり版】 START */
	/**
	 * 区分の存在確認.
	 *
	 * @param map 区分MAP
	 * @param str 区分値
	 * @return
	 */
	public static boolean existDiv(Map<String, String> map, String str) {

		// 入力が無い場合はtrueで返す
		if (CheckUtil.isEmpty(str)) {
			return true;
		}

		if (map.containsKey(str)) {
			return true;
		}
		return false;
	}

	/* アラジンECtoB 3rdバージョンリリース【オープンサイト承認あり版】 END */

	/**
	 *
	 * 引き渡された文字列の整数部が"0"で始まっているか.
	 *
	 * @param string 検証値
	 * @return
	 */
	public static boolean isValidInteger(String string) {

		if (isEmpty(string)) {
			return false;
		}

		String val = StringUtils.replace(string, ",", "");
		if (isEmpty(val)) {
			return false;
		}

		String checkVal = "";
		if (val.indexOf(".") == -1) {

			checkVal = val;

		} else {

			String[] arr = StringUtils.split(val, ".");

			if (arr == null) {
				return false;
			}

			if (arr.length != 2) {
				return false;
			}

			checkVal = arr[0];

		}

		return checkVal.matches("^\\-?(0|[1-9][0-9]*)$");

	}


	/**
	 * 空(NULL)チェック.<br>
	 *
	 * @param string
	 * @return boolean (true:空、NULL false:空、NULL以外)
	 **/
	@SuppressWarnings("rawtypes")
	public static boolean isBlankObject(Object value) {

		if (value == null) {
			return true;
		}

		if (value instanceof String) {
			String values = (String)value;
			if (values.isEmpty()) {
				return true;
			}
		} else if (value instanceof String[]) {
			String[] values = (String[])value;
			if (values.length == 0) {
				return true;
			}
		} else if (value instanceof List) {
			List values = (List)value;
			if (values.isEmpty()) {
				return true;
			}
		} else if (value instanceof Map) {
			Map values = (Map)value;
			if (values.isEmpty()) {
				return true;
			}
		}

		return false;

	}

}
