package jp.ill.photon.util;

import java.math.BigDecimal;
import java.util.LinkedHashMap;

public class AoUtil {

	// 整数桁のデフォルト値
	private static final String DEFAULT_DIGIT_NUM = "9";
	// 小数点桁のデフォルト値
	private static final String DEFAULT_DECIMAL_POINTNUM = "0";
	// 端数処理区分のデフォルト値
	private static final String DEFAULT_ROUNDING_DIV = "5";

	/**
	 * 数値の表示パターン<br>
	 * デフォルトは表示パターン1になります。<br>
	 * 必要なパターンのみ定義しています。<br>
	 *
	 * ■表示パターン1<br>
	 * 0 → 0<br>
	 * 12,345.00 → 12,345.00<br>
	 * 12,345.12 → 12,345.12<br>
	 *
	 * ■表示パターン(カンマなし)<br>
	 * 0 → 0<br>
	 * 12,345.00 → 12345.00<br>
	 * 12,345.12 → 12345.12<br>
	 */
	public enum DispPattern {
		PATTERN_1, PATTERN_NO_COMMA,
	}

	/**
	 * 売上数量フォーマット処理<br>
	 * 表示パターン1で表示します。<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 売上数量
	 * @param digitNum 売上数量整数桁
	 * @param decimalPointNum 売上数量小数点桁
	 * @param roundingDiv 売上数量端数処理区分
	 * @return String フォーマット変換後の売上数量
	 */
	public static String convAoRoundingSalesQuantity(	String value,
			String digitNum,
			String decimalPointNum,
			String roundingDiv) {
		return convAoRoundingSalesQuantity(value, DispPattern.PATTERN_1,
				digitNum, decimalPointNum, roundingDiv);
	}

	/**
	 * 売上数量フォーマット処理<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 売上数量
	 * @param dispPattern 表示パターン
	 * @param digitNum 売上数量整数桁
	 * @param decimalPointNum 売上数量小数点桁
	 * @param roundingDiv 売上数量端数処理区分
	 * @return String フォーマット変換後の売上数量
	 */
	public static String convAoRoundingSalesQuantity(	String value,
														DispPattern dispPattern,
														String digitNum,
														String decimalPointNum,
														String roundingDiv) {

		// フォーマット変更
		String strResult = convAoRounding(value, "1", digitNum,
				decimalPointNum, roundingDiv, dispPattern);

		return strResult;
	}

	// Aladdin-EC 標準化_ユーザー51 20140430 ADD-S
	/**
	 * 売上数量フォーマット処理<br>
	 * 表示パターン1で表示します。<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 入数
	 * @param value2 ケース数
	 * @param digitNum 売上数量整数桁
	 * @param decimalPointNum 売上数量小数点桁
	 * @param roundingDiv 売上数量端数処理区分
	 * @return String フォーマット変換後の売上数量
	 */
	public static String convAoRoundingSalesQuantity(	String value,
														String value2,
														String digitNum,
														String decimalPointNum,
														String roundingDiv) {
		return convAoRoundingSalesQuantity(value, value2,
				DispPattern.PATTERN_1, digitNum, decimalPointNum, roundingDiv);
	}

	/**
	 * 売上数量フォーマット処理<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 入数
	 * @param value2 ケース数
	 * @param dispPattern 表示パターン
	 * @param digitNum 売上数量整数桁
	 * @param decimalPointNum 売上数量小数点桁
	 * @param roundingDiv 売上数量端数処理区分
	 * @return String フォーマット変換後の売上数量
	 */
	public static String convAoRoundingSalesQuantity(	String value,
														String value2,
														DispPattern dispPattern,
														String digitNum,
														String decimalPointNum,
														String roundingDiv) {

		// フォーマット変更
		String strResult = convAoRounding(value, value2, digitNum,
				decimalPointNum, roundingDiv, dispPattern);

		return strResult;
	}

	// Aladdin-EC 標準化_ユーザー51 20140430 ADD-E

	/**
	 * 売上単価フォーマット処理<br>
	 * 表示パターン1で表示します。<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 売上単価
	 * @param digitNum 売上単価整数桁
	 * @param decimalPointNum 売上単価小数点桁
	 * @param roundingDiv 売上単価端数処理区分
	 * @return String フォーマット変換後の売上単価
	 */
	public static String convAoRoundingSalesUnitPrice(	String value,
			String digitNum,
			String decimalPointNum,
			String roundingDiv) {
		return convAoRoundingSalesUnitPrice(value, DispPattern.PATTERN_1,
				digitNum, decimalPointNum, roundingDiv);
	}

	/**
	 * 売上単価フォーマット処理<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 売上単価
	 * @param dispPattern 表示パターン
	 * @param digitNum 売上単価整数桁
	 * @param decimalPointNum 売上単価小数点桁
	 * @param roundingDiv 売上単価端数処理区分
	 * @return String フォーマット変換後の売上単価
	 */
	public static String convAoRoundingSalesUnitPrice(	String value,
														DispPattern dispPattern,
														String digitNum,
														String decimalPointNum,
														String roundingDiv) {
		// フォーマット変更
		String strResult = convAoRounding(value, "1", digitNum,
				decimalPointNum, roundingDiv, dispPattern);

		return strResult;
	}

	/**
	 * 売上金額フォーマット処理<br>
	 * 表示パターン1で表示します。<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 売上金額
	 * @param digitNum 売上金額整数桁
	 * @param roundingDiv 売上金額端数処理区分
	 * @return String フォーマット変換後の売上金額
	 */
	public static String convAoRoundingSalesPrice(	String value,
			String digitNum,
			String roundingDiv) {
		return convAoRoundingSalesPrice(value, "1", DispPattern.PATTERN_1,
				digitNum, roundingDiv);
	}

	/**
	 * 売上金額フォーマット処理<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 売上金額
	 * @param dispPattern 表示パターン
	 * @param digitNum 売上金額整数桁
	 * @param roundingDiv 売上金額端数処理区分
	 * @return String フォーマット変換後の売上金額
	 */
	public static String convAoRoundingSalesPrice(	String value,
													DispPattern dispPattern,
													String digitNum,
													String roundingDiv) {
		return convAoRoundingSalesPrice(value, "1", dispPattern, digitNum, roundingDiv);
	}

	/**
	 * 売上金額フォーマット処理<br>
	 * 表示パターン1で表示します。<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 売上単価
	 * @param value2 注文数量
	 * @param digitNum 売上金額整数桁
	 * @param roundingDiv 売上金額端数処理区分
	 * @return String フォーマット変換後の売上金額
	 */
	public static String convAoRoundingSalesPrice(	String value,
													String value2,
													String digitNum,
													String roundingDiv) {
		return convAoRoundingSalesPrice(value, value2, DispPattern.PATTERN_1,
				digitNum, roundingDiv);
	}

	/**
	 * 売上金額フォーマット処理<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 売上単価
	 * @param value2 注文数量
	 * @param dispPattern 表示パターン
	 * @param digitNum 売上金額整数桁
	 * @param roundingDiv 売上数量端数処理区分
	 * @return String フォーマット変換後の売上金額
	 */
	public static String convAoRoundingSalesPrice(	String value,
													String value2,
													DispPattern dispPattern,
													String digitNum,
													String roundingDiv) {

		// 小数点桁は固定
		String decimalPointNum = "0";

		// フォーマット変更
		String strResult = convAoRounding(value, value2, digitNum,
				decimalPointNum, roundingDiv, dispPattern);

		return strResult;
	}

	/**
	 * 在庫数量フォーマット処理<br>
	 * 表示パターン1で表示します。<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 在庫数量
	 * @param dispPattern 表示パターン
	 * @param digitNum 在庫数量整数桁
	 * @param decimalPointNum 在庫数量小数点桁
	 * @return String フォーマット変換後の在庫数量
	 */
	public static String convAoRoundingStockQuantity(	String value,
			String digitNum,
			String decimalPointNum,
			String roundingDiv) {
		return convAoRoundingStockQuantity(value, DispPattern.PATTERN_1,
				digitNum, decimalPointNum, roundingDiv);
	}

	/**
	 * 在庫数量フォーマット処理<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 在庫数量
	 * @param dispPattern 表示パターン
	 * @param digitNum 在庫数量整数桁
	 * @param decimalPointNum 在庫数量小数点桁
	 * @return String フォーマット変換後の在庫数量
	 */
	public static String convAoRoundingStockQuantity(	String value,
														DispPattern dispPattern,
														String digitNum,
														String decimalPointNum,
														String roundingDiv) {
		// 端数処理区分は固定
		//String roundingDiv = "5";

		// フォーマット変更
		String strResult = convAoRounding(value, "1", digitNum,
				decimalPointNum, roundingDiv, dispPattern);

		return strResult;
	}

	// Aladdin-EC 標準化_ユーザー51 20140416 ADD-S
	/**
	 * 入数フォーマット処理<br>
	 * 表示パターン1で表示します。<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 入数
	 * @param digitNum 入数整数桁
	 * @param decimalPointNum 入数小数点桁
	 * @param roundingDiv 端数処理区分
	 * @return String フォーマット変換後の入数
	 */
	public static String convAoRoundingUnitQuantity(String value,
			String digitNum,
			String decimalPointNum,
			String roundingDiv) {
		return convAoRoundingUnitQuantity(value, DispPattern.PATTERN_1,
				digitNum, decimalPointNum, roundingDiv);
	}

	/**
	 * 入数フォーマット処理<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 入数
	 * @param dispPattern 表示パターン
	 * @param digitNum 入数整数桁
	 * @param decimalPointNum 入数小数点桁
	 * @param roundingDiv 端数処理区分
	 * @return String フォーマット変換後の入数
	 */
	public static String convAoRoundingUnitQuantity(String value,
													DispPattern dispPattern,
													String digitNum,
													String decimalPointNum,
													String roundingDiv) {

		// フォーマット変更
		String strResult = convAoRounding(value, "1", digitNum,
				decimalPointNum, roundingDiv, dispPattern);

		return strResult;
	}

	// Aladdin-EC 標準化_ユーザー51 20140416 ADD-E
	// Aladdin-EC 標準化_管理28 20140409 ADD-S
	/**
	 * ケース数フォーマット処理<br>
	 * 表示パターン1で表示します。<br>
	 * ※端数処理も含みます。
	 *
	 * @param value ケース数
	 * @param digitNum ケース数整数桁
	 * @param decimalPointNum ケース数小数点桁
	 * @param roundingDiv 端数処理区分
	 * @return String フォーマット変換後のケース数
	 */
	public static String convAoRoundingCaseQuantity(String value,
			String digitNum,
			String decimalPointNum,
			String roundingDiv) {
		return convAoRoundingCaseQuantity(value, DispPattern.PATTERN_1,
				digitNum, decimalPointNum, roundingDiv);
	}

	/**
	 * ケース数フォーマット処理<br>
	 * ※端数処理も含みます。
	 *
	 * @param value ケース数
	 * @param dispPattern 表示パターン
	 * @param digitNum ケース数整数桁
	 * @param decimalPointNum ケース数小数点桁
	 * @param roundingDiv 端数処理区分
	 * @return String フォーマット変換後のケース数
	 */
	public static String convAoRoundingCaseQuantity(String value,
													DispPattern dispPattern,
													String digitNum,
													String decimalPointNum,
													String roundingDiv) {

		// フォーマット変更
		String strResult = convAoRounding(value, "1", digitNum,
				decimalPointNum, roundingDiv, dispPattern);

		return strResult;
	}

	// Aladdin-EC 標準化_管理28 20140409 ADD-E

	/**
	 * AOフォーマット処理<br>
	 * ※端数処理も含みます。
	 *
	 * @param value 変換する値(単価)
	 * @param value2 変換する値(数量)
	 * @param digitNum 整数桁
	 * @param decimalPointNum 小数点桁
	 * @param roundingDiv 端数処理区分
	 * @param dispPattern 表示パターン
	 * @return String フォーマット変換後の値
	 */
	public static String convAoRounding(	String value,
											String value2,
											String digitNum,
											String decimalPointNum,
											String roundingDiv,
											DispPattern dispPattern) {
		// カンマ削除
		value = value.replace(",", "");
		value2 = value2.replace(",", "");

		// 変換する値のチェック
		if (!CheckUtil.isDouble(value)) {
			value = "0";
		}
		// 前後のスペースを削除
		value = value.trim();
		// 変換する値のチェック
		if (!CheckUtil.isDouble(value2)) {
			value2 = "0";
		}
		// 前後のスペースを削除
		value2 = value2.trim();

		// 整数桁・小数点桁・端数処理区分が取得できない場合はデフォルトの値を設定
		if (digitNum == null || !CheckUtil.isNumCheck(digitNum)) {
			digitNum = DEFAULT_DIGIT_NUM;
		}
		if (decimalPointNum == null || !CheckUtil.isNumCheck(decimalPointNum)) {
			decimalPointNum = DEFAULT_DECIMAL_POINTNUM;
		}
		if (roundingDiv == null || !CheckUtil.isNumCheck(roundingDiv)) {
			roundingDiv = DEFAULT_ROUNDING_DIV;
		}

		// 端数処理(単価 * 数量)
		BigDecimal bdValue = new BigDecimal(value);
		BigDecimal bdValue2 = new BigDecimal(value2);
		BigDecimal bdResult = null;
		// 端数処理区分により端数処理を行う
		if (FractionDiv.DOWN.equals(roundingDiv)) {
			// 切り捨て
			bdResult = bdValue.multiply(bdValue2).setScale(
					Integer.parseInt(decimalPointNum), BigDecimal.ROUND_DOWN);
		} else if (FractionDiv.HALF_UP.equals(roundingDiv)) {
			// 四捨五入
			bdResult = bdValue.multiply(bdValue2)
					.setScale(Integer.parseInt(decimalPointNum),
							BigDecimal.ROUND_HALF_UP);
		} else if (FractionDiv.UP.equals(roundingDiv)) {
			// 切り上げ
			bdResult = bdValue.multiply(bdValue2).setScale(
					Integer.parseInt(decimalPointNum), BigDecimal.ROUND_UP);
		}
		String strRounding = bdResult.toString();

		// 整数部と小数部の桁数のフォーマット変更
		int valueDotPos = strRounding.indexOf(".");
		String valueIntgerPart = "";
		String valueDecimalPart = "";
		if (valueDotPos != -1) {
			valueIntgerPart = strRounding.substring(0, valueDotPos);
			valueDecimalPart = strRounding.substring(valueDotPos + 1);
		} else {
			valueIntgerPart = strRounding;
			valueDecimalPart = "";
		}

		// 小数桁数のフォーマット変更
		String strResult = "";
		if (Integer.parseInt(decimalPointNum) != 0) {
			int valueIntgerPartLen = valueDecimalPart.length();
			for (int i = valueIntgerPartLen; i < Integer
					.parseInt(decimalPointNum); i++) {
				valueDecimalPart += "0";
			}
		}
		// カンマ区切り
		if (DispPattern.PATTERN_1.equals(dispPattern)) {
			// パターン1
			// 整数部はカンマ区切りへの変換のみ
			valueIntgerPart = convDecimalFormat(valueIntgerPart);
		}

		// 結果の作成
		if (Integer.parseInt(decimalPointNum) != 0) {
			// 小数点以下あり
			strResult = valueIntgerPart + "." + valueDecimalPart;
		} else {
			// 小数点以下なし
			strResult = valueIntgerPart;
		}

		return strResult;
	}

	/**
	 * 数値文字列をカンマ区切りに変更<br>
	 *
	 * @param str 数値文字列
	 * @return 変換後の数値文字列
	 */
	public static String convDecimalFormat(String str) {
		String result = "";
		if (str == null)
			return result;
		// 整数部と小数部を分解
		int dotPos = str.indexOf(".");
		String intgerPart = "";
		String decimalPart = "";
		if (dotPos != -1) {
			intgerPart = str.substring(0, dotPos);
			decimalPart = str.substring(dotPos + 1);
		} else {
			intgerPart = str;
			decimalPart = "";
		}
		// マイナスを分解
		boolean minus = false;
		if (intgerPart.indexOf("-") == 0) {
			minus = true;
			intgerPart = intgerPart.substring(1);
		}
		// 整数部をカンマ区切りに変更
		int cnt = 0;
		for (int i = intgerPart.length() - 1; i >= 0; i--) {
			cnt++;
			if (cnt > 3) {
				cnt = 1;
				result = "," + result;
			}
			char c = intgerPart.charAt(i);
			result = String.valueOf(c) + result;
		}
		// 小数部があれば結合
		if (!"".equals(decimalPart)) {
			result = result + "." + decimalPart;
		}
		// マイナスがあれば結合
		if (minus) {
			result = "-" + result;
		}
		return result;
	}

	/** 端数処理区分 */
	public static class FractionDiv {
		/** 切り捨て */
		public static final String DOWN = "0";
		/** 四捨五入 */
		public static final String HALF_UP = "5";
		/** 切り上げ */
		public static final String UP = "9";
		/** マップ */
		/* Aladdin-EC ファッション版 #8080 START */
		public static final LinkedHashMap<String, String> FRACTION_DIV_MAP = new LinkedHashMap<String, String>();
		static {
			FRACTION_DIV_MAP.put(DOWN, "切り捨て");
			FRACTION_DIV_MAP.put(HALF_UP, "四捨五入");
			FRACTION_DIV_MAP.put(UP, "切り上げ");
		}
		/* Aladdin-EC ファッション版 #8080 END */
	}

}
