package jp.ill.photon.util;

import java.util.HashMap;
import java.util.Map;

import jp.ill.photon.model.AoSetting;

import org.apache.commons.lang.StringUtils;

/**
 * AOフォーマットチェック用ユーティリティクラス.
 */
public class AoFormatCheckUtil {

	/* AOフォーマット区分 */
	public static class AoFormatDiv {
		/** 売上数量 */
		public static final String SALES_QUANTITY_KN = "売上数量";
		public static final String SALES_QUANTITY = "salesquantity";
		/** 売上単価 */
		public static final String SALES_UNIT_PRICE_KN = "売上単価";
		public static final String SALES_UNIT_PRICE = "unitprice";
		/** 売上金額 */
		public static final String SALES_PRICE_KN = "売上金額";
		public static final String SALES_PRICE = "salesprice";
		/** 在庫数量 */
		public static final String STOCK_QUANTITY_KN = "在庫数量";
		public static final String STOCK_QUANTITY = "stockquantity";
		/** 入数 */
		public static final String UNIT_QUANTITY_KN = "入数";
		public static final String UNIT_QUANTITY = "unitquantity";
		/** ケース数 */
		public static final String CASE_QUANTITY_KN = "ケース数";
		public static final String CASE_QUANTITY = "casequantity";
	}

	public static final String KEY_DIGIT= "digit";
	public static final String KEY_DECIMAL_DIGIT = "decimal_digit";


	/***
	 *
	 * カンマ除去
	 *
	 * @param input 入力
	 * @return カンマ除去後の文字列
	 *
	 */
	public static String removeComma(String input) {
		return StringUtils.defaultString(input).replace(",", "").trim();
	}

	/** 共通 ------------------------------------------------------------------------------------------------------ */
	/**
	 * 汎用AOフォーマットチェック
	 *
	 * @param aoFormatDiv AOフォーマット区分(売上数量/売上単価/売上金額/在庫数量/入数/ケース数)
	 * @param value チェック対象値
	 * @param aoSetting アラジンシステム設定
	 * @return チェック結果(true: OK, false: NG)
	 */
	public static boolean aoFormatCheck(String aoFormatDiv, String value, AoSetting aoSetting) {

		// チェック対象値が設定されていない場合
		if (UtilTools.isEmpty(value)) {
			return false;
		}

		// カンマ除去
		value = value.replaceAll(",","").trim();

		// チェック対象値が数値として認識できない場合
		if (!CheckUtil.isBigDecimal(value)) {
			return false;
		}

		// AOフォーマット区分が設定されていない場合
		if (UtilTools.isEmpty(aoFormatDiv)) {
			return false;
		}

		/** 売上数量フォーマットチェック */
		if (AoFormatDiv.SALES_QUANTITY.equals(aoFormatDiv.replace("_","").toLowerCase())
				|| AoFormatDiv.SALES_QUANTITY_KN.equals(aoFormatDiv)){

			// 売上数量整数桁
			String digit = aoSetting.getSalesQuantityDigitNum();
			// 売上数量小数桁
			String decimalDigit = aoSetting.getSalesQuantityDecimalPointNum();
			// 売上数量パターンの作成
			String matchPattern = digitDecimalRegPattern(digit, decimalDigit);
			// 売上数量フォーマットチェック
			if (!value.matches(matchPattern)) {
				return false;
			}
			return true;

		/** 売上単価フォーマットチェック */
		} else 	if (AoFormatDiv.SALES_UNIT_PRICE.equals(aoFormatDiv.replace("_","").toLowerCase())
				|| AoFormatDiv.SALES_UNIT_PRICE_KN.equals(aoFormatDiv)){

			// 売上単価整数桁
			String digit = aoSetting.getSalesUnitPriceDigitNum();
			// 売上単価小数桁
			String decimalDigit = aoSetting.getSalesUnitPriceDecimalPointNum();
			// 売上単価パターンの作成
			String matchPattern = digitDecimalRegPattern(digit, decimalDigit);
			// 売上単価フォーマットチェック
			if (!value.matches(matchPattern)) {
				return false;
			}
			return true;


		/** 売上金額フォーマットチェック */
		} else 	if (AoFormatDiv.SALES_PRICE.equals(aoFormatDiv.replace("_","").toLowerCase())
				|| AoFormatDiv.SALES_PRICE_KN.equals(aoFormatDiv)){

			// 売上金額整数桁
			String digit = aoSetting.getSalesPriceDigitNum();
			// 売上金額パターンの作成
			String matchPattern = digitRegPattern(digit);
			// 売上金額フォーマットチェック
			if (!value.matches(matchPattern)) {
				return false;
			}
			return true;

		/** 在庫数量フォーマットチェック */
		} else 	if (AoFormatDiv.STOCK_QUANTITY.equals(aoFormatDiv.replace("_","").toLowerCase())
				|| AoFormatDiv.STOCK_QUANTITY_KN.equals(aoFormatDiv)){

			// 在庫数量整数桁
			String digit = aoSetting.getStockQuantityDigitNum();
			// 在庫数量小数桁
			String decimalDigit = aoSetting.getStockQuantityDecimalPointNum();
			// 在庫数量パターンの作成
			String matchPattern = digitDecimalRegPattern(digit, decimalDigit);
			// 在庫数量フォーマットチェック
			if (!value.matches(matchPattern)) {
				return false;
			}
			return true;

		/** 入数フォーマットチェック */
		} else 	if (AoFormatDiv.UNIT_QUANTITY.equals(aoFormatDiv.replace("_","").toLowerCase())
				|| AoFormatDiv.UNIT_QUANTITY_KN.equals(aoFormatDiv)){

			// 入数整数桁
			String digit = aoSetting.getUnitQuantityDigitNum();
			// 入数小数桁
			String decimalDigit = aoSetting.getUnitQuantityDecimalPointNum();
			// 入数パターンの作成
			String matchPattern = digitDecimalRegPattern(digit, decimalDigit);
			// 入数フォーマットチェック
			if (!value.matches(matchPattern)) {
				return false;
			}
			return true;

		/** ケース数フォーマットチェック */
		} else 	if (AoFormatDiv.CASE_QUANTITY.equals(aoFormatDiv.replace("_","").toLowerCase())
				|| AoFormatDiv.CASE_QUANTITY_KN.equals(aoFormatDiv)){

			// ケース数整数桁
			String digit = aoSetting.getCaseQuantityDigitNum();
			// ケース数小数桁
			String decimalDigit = aoSetting.getCaseQuantityDecimalPointNum();
			// ケース数パターンの作成
			String matchPattern = digitDecimalRegPattern(digit, decimalDigit);
			// ケース数フォーマットチェック
			if (!value.matches(matchPattern)) {
				return false;
			}
			return true;

		}
		return true;
	}

	public static Map<String, String> getAoRange(String aoFormatDiv,  AoSetting aoSetting) {

		Map<String, String> ret = new HashMap<String, String>();

		// AOフォーマット区分が設定されていない場合
		if (UtilTools.isEmpty(aoFormatDiv)) {
			return ret;
		}

		String digit = null;
		String decimalDigit = null;

		/** 売上数量フォーマットチェック */
		if (AoFormatDiv.SALES_QUANTITY.equals(aoFormatDiv.replace("_","").toLowerCase())
				|| AoFormatDiv.SALES_QUANTITY_KN.equals(aoFormatDiv)){

			// 売上数量整数桁
			digit = aoSetting.getSalesQuantityDigitNum();
			// 売上数量小数桁
			decimalDigit = aoSetting.getSalesQuantityDecimalPointNum();

		/** 売上単価フォーマットチェック */
		} else 	if (AoFormatDiv.SALES_UNIT_PRICE.equals(aoFormatDiv.replace("_","").toLowerCase())
				|| AoFormatDiv.SALES_UNIT_PRICE_KN.equals(aoFormatDiv)){

			// 売上単価整数桁
			digit = aoSetting.getSalesUnitPriceDigitNum();
			// 売上単価小数桁
			decimalDigit = aoSetting.getSalesUnitPriceDecimalPointNum();

		/** 売上金額フォーマットチェック */
		} else 	if (AoFormatDiv.SALES_PRICE.equals(aoFormatDiv.replace("_","").toLowerCase())
				|| AoFormatDiv.SALES_PRICE_KN.equals(aoFormatDiv)){

			// 売上金額整数桁
			digit = aoSetting.getSalesPriceDigitNum();

		/** 在庫数量フォーマットチェック */
		} else 	if (AoFormatDiv.STOCK_QUANTITY.equals(aoFormatDiv.replace("_","").toLowerCase())
				|| AoFormatDiv.STOCK_QUANTITY_KN.equals(aoFormatDiv)){

			// 在庫数量整数桁
			digit = aoSetting.getStockQuantityDigitNum();
			// 在庫数量小数桁
			decimalDigit = aoSetting.getStockQuantityDecimalPointNum();

		/** 入数フォーマットチェック */
		} else 	if (AoFormatDiv.UNIT_QUANTITY.equals(aoFormatDiv.replace("_","").toLowerCase())
				|| AoFormatDiv.UNIT_QUANTITY_KN.equals(aoFormatDiv)){

			// 入数整数桁
			digit = aoSetting.getUnitQuantityDigitNum();
			// 入数小数桁
			decimalDigit = aoSetting.getUnitQuantityDecimalPointNum();

		/** ケース数フォーマットチェック */
		} else 	if (AoFormatDiv.CASE_QUANTITY.equals(aoFormatDiv.replace("_","").toLowerCase())
				|| AoFormatDiv.CASE_QUANTITY_KN.equals(aoFormatDiv)){

			// ケース数整数桁
			digit = aoSetting.getCaseQuantityDigitNum();
			// ケース数小数桁
			decimalDigit = aoSetting.getCaseQuantityDecimalPointNum();

		}

		ret.put(KEY_DIGIT, digit);
		ret.put(KEY_DECIMAL_DIGIT, decimalDigit);

		return ret;

	}

	/***
	 *
	 * 整数桁のみ正規表現パターン文字列の作成
	 *
	 * @param digit 整数桁
	 * @return カンマ除去後の文字列
	 *
	 */
	public static String digitRegPattern(String digit) {
		return "^\\-?([0-9]{1," + digit + "})$";
	}

	/***
	 *
	 * 整数桁＋小数桁正規表現パターン文字列の作成
	 *
	 * @param digit 整数桁
	 * @param decimalDigit 小数桁
	 * @return カンマ除去後の文字列
	 *
	 */
	public static String digitDecimalRegPattern(String digit, String decimalDigit) {
		return "^\\-?(([0-9]{1," + digit + "}\\.[0-9]{0," + decimalDigit + "})|([0-9]{1," + digit + "}))$";
	}


	/** 注文数量関連 ---------------------------------------------------------------------------------------------- */
	/**
	 * 注文数量加工
	 *
	 * @param quantity 注文数量
	 * @param unitPrice 売上単価
	 * @param division 分割フラグ
	 * @param aladdinSetting アラジンシステム設定
	 * @return
	 */
	public static String makeQuantity(  String quantity,
										String unitPrice,
										AoSetting aoSetting,
										Map<String, Object> userForm) {
		// 余分な文字を除去
		quantity = removeComma(quantity);

		// 数量が設定されていない、もしくは数値ではない、もしくは数量が0以下の場合
		if (!checkMakeQuantity(quantity, unitPrice, aoSetting, userForm)) {
			// 数量を1で返却
			return "1";
		}

		return quantity;
	}

	/**
	 * 注文数量チェック<br>
	 * 注文金額を加味して注文数量に問題がないかチェックします。
	 *
	 * @param quantity 注文数量
	 * @param unitPrice 単価
	 * @param aladdinSetting アラジンシステム設定
	 * @return
	 */
	public static boolean checkMakeQuantity(String quantity,
											String unitPrice,
											AoSetting aoSetting,
											Map<String, Object> userForm) {

		// 余分な文字を除去
		quantity = removeComma(quantity);
		unitPrice = removeComma(unitPrice);

		// 数量が設定されていない、もしくは数値ではない、もしくは数量が0以下の場合
		if (StringUtils.isEmpty(quantity)) {
			return false;
		}

		// 注文数量のチェック
		if (!checkQuantityPriceFormat(quantity, unitPrice, aoSetting, userForm)) {
			return false;
		}

		// 注文数量が0の場合
		if (Double.parseDouble(quantity) == 0) {
			return false;
		}

		return true;
	}

	/**
	 *
	 * 注文数量フォーマットチェック
	 *
	 * @param quantity 検査値
	 * @param aoSetting アラジンシステム設定
	 * @return 結果(true:チェックOK / false:チェックNG)
	 *
	 */
	public static boolean checkQuantityFormat( String quantity,
												AoSetting aoSetting) {

		// 売上数量桁数チェック
		return checkQuantityFormat(quantity,
				aoSetting.getSalesQuantityDigitNum(),
				aoSetting.getSalesQuantityDecimalPointNum()
		);
	}

	/**
	 * 注文数量フォーマットチェック
	 * 注文数量のフォーマットに問題がないかどうかのみチェックします。
	 *
	 * @param quantity 注文数量
	 * @param aoSetting アラジンシステム設定
	 * @return
	 */
	public static boolean checkMakeQuantityFormat( String quantity, AoSetting aoSetting) {

		// 余分な文字を除去
		quantity = removeComma(quantity);

		// 数量が設定されていない、もしくは数値ではない、もしくは数量が0以下の場合
		if (StringUtils.isEmpty(quantity)) {
			return false;
		}

		// 注文数量のフォーマットチェック
		if (!checkQuantityFormat(quantity, aoSetting)) {
			return false;
		}

		// 注文数量が0の場合
		if (Double.parseDouble(quantity) == 0) {
			return false;
		}

		return true;
	}

	/**
	 *
	 * 注文数量、ケース数の桁数チェック
	 *
	 * @param quantity 検査値
	 * @param quantityDigitNum 整数桁
	 * @param quantityDecimalPointNum 小数点桁
	 * @return 結果(true:チェックOK / false:チェックNG)
	 *
	 */
	public static boolean checkQuantityFormat(String quantity, String quantityDigitNum, String quantityDecimalPointNum) {

		// ケース数パターンの作成
		String salesQuantityMatchPattern = "^(([0-9]{1,"
			+ quantityDigitNum + "}\\.[0-9]{0,"
			+ quantityDecimalPointNum + "})|([0-9]{1,"
			+ quantityDigitNum + "}))$";

		// ケース数桁数チェック
		if (!quantity.matches(salesQuantityMatchPattern)) {
			return false;
		}

		return true;
	}



	/** 注文金額関連 ---------------------------------------------------------------------------------------------- */
	/**
	 * 注文金額フォーマットチェック
	 *
	 * @param quantity 注文数量
	 * @param unitPrice 売上単価
	 * @param aladdinSetting アラジンシステム設定
	 * @return boolean (true:合致する false:合致しない)
	 */
	private static boolean checkQuantityPriceFormat(String quantity,
													String unitPrice,
													AoSetting aoSetting,
													Map<String, Object> userForm ) {
		// 売上数量桁数チェック
		if (!checkQuantityFormat(quantity, aoSetting)) {
			return false;
		}

		// 売上金額(売上数量 * 売上単価)が売上数量の桁数を超えていないかチェック
		// 売上金額整数桁
		String salesPriceDigitNum = aoSetting.getSalesPriceDigitNum();

		// 売上金額取得
		String price = AoUtil.convAoRoundingSalesPrice(quantity, unitPrice,
				AoUtil.DispPattern.PATTERN_NO_COMMA, salesPriceDigitNum, StringUtil.defaultString(userForm.get("price_fraction_div"), "0") );

		// 売上金額パターンの作成
		String priceMatchPattern = "^([0-9]{1," + salesPriceDigitNum + "})$";

		// 売上金額桁数チェック
		if (!price.matches(priceMatchPattern)) {
			return false;
		}

		return true;
	}

	/**
	 * 注文金額チェック
	 *
	 * @param price 売上金額
	 * @param aoSetting アラジンシステム設定
	 * @return
	 */
	public static boolean checkMakePrice(String price, AoSetting aoSetting) {
		// 余分な文字を除去
		price = StringUtils.defaultString(price);
		price = price.replace(",", "").trim();

		// 設定されていない、もしくは数値ではない、もしくは数値が0以下の場合
		if (UtilTools.isEmpty(price)) {
			return false;
		}

		// 売上のチェック
		if (!checkPriceFormat(price, aoSetting)) {
			return false;
		}

		return true;
	}

	/**
	 * 注文金額フォーマットチェック
	 *
	 * @param price 売上金額
	 * @param aoSetting アラジンシステム設定
	 * @return boolean (true:合致する false:合致しない)
	 */
	private static boolean checkPriceFormat(String price, AoSetting aoSetting) {
		// 売上金額(売上数量 * 売上単価)が売上数量の桁数を超えていないかチェック
		// 売上金額整数桁
		String salesPriceDigitNum = aoSetting.getSalesPriceDigitNum();

		// 売上金額パターンの作成
		String priceMatchPattern = "^([0-9]{1," + salesPriceDigitNum + "})$";

		// 売上金額桁数チェック
		if (!price.matches(priceMatchPattern)) {
			return false;
		}

		return true;
	}



	/** ケース数関連 ---------------------------------------------------------------------------------------------- */
	/**
	 * ケース数加工
	 *
	 * @param caseQuantity ケース数
	 * @param aoSetting アラジンシステム設定
	 * @return
	 */
	public static String makeCaseQuantity(  String caseQuantity, AoSetting aoSetting) {

		// 余分な文字を除去
		caseQuantity = removeComma(caseQuantity);

		// ケース数が設定されていない、もしくは数値ではない、もしくはケース数が0以下の場合
		if (!checkMakeCaseQuantityFormat(caseQuantity, aoSetting)) {
			// ケース数を1で返却
			return "1";
		}

		return caseQuantity;
	}

	/**
	 * ケース数フォーマットチェック
	 * ケース数のフォーマットに問題がないかどうかのみチェックします。
	 *
	 * @param caseQuantity ケース数
	 * @param aoSetting アラジンシステム設定
	 * @return boolean (true:合致する false:合致しない)
	 */
	public static boolean checkMakeCaseQuantityFormat( String caseQuantity, AoSetting aoSetting ) {

		// 余分な文字を除去
		caseQuantity = removeComma(caseQuantity);

		// ケース数が設定されていない、もしくは数値ではない、もしくは数量が0以下の場合
		if (StringUtils.isEmpty(caseQuantity)) {
			return false;
		}

		// ケース数のフォーマットチェック
		if (!checkCaseQuantityFormat(caseQuantity, aoSetting)) {
			return false;
		}

		// ケース数が0の場合
		if (Double.parseDouble(caseQuantity) == 0) {
			return false;
		}

		return true;
	}

	/**
	 *
	 * ケース数フォーマットチェック
	 *
	 * @param quantity 検査値
	 * @param aoSetting アラジンシステム設定
	 * @return 結果(true:チェックOK / false:チェックNG)
	 *
	 */
	private static boolean checkCaseQuantityFormat( String caseQuantity, AoSetting aoSetting) {

		// ケース数の桁数チェック
		return checkQuantityFormat(caseQuantity,
				aoSetting.getCaseQuantityDigitNum(),
				aoSetting.getCaseQuantityDecimalPointNum()
		);
	}

}
