package jp.ill.photon.tool;

import static org.junit.Assert.assertTrue;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonException;
import jp.ill.photon.model.SearchForm;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.db.SqlFileSelectModule;
import jp.ill.photon.util.DateUtil;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.StringUtil;

/**
 * Dome形式のSQLファイルを実行して結果をファイルに出力するツール
 * 
 * @author h_tanaka
 *
 */
public class SqlRunner {
	public static void main(String args[]) throws PhotonException {
		if (args == null || args.length < 2) {
			logger.error("Wrong argument number. exiting.");
			return;
		}

		SqlRunner runner = new SqlRunner();
		runner.execute(args[0], args[1]);

	}

	protected void execute(	String sqlFilePathString,
							String paramFileDirPathString)
			throws PhotonException {
		Path sqlFilePath = Paths.get(sqlFilePathString);
		Path paramFileDirPath = Paths.get(paramFileDirPathString);

		assertTrue("SQLファイルが存在しません", Files.exists(sqlFilePath));
		assertTrue("パラメータファイル用フォルダが存在しません", Files.exists(paramFileDirPath));

		LocalDateTime localDT = LocalDateTime.now();
		String formattedDateTime = DateTimeFormatter.ofPattern("yyyyMMddHHmmss")
				.format(localDT);

		List<Path> paramFileList = null;
		try {
			paramFileList = Files
					.list(paramFileDirPath).filter(p -> p.getFileName()
							.toString().toLowerCase().endsWith(".json"))
					.collect(Collectors.toList());
		} catch (IOException e) {
			logger.error("file list reading error", e);
			return;
		}

		if (paramFileList.isEmpty()) {
			logger.info("no param files. exiting.");
			return;
		}

		Path resultExportDirPath = Paths.get(paramFileDirPath.toString(),
				"result_" + formattedDateTime);
		if (!Files.exists(resultExportDirPath)) {
			try {
				Files.createDirectory(resultExportDirPath);
			} catch (IOException e) {
				logger.error("result folder creating error", e);
				return;
			}
		}

		ActionDto dto = new ActionDto();
		ModuleContext context = new ModuleContext(dto);
		SqlFileSelectModule module = new SqlFileSelectModule();

		try {
			String jsonString = null;
			Map<String, Object> paramMap = null;
			SearchForm searchForm = null;
			ModuleResult result = null;
			List<Map<String, Object>> resultList = null;
			for (Path paramFilePath : paramFileList) {
				jsonString = Files
						.readAllLines(paramFilePath, Charset.forName("UTF-8"))
						.stream().collect(Collectors.joining("\n"));
				paramMap = JsonUtil.jsonToMap(jsonString);
				searchForm = SearchForm.valueOf(paramMap);
				module.setSqlFilePath(sqlFilePath.toString());
				module.setSearchForm(searchForm);

				long sts = Long.parseLong(DateUtil.getYYYYMMDDHHMIMS());
				result = module.execute(context);
				long ts = (Long.parseLong(DateUtil.getYYYYMMDDHHMIMS()) - sts);
				logger.info("sql execute time:" + StringUtil.defaultString(ts));

				resultList = (List) result.getReturnData().get("list");

				try (BufferedWriter bw = getBufferedWriter(
						Paths.get(resultExportDirPath.toString(),
								paramFilePath.getFileName().toString()),
						"UTF-8")) {

					bw.write("[");
					bw.newLine();
					for (Map<String, Object> row : resultList) {
						bw.write(JsonUtil.mapToJson(row));
						bw.newLine();
					}
					bw.write("]");

					// ファイル出力完了
				} catch (IOException e2) {
					logger.error("ファイルの出力に失敗しました:" + paramFilePath.toString(),
							e2);
					return;
				}

				logger.info("結果を出力しました：" + paramFilePath.toString());
			}
		} catch (IOException e) {
			logger.error("io error", e);
			return;
		}

	}

	protected BufferedWriter getBufferedWriter(Path path, String encoding)
			throws IOException {
		BufferedWriter bw = null;
		if (encoding.toLowerCase().equals("utf-8")) {
			OutputStream fos = Files.newOutputStream(path);
			// fos.write(0xef);
			// fos.write(0xbb);
			// fos.write(0xbf);
			bw = new BufferedWriter(new OutputStreamWriter(fos, encoding));
		} else {
			bw = Files.newBufferedWriter(path, Charset.forName(encoding));
		}

		return bw;
	}

	/** ログ用変数 */
	protected static final LogUtil logger = new LogUtil(SqlRunner.class);
}
