package jp.ill.photon.service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

import jp.ill.photon.action.ActionDispatcher;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonFrameworkException;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.exception.PhotonPageNotFoundException;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.StringUtil;

/**
 * ジョブスケジュール実行サービス
 *
 * @author h_tanaka
 *
 */
public class JobService implements PhotonService {

	public static void main(String[] args) {

		JobService service = new JobService();
		service.execute();
	}

	/**
	 * 複数のジョブ実行用スレッドを起動する
	 */
	public void execute() {

		List<Map<String, Object>> tenantList = getTenantList();

		String tenantId = null;
		for (Map<String, Object> tenant : tenantList) {
			tenantId = StringUtil
					.defaultString(tenant.getOrDefault("tenant_id", "1"), "1");

			logger.info(
					String.format("[JOB_SERVICE-%s] execute start", tenantId));

			// 並行処理用にスレッドを作成して開始
			int threadNum = 1;
			ExecutorService pool = Executors.newFixedThreadPool(threadNum);
			JobThread thread = null;
			for (int i = 1; i <= threadNum; i++) {
				thread = new JobThread(i, tenantId);
				pool.submit(thread);
			}

			logger.info(
					String.format("[JOB_SERVICE-%s] execute end", tenantId));
		}
	}

	/**
	 * スレッドID別ジョブスケジュール実行スレッド
	 *
	 * @author h_tanaka
	 *
	 */
	private static class JobThread implements Runnable {

		private int threadId;

		private String tenantId;

		public JobThread(int threadId, String tenantId) {
			this.threadId = threadId;
			this.tenantId = tenantId;
		}

		@Override
		@SuppressWarnings("unchecked")
		public void run() {
			logger.info(String.format("[JOB_SERVICE-%s][thread-%d] run start",
					tenantId, threadId));

			ActionDispatcher dispatcher = ActionDispatcher.getInstance();
			ActionDto dto = null;

			// ジョブスケジュールリストを定期的に取得する
			for (;;) {
				try {
					Thread.sleep(1000 * 5);
					// 自スレッド用のジョブリストを取得する
					Map<String, Object> parameters = new HashMap<>();
					parameters.put("thread_id",
							new String[] { String.valueOf(threadId) });
					dto = dispatcher.dispatch(tenantId, "batch",
							"/job_schedule_by_thread", null, parameters, null,
							null);

					List<Map<String, Object>> jobScheduleList = (List<Map<String, Object>>) dto
							.get("job_schedule.list");

					if (jobScheduleList.size() == 0) {
						continue;
					}

					// ジョブスケジュールが存在する場合は登録されているアクションを実行する
					for (Map<String, Object> jobSchedule : jobScheduleList) {
						processJobSchedule(dispatcher, jobSchedule);
					}

				} catch (Exception e) {
					logger.error(String.format("[JOB_SERVICE-%s][thread-%d] %s",
							tenantId, threadId, e), e);
				}
			}
		}

		/**
		 * ジョブスケジュールに設定されたアクションを実行する
		 *
		 * @param dispatcher
		 * @param jobSchedule
		 * @throws PhotonFrameworkException
		 * @throws PhotonModuleException
		 */
		protected void processJobSchedule(	ActionDispatcher dispatcher,
											Map<String, Object> jobSchedule)
				throws PhotonFrameworkException, PhotonModuleException,
				PhotonPageNotFoundException {
			String scheduleId = String.valueOf(jobSchedule.get("schedule_id"));
			String jobId = String.valueOf(jobSchedule.get("job_id"));
			String appId = String.valueOf(jobSchedule.get("app_id"));
			String urlPattern = String.valueOf(jobSchedule.get("url_pattern"));
			Map<String, Object> actionParams = (Map<String, Object>) jobSchedule
					.get("job_params");

			logger.info(String.format(
					"[JOB_SERVICE-%s][thread-%d][%s] process start", tenantId,
					threadId, scheduleId));

			// dispatcherのパラメータとして渡せるように要素をString[]型に変換
			Map<String, Object> parameters = actionParams.entrySet().stream()
					.collect(Collectors.toMap(e -> e.getKey(),
							e -> new String[] {
									String.valueOf(e.getValue()) }));
			parameters.put("schedule_id", new String[] { scheduleId });

			// アクションを実行
			try {
				dispatcher.dispatch(tenantId, appId, urlPattern, null,
						parameters, null, null);
			} catch (Exception e) {
				sendErrorMail(tenantId, jobId, scheduleId, e);
				setErrorStatus(tenantId, jobId, scheduleId);
				throw e;
			}

			logger.info(
					String.format("[JOB_SERVICE-%s][thread-%d][%s] process end",
							tenantId, threadId, scheduleId));
		}

		/**
		 * システムエラーメールを送信
		 * 
		 * @param ex
		 */
		protected void sendErrorMail(	String tenantId,
										String jobId,
										String scheduleId,
										Exception ex) {

			Map<String, Object> initParameters = new HashMap<>();

			Map<String, Object> parameters = new HashMap<>();
			parameters.put("error_msg", new String[] { ex.getMessage() });
			parameters.put("job_id", new String[] { jobId });
			parameters.put("schedule_id", new String[] { scheduleId });

			ActionDispatcher dispatcher = ActionDispatcher.getInstance();
			try {
				dispatcher.dispatch(tenantId, "batch", "/send_job_error_mail",
						initParameters, parameters, null, null);
			} catch (Exception e) {
				logger.error("システムエラーメール送信エラー", e);
			}
		}

		protected void setErrorStatus(	String tenantId,
										String jobId,
										String scheduleId) {
			Map<String, Object> initParameters = new HashMap<>();

			Map<String, Object> parameters = new HashMap<>();
			parameters.put("job_id", new String[] { jobId });
			parameters.put("schedule_id", new String[] { scheduleId });
			parameters.put("job_stat", new String[] { "9" });

			ActionDispatcher dispatcher = ActionDispatcher.getInstance();
			try {
				dispatcher.dispatch(tenantId, "batch", "/update_job_status",
						initParameters, parameters, null, null);
			} catch (Exception e) {
				logger.error("エラー終了ステータス更新エラー", e);
			}

		}
	}

	/** ログ用変数 */
	protected static final LogUtil logger = new LogUtil(JobService.class);
}
