package jp.ill.photon.service;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import jp.ill.photon.action.ActionDispatcher;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonFrameworkException;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.exception.PhotonPageNotFoundException;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.StringUtil;

public class DirectoryWatchService implements PhotonService {

	public static void main(String[] args) {

		DirectoryWatchService service = new DirectoryWatchService();
		service.execute();
	}

	public void execute() {
		List<Map<String, Object>> tenantList = getTenantList();

		String tenantId = null;
		for (Map<String, Object> tenant : tenantList) {
			tenantId = StringUtil
					.defaultString(tenant.getOrDefault("tenant_id", "1"), "1");
			try {
				executeByTenantId(tenantId);
			} catch (PhotonFrameworkException e) {
				logger.error(String.format("[DIR_WATCH_SERVICE-%s] %s",
						tenantId, e));
			} catch (PhotonModuleException e) {
				logger.error(String.format("[DIR_WATCH_SERVICE-%s] %s",
						tenantId, e));
			} catch (PhotonPageNotFoundException e) {
				logger.error(String.format("[DIR_WATCH_SERVICE-%s] %s",
						tenantId, e));
			}
		}
	}

	public void executeByTenantId(String tenantId)
			throws PhotonFrameworkException, PhotonModuleException,
			PhotonPageNotFoundException {
		logger.info(String.format("[DIR_WATCH_SERVICE-%s] execute start",
				tenantId));

		ActionDispatcher dispatcher = ActionDispatcher.getInstance();

		// 監視設定リストを取得する
		Map<String, Object> parameters = new HashMap<>();
		ActionDto dto = dispatcher.dispatch(tenantId, "batch",
				"/watch_setting_list", null, parameters, null, null);
		List<Map<String, Object>> settingList = (List<Map<String, Object>>) dto
				.get("watch_setting.list");

		// 設定毎にスレッドを作成して開始
		ExecutorService pool = Executors.newCachedThreadPool();
		WatchThread thread = null;
		for (Map<String, Object> setting : settingList) {
			thread = new WatchThread(tenantId);
			thread.setWatchSetting(setting);
			pool.submit(thread);
		}

		// TODO 監視設定を定期的にチェックして設定が更新されたらスレッドの追加、削除を行う。
		logger.info(
				String.format("[DIR_WATCH_SERVICE-%s] execute end", tenantId));
	}

	private static class WatchThread implements Runnable {

		private Map<String, Object> watchSetting;

		private String tenantId;

		public static final String WORK_DIR_PATH = "work";

		public WatchThread(String tenantId) {
			this.tenantId = tenantId;
		}

		@Override
		public void run() {

			Map<String, Object> setting = getWatchSetting();
			if (setting == null) {
				return;
			}

			String settingId = String.valueOf(setting.get("setting_id"));

			String dirPath = String.valueOf(setting.get("dir_path"));
			Map<Pattern, String> jobParams = convertFilePatternJobMap(
					(Map<String, Object>) setting.get("job_params"));

			List<Pattern> filePatternList = jobParams.entrySet().stream()
					.map(e -> e.getKey())
					.sorted(Comparator.comparing(Pattern::toString).reversed())
					.collect(Collectors.toList());

			logger.info(String.format("[DIR_WATCH_SERVICE-%s][%s] run start",
					tenantId, settingId));

			File dir = new File(dirPath);
			if (!dir.exists()) {
				logger.error(String.format(
						"[DIR_WATCH_SERVICE-%s][%s] dir_path:%s not found",
						tenantId, settingId, dirPath));
				return;
			}

			// 処理中ファイル格納用フォルダを作成
			File workDir = dir.toPath().resolve(WORK_DIR_PATH).toFile();
			if (!workDir.exists() && !workDir.mkdir()) {
				logger.error(String.format(
						"[DIR_WATCH_SERVICE-%s][%s] work dir creation error: %s",
						tenantId, settingId, workDir.getAbsolutePath()));
				return;
			}

			for (;;) {
				try {
					Thread.sleep(1000 * 5);
					processFiles(filePatternList, dir, settingId, jobParams,
							workDir);
				} catch (Exception e) {
					logger.error(String.format("[DIR_WATCH_SERVICE-%s][%s] %s",
							tenantId, settingId, e), e);
				}
			}
		}

		/**
		 * 監視用フォルダで見つけたファイルの一覧を処理する
		 * 
		 * @param filePatternList
		 * @param dir
		 * @param settingId
		 * @param jobParams
		 * @param workDir
		 * @throws PhotonFrameworkException
		 * @throws PhotonModuleException
		 * @throws PhotonPageNotFoundException
		 */
		protected void processFiles(List<Pattern> filePatternList,
									File dir,
									String settingId,
									Map<Pattern, String> jobParams,
									File workDir)
				throws PhotonFrameworkException, PhotonModuleException,
				PhotonPageNotFoundException {
			ActionDispatcher dispatcher = ActionDispatcher.getInstance();
			Matcher match = null;
			File movedFile = null;
			for (File file : dir.listFiles()) {

				// ファイル以外は無視
				if (!file.isFile()) {
					continue;
				}

				for (Pattern pattern : filePatternList) {
					match = pattern.matcher(file.toString());
					if (match.find()) {
						logger.info(String.format(
								"[DIR_WATCH_SERVICE-%s][%s] file:%s", tenantId,
								settingId, file));
						if (jobParams.containsKey(pattern)) {

							try {
								// ファイルを処理用フォルダに移動
								movedFile = moveFileToWorkDir(file, workDir);

								// 移動に成功した場合のみファイルを処理
								processFile(dispatcher, movedFile,
										jobParams.get(pattern));
							} catch (IOException e) {
								logger.error(String.format(
										"[DIR_WATCH_SERVICE-%s][%s] work dir move error:%s",
										tenantId, settingId, file.getName()));
							}
						}
						break;
					}
				}
			}

		}

		/**
		 * ファイルを指定したフォルダに移動する
		 * 
		 * @param file
		 * @param newDir
		 * @return
		 * @throws IOException
		 */
		protected File moveFileToWorkDir(File file, File newDir)
				throws IOException {
			Path newPath = Files.move(file.toPath(),
					newDir.toPath().resolve(file.getName()),
					StandardCopyOption.REPLACE_EXISTING);
			return newPath.toFile();
		}

		/**
		 * ファイルパターンに設定されたジョブを登録
		 *
		 * @param dispatcher
		 * @param file
		 * @param jobId
		 * @throws PhotonFrameworkException
		 * @throws PhotonModuleException
		 */
		protected void processFile(	ActionDispatcher dispatcher,
									File file,
									String jobId)
				throws PhotonFrameworkException, PhotonModuleException,
				PhotonPageNotFoundException {
			Map<String, Object> setting = getWatchSetting();
			String settingId = String.valueOf(setting.get("setting_id"));

			Map<String, Object> parameters = new HashMap<>();
			parameters.put("file_dir",
					new String[] { file.getParentFile().getAbsolutePath() });
			parameters.put("file_path", new String[] { file.getName() });
			parameters.put("job_id", new String[] { jobId });
			parameters.put("setting_id", new String[] { settingId });

			dispatcher.dispatch(tenantId, "batch",
					"/add_csv_import_job_schedule", null, parameters, null,
					null);
		}

		/**
		 * ジョブパラメータマップを検索パターンマップに変換する
		 *
		 * @param patternJobMap
		 * @return
		 */
		protected Map<Pattern, String> convertFilePatternJobMap(Map<String, Object> patternJobMap) {
			Map<Pattern, String> newMap = new HashMap<>();

			for (Map.Entry<String, Object> entry : patternJobMap.entrySet()) {
				newMap.put(Pattern.compile(entry.getKey()),
						String.valueOf(entry.getValue()));
			}

			return newMap;
		}

		public Map<String, Object> getWatchSetting() {
			return watchSetting;
		}

		public void setWatchSetting(Map<String, Object> watchSetting) {
			this.watchSetting = watchSetting;
		}
	}

	/** ログ用変数 */
	protected static final LogUtil logger = new LogUtil(
			DirectoryWatchService.class);
}
