package jp.ill.photon.module.validation;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.lang.StringUtils;

import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleFactory;
import jp.ill.photon.module.ModuleRepository;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.ParamUtil;

public class Validator {

	private Map<String, String> paramNames;

	private Map<String, String> paramValidationSetIdMap;

	private Map<String, Object> validationSetSettings;

	private Map<String, AbstractValidationModule> moduleMap;

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(SingleValidator.class);

	public Validator(String tenantId,
			List<Map<String, Object>> columnSettingList) {

		// フィールドコードとフィールド名のマップを作成する
		Map<String, String> paramNames = columnSettingList.stream().collect(
				Collectors.toMap(s -> String.valueOf(s.get("field_cd")),
						s -> String.valueOf(s.get("field_name"))));

		// フィールドコードとバリデーションセットIDのマップを作成する
		Map<String, String> paramValidationSetIdMap = columnSettingList.stream()
				.collect(Collectors.toMap(
						s -> String.valueOf(s.get("field_cd")), s -> String
								.valueOf(s.get("validation_set_id"))));

		// 使用するバリデーションセットIDのリストを作成する
		List<String> validationSetIdList = columnSettingList.stream()
				.map(s -> String.valueOf(s.get("validation_set_id"))).distinct()
				.collect(Collectors.toList());

		ModuleRepository moduleRepository = ModuleRepository.newInstance();

		// バリデーションセットIDをキーとしてバリデーションセット設定を値として持つマップを取得する
		Map<String, Object> validationSetSettings = moduleRepository
				.getValidationSetSettings(tenantId, validationSetIdList);

		// 使用するバリデーションモジュールのモジュールIDのリストをバリデーションセット設定から抽出する
		List<String> moduleIdList = getModuleIdListFromValidationSetSettings(
				validationSetSettings);

		// バリデーションモジュール設定を取得
		Map<String, Object> moduleSettings = moduleRepository
				.getModuleSettings(moduleIdList);

		// 各バリデーションモジュールクラスをインスタンス化してモジュールIDをキーとするマップにまとめる
		Map<String, AbstractValidationModule> moduleMap = moduleSettings
				.entrySet().stream()
				.collect(Collectors.toMap(s -> s.getKey(),
						s -> (AbstractValidationModule) ModuleFactory
								.get((String) ((Map) s.getValue())
										.get("module_class"))));

		this.paramNames = paramNames;
		this.paramValidationSetIdMap = paramValidationSetIdMap;
		this.validationSetSettings = validationSetSettings;
		this.moduleMap = moduleMap;
	}

	public Map<String, List<String>> validate(	Map<String, Object> paramValueMap,
												ActionDto dto) {
		Map<String, List<String>> errorMessageListMap = new LinkedHashMap<>();

		List<String> errorMessageList = null;
		for (Map.Entry<String, Object> entry : paramValueMap.entrySet()) {
			errorMessageList = validateParam(entry.getKey(), entry.getValue(),
					(Map) this.validationSetSettings
							.get(paramValidationSetIdMap.get(entry.getKey())),
					this.moduleMap, dto);

			if (errorMessageList.size() > 0) {
				errorMessageListMap.put(entry.getKey(), errorMessageList);
			}
		}

		return errorMessageListMap;
	}

	/**
	 * 1パラメータのバリデーションを行い、エラーメッセージのリストを返す
	 *
	 * @param value パラメータ値
	 * @param validationRules
	 * @param moduleMap バリデーションモジュールオブジェクトのマップ
	 * @return
	 */
	protected List<String> validateParam(	String paramName,
											Object paramValue,
											Map<String, Object> validationSetSetting,
											Map<String, AbstractValidationModule> moduleMap,
											ActionDto dto) {
		List<String> errorMessages = new ArrayList<>();

		if (validationSetSetting == null) {
			return errorMessages;
		}

		List<Map<String, Object>> validationList = (List) ((Map) validationSetSetting
				.get("validations")).get("list");

		AbstractValidationModule module = null;
		Map<String, Object> moduleParams = null;
		ModuleResult result = null;
		for (Map<String, Object> validation : validationList) {

			logger.info("　バリデーションモジュールID：" + validation.get("module_id"));

			module = moduleMap.get(validation.get("module_id"));

			module.setParamName(paramName);
			module.setParamValue(String.valueOf(paramValue));
			module.setRuleParams(((Map) validation.get("params")));
			// module.setMessage(String.valueOf(validation.get("message")));
			module.setMessage(
					generateMessage(validation.get("message"), paramName, dto));

			// "skip_blank"は、未設定のときはfalseとして扱う
			Boolean dontCheckIfBlank = (Boolean) validation.get("skip_blank");
			if (dontCheckIfBlank == null) {
				dontCheckIfBlank = false;
			}

			// 実行可能条件を満たさない場合はバリデーションを実施しない
			Map<String, Object> ruleParams = (Map) validation.get("params");
			if(ruleParams != null && !module.isExecutable(new ModuleContext(dto), ruleParams)){
				logger.info("　　実行条件を満たしません。実行はスキップされました。");
				continue;
			}

			// "skip_blank"がtrueで、検査対象値が空の場合は、
			// そのバリデーションを行わない
			if (!dontCheckIfBlank
					|| !StringUtils.isEmpty(module.getParamValue())) {

				ModuleContext context = new ModuleContext(dto);
				result = module.execute(context);

				if (result.getResultType()
						.equals(ModuleResult.ResultTypes.ERROR)) {

					logger.info("　　バリデーションエラー");

					errorMessages.add(String
							.valueOf(result.getReturnData().get("message")));
					boolean isLast = (Boolean) validation.get("is_last");
					if (isLast) {
						break;
					}
				}

			} else {

				logger.info("　　実行はスキップされました。");
				continue;
			}
		}

		return errorMessages;
	}

	/**
	 * バリデーションセットのリストに含まれるモジュールIDのリストを取得する
	 *
	 * @param validationSetSettings
	 * @return
	 */
	protected List<String> getModuleIdListFromValidationSetSettings(Map<String, Object> validationSetSettings) {
		List<String> moduleIdListList = new ArrayList<>();
		Map<String, Object> validations = null;
		List<Object> validationList = null;
		List<String> moduleIdList = null;
		for (Map.Entry<String, Object> setting : validationSetSettings
				.entrySet()) {
			validations = (Map) ((Map) setting.getValue()).get("validations");
			validationList = (List) validations.get("list");
			moduleIdList = validationList.stream()
					.map(s -> (String) ((Map) s).get("module_id"))
					.collect(Collectors.toList());
			moduleIdListList.addAll(moduleIdList);
		}

		return moduleIdListList.stream().distinct()
				.collect(Collectors.toList());
	}

	/**
	 * メッセージを生成する<br />
	 * messageSettingがStringであれば、そのままメッセージとして返却し、<br />
	 * それ以外であれば、設定情報のオブジェクトとみなす。
	 *
	 * @param validationSetSettings
	 * @return
	 */
	@SuppressWarnings("unchecked")
	protected String generateMessage(	Object messageSetting,
										String paramName,
										ActionDto dto) {

		if (String.class.equals(messageSetting.getClass())) {
			return (String) messageSetting;
		}

		String ret = "";
		Map<String, Object> setting = (Map<String, Object>) messageSetting;
		if (setting != null && !setting.isEmpty()) {

			Map<String, Object> baseSetting = (Map<String, Object>) setting
					.get("base");
			List<Map<String, Object>> replaceSetting = (List<Map<String, Object>>) setting
					.get("replace");
			Map<String, Object> markerSetting = (Map<String, Object>) setting
					.get("marker");

			String baseStr = "";
			if (baseSetting != null && !baseSetting.isEmpty()) {
				baseStr = (String) ParamUtil.getParamValueByType(
						(String) baseSetting.get("type"),
						baseSetting.get("val"), dto);
			}

			List<String> replaceStr = new LinkedList<String>();
			if (replaceSetting != null && !replaceSetting.isEmpty()) {
				for (Map<String, Object> rs : replaceSetting) {
					// typeが"name"だった場合は、valの値は無視し、paramNameをセットする
					String type = (String) rs.get("type");
					String value = "";
					if ("name".equals(type)) {
						value = paramName;
					} else {
						value = (String) ParamUtil.getParamValueByType(type,
								rs.get("val"), dto);
					}
					replaceStr.add(value);
				}
			}

			String frontMarkerStr = "";
			String rearMarkerStr = "";
			if (markerSetting != null && !markerSetting.isEmpty()) {
				frontMarkerStr = (String) markerSetting.get("front");
				rearMarkerStr = (String) markerSetting.get("rear");
			}

			// 置換する
			ret = baseStr;
			if (!StringUtils.isEmpty(baseStr) && !replaceStr.isEmpty()
					&& !StringUtils.isEmpty(frontMarkerStr)
					&& !StringUtils.isEmpty(rearMarkerStr)) {

				for (int i = 0; i < replaceStr.size(); i++) {
					String markerStr = frontMarkerStr + Integer.toString(i + 1)
							+ rearMarkerStr;
					// ret = ret.replaceAll(markerStr, replaceStr.get(i));
					ret = ret.replace(markerStr, replaceStr.get(i));
				}

			}

		}

		return ret;

	}

}
