package jp.ill.photon.module.validation;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.annotation.ModuleVersion;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.model.MapListParam;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.ParamUtil;

/**
 * [validate] フォームバリデーション用モジュール.
 *
 * <p>
 * フォーム設定を元にバリデーションを行う。<br/>
 * 内部でFormValidateクラスを使用している。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 *
 * <h3>結果コード=validate_okの場合</h3>
 * <dl>
 * <dt>結果タイプ</dt>
 * <dd>CONTINUE</dd>
 * <dt>validate_result</dt>
 * <dd>validate_ok</dd>
 * </dl>
 * </p>
 *
 * <h3>結果コード=validate_ngの場合</h3>
 * <dl>
 * <dt>結果タイプ</dt>
 * <dd>ERROR</dd>
 * <dt>validate_result</dt>
 * <dd>validate_ng</dd>
 * </dl>
 * </p>
 *
 * @author h_tanaka
 *
 */
@ModuleVersion("1.0.0")
public class ValidateModule implements PhotonModule {

	/**
	 * バリデーション対象データ用モジュールパラメータ.
	 *
	 * <p>
	 * バリデーション対象のデータをform_field_cdをキー、チェック対象値（入力値）を値とするマップ形式で受け取る。<br/>
	 * 値に追加項目として「options」を指定すると、バリデーションモジュールに対して追加パラメータを渡すことができる。<br/>
	 * （モジュールごとに受け取り可能な追加パラメータは各バリデーション用モジュール参照）
	 * </p>
	 *
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 *
	 * <pre>
	 * {
	 *   "user_name": {
	 *     "type": "param",
	 *     "val": "user_name",
	 *     "remarks": "ユーザー入力ユーザー名"
	 *   },
	 *   "mail": {
	 *     "type": "param",
	 *     "val": "mail_address",
	 *     "remarks": "ユーザー入力メールアドレス",
	 *     "options": {
	 *       "check_table": {
	 *         "type": "static",
	 *         "val": "mail"
	 *       }
	 *     }
	 *   }
	 * }
	 * </pre>
	 *
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = true)
	private Map<String, Object> paramJson;

	/**
	 * カラム設定用モジュールパラメータ.
	 *
	 * <p>
	 * 下記のSQLが返すリストと同様の結果が渡されることを想定:<br/>
	 * aec20/table/selectColumnSettingsByFormCd.sql
	 * </p>
	 */
	@ModuleParam(required = false)
	private MapListParam columnSettings;

	/**
	 * 動的後処理設定用モジュールパラメータ.
	 *
	 * <p>
	 * モジュール実行後処理を動的に指定したい場合に設定する。<br/>
	 * 後処理を設定したい結果コード（成功時：validate_ok、失敗時：validate_ng）毎に後処理パラメータを設定する。
	 * </p>
	 *
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 *
	 * <pre>
	 * {
	 *   "validate_ng": {
	 *     "result_type": "forward",
	 *     "next_path": "/user/shohin_list",
	 *     "params": {
	 *       "redisplay_flg": {
	 *         "transfer_type": "static",
	 *         "transfer_val": "1"
	 *       }
	 *     }
	 *   }
	 * }
	 * </pre>
	 *
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = false)
	private Map<String, Object> after;

	/**
	 * 追加パラメータマップ用モジュールパラメータ.
	 *
	 * <p>
	 * モジュールパラメータparamJsonをdtoから取得する場合等に追加で別の値をバリデーション対象に渡したい場合に使用する。<br/>
	 * ここで渡された値はバリデーションモジュールに渡される同名のパラメータにマージされ、バリデーションに利用される。
	 * </p>
	 *
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 *
	 * <pre>
	 * {
	 *   "search_form": {
	 *     "val": {
	 *       "editTableCd": {
	 *         "type": "static",
	 *         "val": "item"
	 *       }
	 *     }
	 *   }
	 * }
	 * </pre>
	 *
	 * </dl>
	 *
	 * 上記の場合、バリデーションモジュールに渡されるパラメータのうち、「search_form」というキーのパラメータの<br/>
	 * 「editTableCd」が「item」という文字列で追加または上書きされる。
	 * </p>
	 */
	@ModuleParam(required = false)
	private Map<String, Object> extraParamMap;

	/**
	 * ページエラーメッセージ追加するかしないかパラメータ.
	 * 0の時に追加しない。デフォルトは1。
	 */
	@ModuleParam(required = false)
    @DefaultParamSetting(transferType = "static", transferVal = "1")
	private int addPageMesgFlg;

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(ValidateModule.class);

	// モジュール内で使用する変数
	ActionDto dto;

	@SuppressWarnings("unchecked")
	@Override
	public ModuleResult execute(ModuleContext context) {

		dto = context.getDto();

		Map<String, Object> actionModule = context.getActionModule();
		String returnPrefix = (String) actionModule.get("return_prefix");

		List<Map<String, Object>> columnSettingList = null;
		if (getColumnSettings() != null) {
			columnSettingList = getColumnSettings().getParamList();
		} else {
			columnSettingList = new ArrayList<>();
		}

		FormValidator validator = new FormValidator(dto.getTenantId(),
				returnPrefix, columnSettingList, extraParamMap);

		// Map<String, Object> dataMap = getDataMap(); // {<field_cd>: <値>,
		// <field_cd>: <値>, ... }
		Map<String, Object> messages = validator.validate(getDataMap(),
				getActionParamMap(), dto);

		ModuleResult result = new ModuleResult();
		if(addPageMesgFlg == 0) {
			Map<String, Object> errFieldMap = new HashMap<String, Object>();
			errFieldMap.put("error_field", messages.get("error_field"));
			result.setMessages(errFieldMap);
		}else {
			result.setMessages(messages);
		}
		if (messages.isEmpty()) {
			result.setResultCode("validate_ok");
			result.setResultType(ModuleResult.ResultTypes.CONTINUE);
			result.getReturnData().put("validate_result", "validate_ok");
		} else {
			result.setResultCode("validate_ng");
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.getReturnData().put("validate_result", "validate_ng");
		}

		// モジュールのパラメータ"after"に値がセットされていたとき
		if (after != null) {
			logger.info("　パラメータ[after]が設定されていたので、後処理を登録します。");
			Map<String, Object> mp = (Map<String, Object>) after
					.get(result.getResultCode());
			if (mp != null) {
				result.getReturnData().put("after_params", mp.get("params"));
				result.setNextPath((String) mp.get("next_path"));
				result.setResultType((String) mp.get("result_type"));
			}
		}

		return result;
	}

	/***
	 *
	 * パラメータから、フィールドコードと値のペアを作成
	 *
	 * @param dto
	 *
	 * @return
	 *
	 */
	protected Map<String, Object> getDataMap() {

		// パラメータを整理
		Map<String, Object> paramValues = new LinkedHashMap<String, Object>();

		if (paramJson != null) {

			for (Map.Entry<String, Object> param : paramJson.entrySet()) {
				paramValues.put(param.getKey(), ParamUtil
						.getParamObjectValueByType(param.getValue(), dto));
			}

		}

		return paramValues;

	}

	/**
	 * アクション実行時にアクションからバリデーションモジュールに渡すオプションパラメータデータを整備
	 */
	protected Map<String, Map<String, Object>> getActionParamMap() {

		// パラメータを整理
		Map<String, Map<String, Object>> paramOptions = new HashMap<>();

		if (paramJson != null) {

			Object paramValue = null;
			Map<String, Object> paramMap = null;
			Map<String, Object> optionMap = null;
			Map<String, Object> optionValues = null;
			for (Map.Entry<String, Object> param : paramJson.entrySet()) {
				paramValue = param.getValue();
				if (paramValue != null) {
					paramMap = (Map) paramValue;
					if (paramMap.containsKey("options")) {
						optionMap = (Map) paramMap.get("options");
						optionValues = new HashMap<>();
						for (Map.Entry<String, Object> option : optionMap
								.entrySet()) {
							optionValues.put(option.getKey(),
									ParamUtil.getParamObjectValueByType(
											option.getValue(), dto));
						}
						paramOptions.put(param.getKey(), optionValues);
					}
				}
			}

		}

		return paramOptions;

	}

	/**
	 * paramJsonを取得します。
	 *
	 * @return paramJson
	 */
	public Map<String, Object> getParamJson() {
		return paramJson;
	}

	/**
	 * paramJsonを設定します。
	 *
	 * @param paramJson
	 */
	public void setParamJson(Map<String, Object> paramJson) {
		this.paramJson = paramJson;
	}

	/**
	 * addPageMesgFlgを取得します。
	 *
	 * @return addPageMesgFlg
	 */
	public int getAddPageMesgFlg() {
		return addPageMesgFlg;
	}

	/**
	 * addPageMesgFlgを設定します。
	 *
	 * @param addPageMesgFlg
	 */
	public void setAddPageMesgFlg(int addPageMesgFlg) {
		this.addPageMesgFlg = addPageMesgFlg;
	}

	/**
	 * columnSettingsを取得します。
	 *
	 * @return columnSettings
	 */
	public MapListParam getColumnSettings() {
		return columnSettings;
	}

	/**
	 * columnSettingsを設定します。
	 *
	 * @param columnSettings
	 */
	public void setColumnSettings(MapListParam columnSettings) {
		this.columnSettings = columnSettings;
	}

	/**
	 * afterを取得します。
	 *
	 * @return after
	 */
	public Map<String, Object> getAfter() {
		return after;
	}

	/**
	 * afterを設定します。
	 *
	 * @param after
	 */
	public void setAfter(Map<String, Object> after) {
		this.after = after;
	}

	/**
	 * extraParamMapを取得します。
	 *
	 * @return extraParamMap
	 */
	public Map<String, Object> getExtraParamMap() {
		return extraParamMap;
	}

	/**
	 * extraParamMapを設定します。
	 *
	 * @param extraParamMap
	 */
	public void setExtraParamMap(Map<String, Object> extraParamMap) {
		this.extraParamMap = extraParamMap;
	}

}
