package jp.ill.photon.module.validation;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleFactory;
import jp.ill.photon.module.ModuleRepository;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.ParamUtil;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;

public class SingleValidator {

	private String returnPrefix;

	private Map<String, Object> validationSet;

	private Map<String, AbstractValidationModule> moduleMap;

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(SingleValidator.class);

	/** エラーメッセージタイプ */
	public final static class ErrorMessageType {
		public final static String INFO_PAGE = "info_page";
		public final static String INFO_FIELD = "info_field";
		public final static String ERROR_PAGE = "error_page";
		public final static String ERROR_FIELD = "error_field";
	}

	@SuppressWarnings("rawtypes")
	public SingleValidator(String tenantId, String validationSetId,
			String returnPrefix) {

		ModuleRepository moduleRepository = ModuleRepository.newInstance();

		// バリデーションセットIDをキーとしてバリデーションセット設定を値として持つマップを取得する
		List<String> validationSetIdList = new ArrayList<>();
		validationSetIdList.add(validationSetId);
		Map<String, Object> validationSetSettings = moduleRepository
				.getValidationSetSettings(tenantId, validationSetIdList);

		// 使用するバリデーションモジュールのモジュールIDのリストをバリデーションセット設定から抽出する
		List<String> moduleIdList = getModuleIdListFromValidationSetSettings(
				validationSetSettings);

		// バリデーションモジュール設定を取得
		Map<String, Object> moduleSettings = moduleRepository
				.getModuleSettings(moduleIdList);

		// 各バリデーションモジュールクラスをインスタンス化してモジュールIDをキーとするマップにまとめる
		Map<String, AbstractValidationModule> moduleMap = moduleSettings
				.entrySet().stream()
				.collect(Collectors.toMap(s -> s.getKey(),
						s -> (AbstractValidationModule) ModuleFactory
								.get((String) ((Map) s.getValue())
										.get("module_class"))));

		this.returnPrefix = returnPrefix;
		this.validationSet = (Map<String, Object>) validationSetSettings
				.getOrDefault(validationSetId, null);
		this.moduleMap = moduleMap;
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	public Map<String, Object> validate(String fieldCd,
										String paramName,
										Object paramValue,
										ActionDto dto) {

		return validateParam(fieldCd, paramName, paramValue, dto);
	}

	/**
	 * 1パラメータのバリデーションを行い、エラーメッセージのリストを返す
	 *
	 * @param value パラメータ値
	 * @param validationRules
	 * @param moduleMap バリデーションモジュールオブジェクトのマップ
	 * @return
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	protected Map<String, Object> validateParam(String formFieldCd,
												String paramName,
												Object paramValue,
												ActionDto dto) {

		Map<String, Object> errorMessageListMap = new LinkedHashMap<>();

		if (validationSet == null) {
			return errorMessageListMap;
		}

		List<Map<String, Object>> validationList = (List) ((Map) validationSet
				.get("validations")).get("list");

		if (validationList == null) {
			return errorMessageListMap;
		}

		List<ActionDtoMessage> messageList = new ArrayList<>();
		AbstractValidationModule module = null;
		ModuleResult result = null;
		for (Map<String, Object> validation : validationList) {

			logger.info("　バリデーションモジュールID：" + validation.get("module_id"));

			module = moduleMap.get(validation.get("module_id"));

			module.setParamName(paramName);
			module.setParamValue((String) paramValue);
			module.setRuleParams(((Map) validation.get("params")));

			// "skip_blank"は、未設定のときはfalseとして扱う
			Boolean dontCheckIfBlank = (Boolean) validation.get("skip_blank");
			if (dontCheckIfBlank == null) {
				dontCheckIfBlank = false;
			}

			// 実行可能条件を満たさない場合はバリデーションを実施しない
			Map<String, Object> ruleParams = (Map) validation.get("params");
			if(ruleParams != null && !module.isExecutable(new ModuleContext(dto), ruleParams)){
				logger.info("　　実行条件を満たしません。実行はスキップされました。");
				continue;
			}

			// "skip_blank"がtrueで、検査対象値が空の場合は、
			// そのバリデーションを行わない
			if (!dontCheckIfBlank
					|| !StringUtils.isEmpty(module.getParamValue())) {

				ModuleContext context = new ModuleContext(dto);
				result = module.execute(context);

				if (result.getResultType()
						.equals(ModuleResult.ResultTypes.ERROR)) {

					logger.info("　　バリデーションエラー");

					// エラーメッセージ生成
					messageList.add(generateMessageObj(formFieldCd, paramName,
							validation.get("message"), dto));

					boolean isLast = (Boolean) validation.get("is_last");
					if (isLast) {
						break;
					}
				}
			} else {

				logger.info("　　実行はスキップされました。");
				continue;
			}
		}

		if (messageList.size() > 0) {
			errorMessageListMap.put(ErrorMessageType.ERROR_FIELD, messageList);
		}

		return errorMessageListMap;
	}

	/**
	 * バリデーションセットのリストに含まれるモジュールIDのリストを取得する
	 *
	 * @param validationSetSettings
	 * @return
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	protected List<String> getModuleIdListFromValidationSetSettings(Map<String, Object> validationSetSettings) {
		List<String> moduleIdListList = new ArrayList<>();
		Map<String, Object> validations = null;
		List<Object> validationList = null;
		List<String> moduleIdList = null;
		for (Map.Entry<String, Object> setting : validationSetSettings
				.entrySet()) {
			validations = (Map) ((Map) setting.getValue()).get("validations");
			if (validations != null) {
				validationList = (List) validations.get("list");
				if (validationList != null) {
					moduleIdList = validationList.stream()
							.map(s -> (String) ((Map) s).get("module_id"))
							.collect(Collectors.toList());
					moduleIdListList.addAll(moduleIdList);
				}
			}
		}

		return moduleIdListList.stream().distinct()
				.collect(Collectors.toList());
	}

	/***
	 *
	 * メッセージオブジェクトを生成する<br />
	 *
	 * @param formFieldCd
	 * @param formFieldName
	 * @param messageSetting
	 * @param dto
	 *
	 * @return
	 *
	 */
	@SuppressWarnings("unchecked")
	private ActionDtoMessage generateMessageObj(String formFieldCd,
												String formFieldName,
												Object messageSetting,
												ActionDto dto) {

		ActionDtoMessage ret = new ActionDtoMessage();

		ret.setReturnPrefix(returnPrefix);
		ret.setFormFieldCd(formFieldCd);

		Map<String, Object> setting = (Map<String, Object>) messageSetting;
		if (!MapUtils.isEmpty(setting)) {

			// メッセージID取得
			ret.setMessageId(
					ParamUtil.getParamStrValueByType(setting.get("id"), dto));
			// メッセージパラメータ取得
			Map<String, Object> setParams = new HashMap<String, Object>();
			Map<String, Object> params = (Map<String, Object>) ParamUtil
					.getParamObjectValueByType(setting.get("param"), dto);

			if (params != null) {
				for (Map.Entry<String, Object> param : params.entrySet()) {

					String type = (String) ((Map<String, Object>) param
							.getValue()).get("type");
					if ("name".equals(type)) {
						Map<String, Object> setParamItem = new HashMap<String, Object>() {
							{
								put("type", "static");
								put("val", formFieldName);
							}
						};
						setParams.put(param.getKey(), setParamItem);
					} else {
						setParams.put(param.getKey(), param.getValue());
					}

				}
			}
			ret.setParams(setParams);

		}

		return ret;
	}
}
