package jp.ill.photon.module.validation;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.model.MapListParam;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;

/**
 * フォームのnameとvalのマップのリストを一括バリデーションチェックする
 *
 * @author h_tanaka
 *
 */
public class MultiValidateModule implements PhotonModule {

	@ModuleParam(required = false)
	private MapListParam formList;

	@ModuleParam(required = false)
	private String formKey;

	@ModuleParam(required = false)
	private String indexKey;

	@ModuleParam(required = false)
	private MapListParam columnSettings;

	@ModuleParam(required = false)
	private Map<String, Object> after;

	@ModuleParam(required = false)
	private Map<String, Object> extraParamMap;

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(MultiValidateModule.class);

	// モジュール内で使用する変数
	ActionDto dto;

	@SuppressWarnings("unchecked")
	@Override
	public ModuleResult execute(ModuleContext context) {

		dto = context.getDto();

		Map<String, Object> actionModule = context.getActionModule();
		String returnPrefix = (String) actionModule.get("return_prefix");

		List<Map<String, Object>> columnSettingList = null;
		if (getColumnSettings() != null) {
			columnSettingList = getColumnSettings().getParamList();
		} else {
			columnSettingList = new ArrayList<>();
		}

		List<Map<String, Object>> formList = null;
		if (getFormList() != null) {
			formList = getFormList().getParamList();
		} else {
			formList = new ArrayList<>();
		}

		FormValidator validator = null;
		Map<String, Object> messages = new LinkedHashMap<>();
		Map<String, Object> tmpMessages = null;
		List<ActionDtoMessage> messageList = null;
		List<ActionDtoMessage> mergedMessageList = null;
		String formFieldCd = null;
		for (Map<String, Object> form : formList) {
			validator = new FormValidator(dto.getTenantId(), returnPrefix,
					columnSettingList, extraParamMap);
			tmpMessages = validator.validate(form, dto);

			for (Map.Entry<String, Object> entry : tmpMessages.entrySet()) {
				messageList = (List<ActionDtoMessage>) entry.getValue();

				if (!messages.containsKey(entry.getKey())) {
					messages.put(entry.getKey(),
							new ArrayList<ActionDtoMessage>());
				}
				mergedMessageList = (List<ActionDtoMessage>) messages
						.get(entry.getKey());

				for (ActionDtoMessage message : messageList) {
					formFieldCd = message.getFormFieldCd();
					message.setFormFieldCd(
							form.getOrDefault(getFormKey(), "default") + "."
									+ formFieldCd);
					if (!StringUtils.isEmpty(getIndexKey())) {
						message.setFormFieldCd(formFieldCd + "["
								+ form.get(getIndexKey()) + "]");
					}
					mergedMessageList.add(message);
				}
			}
		}

		ModuleResult result = new ModuleResult();
		result.setMessages(messages);
		if (messages.isEmpty()) {
			result.setResultCode("validate_ok");
			result.setResultType(ModuleResult.ResultTypes.CONTINUE);
			result.getReturnData().put("validate_result", "validate_ok");
		} else {
			result.setResultCode("validate_ng");
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.getReturnData().put("validate_result", "validate_ng");
		}

		// モジュールのパラメータ"after"に値がセットされていたとき
		if (after != null) {
			logger.info("　パラメータ[after]が設定されていたので、後処理を登録します。");
			Map<String, Object> mp = (Map<String, Object>) after
					.get(result.getResultCode());
			if (mp != null) {
				result.getReturnData().put("after_params", mp.get("params"));
				result.setNextPath((String) mp.get("next_path"));
				result.setResultType((String) mp.get("result_type"));
			}
		}

		return result;
	}

	/**
	 * columnSettingsを取得します。
	 *
	 * @return columnSettings
	 */
	public MapListParam getColumnSettings() {
		return columnSettings;
	}

	/**
	 * columnSettingsを設定します。
	 *
	 * @param columnSettings
	 */
	public void setColumnSettings(MapListParam columnSettings) {
		this.columnSettings = columnSettings;
	}

	/**
	 * afterを取得します。
	 *
	 * @return after
	 */
	public Map<String, Object> getAfter() {
		return after;
	}

	/**
	 * afterを設定します。
	 *
	 * @param after
	 */
	public void setAfter(Map<String, Object> after) {
		this.after = after;
	}

	public MapListParam getFormList() {
		return formList;
	}

	public void setFormList(MapListParam formList) {
		this.formList = formList;
	}

	public String getFormKey() {
		return formKey;
	}

	public void setFormKey(String formKey) {
		this.formKey = formKey;
	}

	public String getIndexKey() {
		return indexKey;
	}

	public void setIndexKey(String indexKey) {
		this.indexKey = indexKey;
	}

	public Map<String, Object> getExtraParamMap() {
		return extraParamMap;
	}

	public void setExtraParamMap(Map<String, Object> extraParamMap) {
		this.extraParamMap = extraParamMap;
	}

}
