package jp.ill.photon.module.validation;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;

import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleFactory;
import jp.ill.photon.module.ModuleRepository;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.util.CheckUtil;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.MapUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringUtil;

public class FormValidator {

	private Map<String, String> paramNames;

	private Map<String, String> paramValidationSetIdMap;

	private Map<String, Object> validationSetSettings;

	private Map<String, AbstractValidationModule> moduleMap;

	private String returnPrefix;

	private Map<String, Object> errorMessageListMap;

	private Map<String, Object> valueMap;

	private Map<String, Object> extraParamMap;

	private Map<String, Map<String, Object>> resultCacheMap;

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(FormValidator.class);

	/** エラーメッセージタイプ */
	public final static class ErrorMessageType {
		public final static String INFO_PAGE = "info_page";
		public final static String INFO_FIELD = "info_field";
		public final static String ERROR_PAGE = "error_page";
		public final static String ERROR_FIELD = "error_field";
	}

	public FormValidator(String tenantId, String returnPrefix,
			List<Map<String, Object>> columnSettingList) {
		this(tenantId, returnPrefix, columnSettingList, null);
	}

	@SuppressWarnings("rawtypes")
	public FormValidator(String tenantId, String returnPrefix,
			List<Map<String, Object>> columnSettingList,
			Map<String, Object> extraParam) {

		// フィールドコードとフィールド名のマップを作成する
		Map<String, String> paramNames = columnSettingList.stream()
				.collect(Collectors.toMap(
						s -> StringUtil.defaultString(s.get("field_cd"), ""),
						s -> StringUtil.defaultString(s.get("field_name"),
								"")));

		// フィールドコードとバリデーションセットIDのマップを作成する
		Map<String, String> paramValidationSetIdMap = columnSettingList.stream()
				.collect(Collectors.toMap(
						s -> StringUtil.defaultString(s.get("field_cd"), ""),
						s -> StringUtil.defaultString(
								s.get("validation_set_id"), "")));

		// 使用するバリデーションセットIDのリストを作成する
		List<String> validationSetIdList = columnSettingList.stream().map(
				s -> StringUtil.defaultString(s.get("validation_set_id"), ""))
				.distinct().collect(Collectors.toList());

		ModuleRepository moduleRepository = ModuleRepository.newInstance();

		// バリデーションセットIDをキーとしてバリデーションセット設定を値として持つマップを取得する
		Map<String, Object> validationSetSettings = moduleRepository
				.getValidationSetSettings(tenantId, validationSetIdList);

		// 使用するバリデーションモジュールのモジュールIDのリストをバリデーションセット設定から抽出する
		List<String> moduleIdList = getModuleIdListFromValidationSetSettings(
				validationSetSettings);

		// バリデーションモジュール設定を取得
		Map<String, Object> moduleSettings = moduleRepository
				.getModuleSettings(moduleIdList);

		// 各バリデーションモジュールクラスをインスタンス化してモジュールIDをキーとするマップにまとめる
		Map<String, AbstractValidationModule> moduleMap = moduleSettings
				.entrySet().stream()
				.collect(Collectors.toMap(s -> s.getKey(),
						s -> (AbstractValidationModule) ModuleFactory
								.get((String) ((Map) s.getValue())
										.get("module_class"))));

		this.paramNames = paramNames;
		this.paramValidationSetIdMap = paramValidationSetIdMap;
		this.validationSetSettings = validationSetSettings;
		this.moduleMap = moduleMap;
		this.returnPrefix = returnPrefix;
		this.errorMessageListMap = new LinkedHashMap<>();
		this.valueMap = new HashMap<>();
		this.extraParamMap = extraParam;
		this.resultCacheMap = new HashMap<>();
	}

	public Map<String, Object> validate(Map<String, Object> paramValueMap,
										ActionDto dto) {
		return validate(paramValueMap,
				new HashMap<String, Map<String, Object>>(), dto, false, false);
	}

	public Map<String, Object> validate(Map<String, Object> paramValueMap,
										Map<String, Map<String, Object>> actionParamMap,
										ActionDto dto) {
		return validate(paramValueMap, actionParamMap, dto, true, false);
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	public Map<String, Object> validate(Map<String, Object> paramValueMap,
										Map<String, Map<String, Object>> actionParamMap,
										ActionDto dto,
										boolean withPageMessage,
										boolean isResultCacheable) {

		this.valueMap = paramValueMap;

		if (actionParamMap == null) {
			actionParamMap = new HashMap<>();
		}

		boolean validateResult = true;
		for (Map.Entry<String, Object> entry : paramValueMap.entrySet()) {

			String fieldCd = entry.getKey();
			String fieldName = (paramNames != null
					&& paramNames.containsKey(fieldCd))
							? paramNames.get(fieldCd) : fieldCd;

			if (!validateParam(fieldCd, fieldName, entry.getValue(),
					(Map) this.validationSetSettings
							.get(paramValidationSetIdMap.get(entry.getKey())),
					actionParamMap.getOrDefault(entry.getKey(),
							new HashMap<String, Object>()),
					this.moduleMap, dto, isResultCacheable)) {
				validateResult = false;
			}
		}

		if (!validateResult && withPageMessage) {
			// エラーが存在したらページエラーメッセージを追加
			ActionDtoMessage pageError = new ActionDtoMessage();
			pageError.setMessageId("commonValidateError");
			pageError.setParams(new HashMap<String, Object>());
			addErrMsgListMap(ErrorMessageType.ERROR_PAGE, pageError, 0);
		}

		return errorMessageListMap;

	}

	/**
	 * 1パラメータのバリデーションを行い、エラーメッセージのリストを返す
	 *
	 * @param value パラメータ値
	 * @param validationRules
	 * @param moduleMap バリデーションモジュールオブジェクトのマップ
	 * @return
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	protected boolean validateParam(String formFieldCd,
									String paramName,
									Object paramValue,
									Map<String, Object> validationSetSetting,
									Map<String, Object> actionParams,
									Map<String, AbstractValidationModule> moduleMap,
									ActionDto dto,
									boolean isResultCacheable) {

		boolean validateResult = true; // エラーなしからスタート
		if (validationSetSetting == null) {
			return validateResult;
		}

		List<Map<String, Object>> validationList = (List) ((Map) validationSetSetting
				.get("validations")).get("list");

		if (validationList == null) {
			return validateResult;
		}

		logger.info(String.format(
				"チェック対象：formFieldCd：%s、paramName：%s、paramValue：%s", formFieldCd,
				paramName, paramValue));

		AbstractValidationModule module = null;
		ModuleResult result = null;
		for (Map<String, Object> validation : validationList) {

			logger.info("　バリデーションモジュールID：" + validation.get("module_id"));

			module = moduleMap.get(validation.get("module_id"));

			module.setCacheable(isResultCacheable);
			Map<String, Object> moduleResultCacheMap = null;
			if (isResultCacheable) {
				moduleResultCacheMap = this.resultCacheMap.getOrDefault(
						validation.get("module_id"), new HashMap<>());
				if (moduleResultCacheMap.size() >= 100) {
					moduleResultCacheMap.clear();
				}
			}
			module.setResultCacheMap(moduleResultCacheMap);

			module.setParamMap(this.valueMap);
			module.setParamName(paramName);
			if (paramValue == null || (paramValue != null
					&& ((paramValue instanceof String)))) {
				module.setParamValue((String) paramValue);
				module.setParamRawValue(null);
			} else {
				module.setParamRawValue(paramValue);
				module.setParamValue(null);
			}
			boolean isBlankValue = CheckUtil.isBlankObject(paramValue);
			module.setActionParams(actionParams);

			Map<String, Object> params = ((Map) validation.get("params"));
			if (this.extraParamMap != null) {
				params = MapUtil.mergeMapHierarchically(params,
						this.extraParamMap);
			}
			module.setRuleParams(params);

			// "skip_blank"は、未設定のときはfalseとして扱う
			Boolean dontCheckIfBlank = (Boolean) validation.get("skip_blank");
			if (dontCheckIfBlank == null) {
				dontCheckIfBlank = false;
			}

			// 実行可能条件を満たさない場合はバリデーションを実施しない
			Map<String, Object> ruleParams = convertRuleParams(formFieldCd,
					paramName, params, dto);
			if (ruleParams != null && !module
					.isExecutable(new ModuleContext(dto), ruleParams)) {
				logger.info("　　実行条件を満たしません。実行はスキップされました。");
				continue;
			}

			// "skip_blank"がtrueで、検査対象値が空の場合は、
			// そのバリデーションを行わない
			if (!dontCheckIfBlank || !isBlankValue) {

				ModuleContext context = new ModuleContext(dto);
				result = module.execute(context);

				if (isResultCacheable) {
					moduleResultCacheMap = module.getResultCacheMap();
					this.resultCacheMap.put(
							StringUtil.defaultString(
									validation.get("module_id"), ""),
							moduleResultCacheMap);
				}

				if (result.getResultType()
						.equals(ModuleResult.ResultTypes.ERROR)) {

					validateResult = false;
					logger.info("　　バリデーションエラー");

					Map<String, Object> validationResults = (Map<String, Object>) result
							.getReturnData()
							.getOrDefault("validate_results", new HashMap<>());

					// エラーメッセージセット
					generateMessageObj(formFieldCd, paramName,
							validation.get("message"), validationResults, dto);

					boolean isLast = (Boolean) validation.get("is_last");
					if (isLast) {
						break;
					}
				}

			} else {

				logger.info("　　実行はスキップされました。");
				continue;
			}

		}

		return validateResult;
	}

	/**
	 * バリデーションセットのリストに含まれるモジュールIDのリストを取得する
	 *
	 * @param validationSetSettings
	 * @return
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	protected List<String> getModuleIdListFromValidationSetSettings(Map<String, Object> validationSetSettings) {
		List<String> moduleIdListList = new ArrayList<>();
		Map<String, Object> validations = null;
		List<Object> validationList = null;
		List<String> moduleIdList = null;
		for (Map.Entry<String, Object> setting : validationSetSettings
				.entrySet()) {
			validations = (Map) ((Map) setting.getValue()).get("validations");
			if (validations != null) {
				validationList = (List) validations.get("list");
				if (validationList != null) {
					moduleIdList = validationList.stream()
							.map(s -> (String) ((Map) s).get("module_id"))
							.collect(Collectors.toList());
					moduleIdListList.addAll(moduleIdList);
				}
			}
		}

		return moduleIdListList.stream().distinct()
				.collect(Collectors.toList());
	}

	/**
	 * パラメータを変換
	 *
	 * @param formFieldCd
	 * @param paramName
	 * @param ruleParams
	 * @param dto
	 * @return
	 */
	protected Map<String, Object> convertRuleParams(String formFieldCd,
													String paramName,
													Map<String, Object> ruleParams,
													ActionDto dto) {
		if (ruleParams == null) {
			return null;
		}

		String condFieldName = "*check_cond_layout*";
		String targetFieldName = "*check_target_values*";

		Map<String, Object> convertedRuleParams = new HashMap<>();
		if (ruleParams.containsKey(condFieldName)) {
			Map<String, Object> targetMap = (Map<String, Object>) ruleParams
					.get(targetFieldName);
			Map<String, Object> valMap = (Map<String, Object>) targetMap
					.getOrDefault("transfer_val", new HashMap<>());
			Map<String, Object> newMap = new HashMap<>();
			for (Map.Entry<String, Object> entry : valMap.entrySet()) {
				newMap.put(entry.getKey(), convertRuleParam(entry.getKey(),
						(Map) entry.getValue(), dto));
			}

			convertedRuleParams.put(condFieldName,
					ruleParams.get(condFieldName));

			Map<String, Object> convertedTargetValueMap = new HashMap<>();
			convertedTargetValueMap.put("transfer_type",
					targetMap.getOrDefault("transfer_type", "static_json"));
			convertedTargetValueMap.put("transfer_val", newMap);
			convertedRuleParams.put(targetFieldName, convertedTargetValueMap);

		}

		return convertedRuleParams;

	}

	protected Map<String, Object> convertRuleParam(	String paramName,
													Map<String, Object> param,
													ActionDto dto) {
		String type = (String) ((Map<String, Object>) param).get("type");

		String value = (String) ((Map<String, Object>) param).get("val");

		if (type == null || value == null) {
			return param;
		}

		Map<String, Object> setParamItem = new HashMap<>();
		for (Map.Entry<String, Object> entry : param.entrySet()) {
			setParamItem.put(entry.getKey(), entry.getValue());
		}

		String returnVal = null;
		switch (type) {
		case "name":
			setParamItem.put("type", "static");
			setParamItem.put("val", paramName);
			break;
		case "form":
			returnVal = (String) valueMap.get(value);
			setParamItem.put("type", "static");
			setParamItem.put("val", returnVal);
			break;
		default:
			setParamItem.put("type", type);
			setParamItem.put("val", value);
		}

		return setParamItem;
	}

	/***
	 *
	 * メッセージオブジェクトを生成する<br />
	 *
	 * @param formFieldCd
	 * @param formFieldName
	 * @param messageSetting
	 * @param dto
	 *
	 * @return
	 *
	 */
	@SuppressWarnings("unchecked")
	private void generateMessageObj(String formFieldCd,
									String formFieldName,
									Object messageSetting,
									Map<String, Object> validationResults,
									ActionDto dto) {

		ActionDtoMessage ret = new ActionDtoMessage();

		ret.setReturnPrefix(returnPrefix);
		ret.setFormFieldCd(formFieldCd);
		String errorMessageType = ErrorMessageType.ERROR_FIELD;

		Map<String, Object> setting = (Map<String, Object>) messageSetting;
		if (!MapUtils.isEmpty(setting)) {

			// メッセージID取得
			ret.setMessageId(
					ParamUtil.getParamStrValueByType(setting.get("id"), dto));
			// メッセージパラメータ取得
			Map<String, Object> setParams = new HashMap<String, Object>();
			Map<String, Object> params = (Map<String, Object>) ParamUtil
					.getParamObjectValueByType(setting.get("param"), dto);

			if (!MapUtils.isEmpty(
					(Map<String, Object>) setting.get("message_type"))) {
				String val = ParamUtil.getParamStrValueByType(
						(Map<String, Object>) setting.get("message_type"), dto);
				if (!StringUtils.isEmpty(val)) {
					errorMessageType = val;
				}
			}

			if (params != null) {
				for (Map.Entry<String, Object> param : params.entrySet()) {

					String type = (String) ((Map<String, Object>) param
							.getValue()).get("type");

					String value = (String) ((Map<String, Object>) param
							.getValue()).get("val");

					Map<String, Object> setParamItem = new HashMap<>();
					String returnVal = null;
					switch (type) {
					case "name":
						setParamItem.put("type", "static");
						setParamItem.put("val", formFieldName);
						setParams.put(param.getKey(), setParamItem);
						break;
					case "form":
						returnVal = (String) valueMap.get(value);
						setParamItem.put("type", "static");
						setParamItem.put("val", returnVal);
						setParams.put(param.getKey(), setParamItem);
						break;
					case "result":
						returnVal = StringUtil.defaultString(
								validationResults.getOrDefault(value, ""), "");
						setParamItem.put("type", "static");
						setParamItem.put("val", returnVal);
						setParams.put(param.getKey(), setParamItem);
						break;
					default:
						setParams.put(param.getKey(), param.getValue());
					}

				}
			}
			ret.setParams(setParams);

		}

		// エラーメッセージセット
		addErrMsgListMap(errorMessageType, ret);

	}

	/***
	 *
	 * エラーメッセージを指定したタイプの配列にセット
	 *
	 * @param type
	 * @param obj
	 *
	 */
	@SuppressWarnings("unchecked")
	private void addErrMsgListMap(	String type,
									ActionDtoMessage obj,
									int index) {

		if (StringUtils.isEmpty(type)) {
			return;
		}

		List<ActionDtoMessage> list = (List<ActionDtoMessage>) errorMessageListMap
				.get(type);
		if (list == null) {
			list = new ArrayList<ActionDtoMessage>();
		}

		if (index >= 0) {
			list.add(index, obj);
		} else {
			list.add(obj);
		}

		errorMessageListMap.put(type, list);

	}

	private void addErrMsgListMap(String type, ActionDtoMessage obj) {
		addErrMsgListMap(type, obj, -1);
	}

	public void clearMessages() {
		this.errorMessageListMap = new LinkedHashMap<>();
	}

	public void setExtraParamMap(Map<String, Object> extraParamMap) {
		this.extraParamMap = extraParamMap;
	}
}
