package jp.ill.photon.module.validation;

import java.io.File;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.ParamUtil;

public class FileValidationModule extends AbstractValidationModule {

	/*
	 * 本モジュールにおけるparamValueは JSON文字列としてわたってくることを想定 レイアウトは下記 { "file_name": {
	 * "type": "param_file", "val": [フィールドコード] + ".file_name" },
	 * "absolute_path": { "type": "param_file", "val": [フィールドコード] +
	 * ".absolute_path" } }
	 */

	/* サイズ単位 */
	protected static class SizeUnit {
		public static final String BYTE = "B";
		public static final String KILO_BYTE = "KB";
		public static final String MEGA_BYTE = "MB";
	}

	@SuppressWarnings("unchecked")
	public boolean validate(String paramName,
							Map<String, Object> ruleParams,
							String paramValue,
							ModuleContext context) {

		ActionDto dto = context.getDto();

		// JSON文字列で渡ってきた値をマップ化
		Map<String, Object> valMap = JsonUtil.jsonToMap(paramValue);
		String fileName = "";
		String filePath = "";
		if (valMap != null) {
			fileName = (String) valMap.getOrDefault("file_name", "");
			filePath = (String) valMap.getOrDefault("absolute_path", "");
		}

		// 実行チェック種別を取得
		// サイズチェック
		Map<String, Object> paramExecSizeCheck = (Map<String, Object>) ruleParams
				.get("exec_size_check");
		String valueExecSizeCheck = StringUtils.defaultString(
				ParamUtil.getParamStrValueByType(paramExecSizeCheck, dto),
				"true");
		boolean execSizeCheck = Boolean.parseBoolean(valueExecSizeCheck);

		// 拡張子チェック
		Map<String, Object> paramExecExtensionCheck = (Map<String, Object>) ruleParams
				.get("exec_extension_check");
		String valueExecExtensionCheck = StringUtils.defaultString(
				ParamUtil.getParamStrValueByType(paramExecExtensionCheck, dto),
				"true");
		boolean execExtensionCheck = Boolean
				.parseBoolean(valueExecExtensionCheck);

		// 拡張子リストを取得
		List<String> extensionList = (List<String>) ParamUtil
				.getParamObjectValueByType(ruleParams.get("list"), dto);

		// ファイルサイズを取得
		Map<String, Object> paramMaxSize = (Map<String, Object>) ruleParams
				.get("max_size");
		String valueMaxSize = StringUtils.defaultString(
				ParamUtil.getParamStrValueByType(paramMaxSize, dto), "0");
		BigDecimal bdMaxSize = new BigDecimal(valueMaxSize);

		// ファイルサイズ単を取得
		Map<String, Object> paramSizeUnit = (Map<String, Object>) ruleParams
				.get("size_unit");
		String valueSizeUnit = StringUtils.defaultString(
				ParamUtil.getParamStrValueByType(paramSizeUnit, dto),
				SizeUnit.BYTE);

		// D&Dのファイルかを取得
		Map<String, Object> paramDragAndDrop = (Map<String, Object>) ruleParams
				.get("drag_and_drop");
		String valueDragAndDrop = StringUtils.defaultString(
				ParamUtil.getParamStrValueByType(paramDragAndDrop, dto),
				"true");
		boolean dragAndDrop = Boolean.parseBoolean(valueDragAndDrop);

		// // エラーが出ても続けるかを取得
		// Map<String, Object> paramResume = (Map<String, Object>)
		// ruleParams.get("resume");
		// String valueResume =
		// StringUtils.defaultString(ParamUtil.getParamStrValueByType(paramResume,
		// dto), "true");
		// boolean resume = Boolean.parseBoolean(valueResume);

		// 拡張子チェック
		boolean extensionCheckResult = true;
		if (execExtensionCheck) {
			extensionCheckResult = checkExtention(extensionList, fileName);
			if (!extensionCheckResult) {
				return false;
			}
		}

		// サイズチェック
		boolean sizeCheckResult = true;
		if (execSizeCheck) {
			sizeCheckResult = checkFileSize(filePath, valueSizeUnit, bdMaxSize,
					dragAndDrop);
			if (!sizeCheckResult) {
				return false;
			}
		}

		return true;

	}

	/**
	 *
	 * 拡張子チェック
	 *
	 * @param list
	 * @param checkVal
	 * @return 結果(OK:true / NG:false)
	 */
	private boolean checkExtention(List<String> list, String checkVal) {

		// listで渡された拡張子リストの中の拡張子を持っていればOK
		// 持っていなければNG
		boolean ret = false;
		if (!CollectionUtils.isEmpty(list)) {
			for (String extension : list) {
				if (checkVal.toLowerCase().endsWith(extension)) {
					ret = true;
					break;
				}
			}
		}

		return ret;

	}

	/***
	 *
	 * サイズチェック
	 *
	 * @param filePath
	 * @param valueSizeUnit
	 * @param bdMaxSize
	 * @param dragAndDrop
	 * @return 結果(OK:true / NG:false)
	 */
	private boolean checkFileSize(	String filePath,
									String valueSizeUnit,
									BigDecimal bdMaxSize,
									boolean dragAndDrop) {

		BigDecimal dbSize = null;
		// paramValueで引き渡されたパスからファイル生成
		if (dragAndDrop) {
			byte[] inByte = Base64.decodeBase64(filePath);
			dbSize = new BigDecimal(inByte.length);
		} else {
			File f = new File(filePath);
			dbSize = new BigDecimal(f.length());
		}

		boolean ret = true;
		// 検査値の算出
		BigDecimal bdTestSize = bdMaxSize;
		BigDecimal bdSousand = new BigDecimal(1024);
		if (SizeUnit.KILO_BYTE.equals(valueSizeUnit)) {
			bdTestSize = bdTestSize.multiply(bdSousand);
		} else if (SizeUnit.MEGA_BYTE.equals(valueSizeUnit)) {
			bdTestSize = bdTestSize.multiply(bdSousand).multiply(bdSousand);
		}
		if (bdTestSize.compareTo(dbSize) < 0) {
			ret = false;
		}

		return ret;

	}

}
