package jp.ill.photon.module.validation;

import java.math.BigDecimal;
import java.util.Map;

import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringUtil;

import org.apache.commons.lang3.StringUtils;

public class CaseNumRangeValidationModule extends AbstractValidationModule {

	/**
	 *
	 * ruleParamsの中身
	 *  ------------------------------------------------------------------------
	 *  項目名                      : 説明                     : データ型
	 *  ------------------------------------------------------------------------
	 *  validation_maps             :                          : Map<String, Object>
	 *      quantity_per_case_value : 入数                     : String
	 *      max_value               : 最大値                   : String
	 *      not_include_max         : 最大値を範囲に含まない   : String
	 *      min_value               : 最少値                   : String
	 *      not_include_min         : 最大値を範囲に含まない   : String
	 *
	 */
	@SuppressWarnings("unchecked")
	public boolean validate(String paramName,
							Map<String, Object> ruleParams,
							String paramValue,
							ModuleContext context) {

		ActionDto dto = context.getDto();
		Map<String, Object> validationMaps =
				(Map<String, Object>) ParamUtil.getParamObjectValueByType(ruleParams.get("validation_maps"), dto);

		// チェック対象の値を算出
		BigDecimal val = getCheckTargetValue(paramValue, validationMaps, dto);

		// 取得できなかったときはエラー
		if (val == null) {
			return false;
		}

		// 最大値、最小値を取得
		BigDecimal max = getMaxValue(validationMaps, dto);
		BigDecimal min = getMinValue(validationMaps, dto);

		// 両方とも取得できなかったときはエラー
		if (min == null && max == null) {
			return false;
		}

		// 範囲に最大値を含まないかを取得
		String notIncludeMax = StringUtil.defaultString(
				ParamUtil.getParamStrValueByType(
						validationMaps.get("not_include_max"), dto),
				"");

		// 範囲に最小値を含まないかを取得
		String notIncludeMin = StringUtil.defaultString(
				ParamUtil.getParamStrValueByType(
						validationMaps.get("not_include_min"), dto),
				"");

		boolean isMinOk = true;
		boolean isMaxOk = true;

		if (min != null) {
			if (notIncludeMin.equals("1")) {
				isMinOk = (min.compareTo(val) < 0);
			} else {
				isMinOk = (min.compareTo(val) <= 0);
			}
		}

		if (max != null) {
			if (notIncludeMax.equals("1")) {
				isMaxOk = (val.compareTo(max) < 0);
			} else {
				isMaxOk = (val.compareTo(max) <= 0);
			}
		}

		return isMinOk && isMaxOk;

	}

	/***
	 * チェック対象の値取得
	 *
	 * @param paramValue
	 * @param ruleParams
	 * @param dto
	 * @return
	 */
	private BigDecimal getCheckTargetValue(String paramValue, Map<String, Object> ruleParams, ActionDto dto) {

		String quantityPerCaseValue = "";
		BigDecimal bdVal = null;

		if (ruleParams != null) {
			quantityPerCaseValue =  ParamUtil.getParamStrValueByType(ruleParams.get("quantity_per_case_value"), dto);
		}

		try {
			bdVal = new BigDecimal(quantityPerCaseValue);
		} catch (Exception e) {
		}

		BigDecimal bdParamVal = null;
		try {
			bdParamVal = new BigDecimal(paramValue);
		} catch (Exception e) {
		}

		if (bdVal == null || bdParamVal == null) {
			return null;
		}

		return bdVal.multiply(bdParamVal);

	}

	/***
	 * 最大値取得
	 *
	 * @param ruleParams
	 * @param dto
	 * @return
	 */
	private BigDecimal getMaxValue(Map<String, Object> ruleParams, ActionDto dto) {

		BigDecimal bdVal = null;

		// max_valueを取得
		String maxValue = (String) ParamUtil.getParamObjectValueByType(ruleParams.get("max"), dto);
		if (StringUtils.isEmpty(maxValue)) {
			return null;
		}

		try {
			bdVal = new BigDecimal(maxValue);
		} catch (Exception e) {
		}

		return bdVal;

	}

	/***
	 * 最小値取得
	 *
	 * @param ruleParams
	 * @param dto
	 * @return
	 */
	private BigDecimal getMinValue(Map<String, Object> ruleParams, ActionDto dto) {

		BigDecimal bdVal = null;

		// min_valueを取得
		String minValue = (String) ParamUtil.getParamObjectValueByType(ruleParams.get("min"), dto);
		if (StringUtils.isEmpty(minValue)) {
			return null;
		}

		try {
			bdVal = new BigDecimal(minValue);
		} catch (Exception e) {
		}

		return bdVal;

	}

}
