package jp.ill.photon.module.validation;

import java.math.BigDecimal;
import java.util.Map;

import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.model.AoSetting;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.util.AoFormatCheckUtil;
import jp.ill.photon.util.AoUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringUtil;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;

/**
 * AOフォーマットバリデーション
 * 売上数量/売上単価/売上金額/在庫数量/入数/ケース数 のフォーマットチェック
 *
 * @author y_hirota
 *
*/
public class AoFormatValidationModule extends AbstractValidationModule {

	/** バリデーションの内部でのみ使用する変数 */
	private ActionDto dto;
	private String max;
	private String min;

	public boolean validate(String paramName,
							Map<String, Object> ruleParams,
							String paramValue,
							ModuleContext context) {

		dto = context.getDto();

		// AO設定取得
		Map<String, Object> aoSettingMap = autoCast(dto.get("common", "aladdinsetting"));
		AoSetting aoSetting = AoSetting.valueOf(aoSettingMap);

		// AOフォーマット区分を取得
		Map<String, Object> aoFormatDiv = autoCast(ruleParams.get("ao_format_div"));
		String validAoFormatDiv = StringUtils.defaultString(ParamUtil.getParamStrValueByType(aoFormatDiv, dto), "");

		// 最大値、最小値取得
		setRange(ruleParams, validAoFormatDiv, aoSetting);

		// AOフォーマットチェック
		if(!AoFormatCheckUtil.aoFormatCheck(validAoFormatDiv, paramValue, aoSetting)){
			return false;
		}

		// 範囲チェック
		if (!checkRange(paramValue)) {
			return false;
		}

		return true;

	}

	/** Object型を自動キャスト */
	@SuppressWarnings("unchecked")
	public static <T> T autoCast(Object obj) {
	    T castObj = (T) obj;
	    return castObj;
	}

	/***
	 *
	 * 範囲設定
	 *
	 * @param ruleParams
	 * @param aoFormatDiv
	 * @param aoSetting
	 */
	private void setRange(Map<String, Object> ruleParams, String aoFormatDiv, AoSetting aoSetting) {

		// デフォルト範囲をセット
		StringBuffer sbValue = new StringBuffer();
		Map<String, String> digits = AoFormatCheckUtil.getAoRange(aoFormatDiv, aoSetting);
		String digit = digits.get(AoFormatCheckUtil.KEY_DIGIT);
		String digitDecimal = digits.get(AoFormatCheckUtil.KEY_DECIMAL_DIGIT);

		if (!StringUtils.isEmpty(digit) || !StringUtils.isEmpty(digitDecimal)) {

			int digitVal = Integer.parseInt(digit);
			int decimalVal = Integer.parseInt(StringUtils.defaultString(digitDecimal, "0") );

			sbValue.append(StringUtils.repeat("9", digitVal));

			if (decimalVal > 0) {

				sbValue.append(".");
				sbValue.append(StringUtils.repeat("9", decimalVal));

			}

			max = sbValue.toString();
			min = "-" + sbValue.toString();

		}

		// 設定されていたときのみ置き換え
		Map<String, Object> minMap = autoCast(ruleParams.get("min"));
		if (!MapUtils.isEmpty(minMap)) {
			min = StringUtil.defaultString(
					ParamUtil.getParamStrValueByType(minMap, dto),
					"").replaceAll(",", "");
		}

		Map<String, Object> maxMap = autoCast(ruleParams.get("max"));
		if (!MapUtils.isEmpty(maxMap)) {
			max = StringUtil.defaultString(
					ParamUtil.getParamStrValueByType(maxMap, dto),
					"").replaceAll(",", "");
		}

		// 最大値、最小値をセット
		// （エラーメッセージ出力用）
		getValidateResults().put("max", max);
		getValidateResults().put("max_formatted", AoUtil.convDecimalFormat(max));
		getValidateResults().put("min", min);
		getValidateResults().put("min_formatted", AoUtil.convDecimalFormat(min));

	}


	/***
	 *
	 * 範囲チェック
	 *
	 * @param val
	 * @return
	 */
	private boolean checkRange(String val) {

		if (min == null && max == null) {
			return false;
		}

		boolean isMinOk = true;
		boolean isMaxOk = true;

		if (min != null) {
			try {
				isMinOk = (new BigDecimal(min).compareTo(new BigDecimal(val)) <= 0);
			} catch (Exception e) {
				isMinOk = false;
			}
		}

		if (max != null) {
			try {
				isMaxOk = (new BigDecimal(val).compareTo(new BigDecimal(max)) <= 0);
			} catch (Exception e) {
				isMaxOk = false;
			}
		}

		return isMinOk && isMaxOk;

	}

}
