package jp.ill.photon.module.userchargeinfo;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.doma.SqlSelectQueryFactory;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.StringUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.seasar.doma.MapKeyNamingType;
import org.seasar.doma.internal.jdbc.command.MapResultListHandler;
import org.seasar.doma.jdbc.command.SelectCommand;
import org.seasar.doma.jdbc.query.SqlSelectQuery;
import org.seasar.doma.jdbc.tx.TransactionManager;

/**
 * 得意先担当者情報テーブルデータをマージするモジュール
 *
 * @author m_fukukawa
 *
 */
public class MergeUserChargeInfoModule implements PhotonModule {

	@ModuleParam(required=true)
	private String tenantId;

	@ModuleParam(required=true)
	private String userChargeId;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "static", transferVal = "")
	private String userChargeName;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "static", transferVal = "")
	private String userCd;

	@ModuleParam(required=false)
	private Map<String, Object> deliveryInfo;

	@ModuleParam(required=false)
	private String deliveryPrivilege;

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.exSqlFileDir.note")
	private String sqlFileDirPath;

	private JsonDataDao dao;

	public static class AuthorityDiv {
		private static String EDITABLE = "2";
		private static String VIEW_ONLY = "1";
		private static String NONE = "0";
	}

	@Override
	public ModuleResult execute(ModuleContext context) throws PhotonModuleException {

		Map<String, Object> tmpMap = new HashMap<String, Object>();
		if (!MapUtils.isEmpty(deliveryInfo)) {
			tmpMap = deliveryInfo;
		}

		// "delivery_privilege"が設定されていないときには、DBから権限データを取得する。
		String privilege = deliveryPrivilege;
		if (StringUtils.isEmpty(privilege)) {
			privilege = getPrivilege();
		}

		// お届け先利用権限がOFFだった場合、自社をセット
		if (!AuthorityDiv.EDITABLE.equals(privilege)) {
			tmpMap = getDeliveryFromUserCd();
		}

		// 得意先担当者情報テーブル 登録or更新
		addUserChargeInfo(context, tmpMap);

		ModuleResult result = new ModuleResult();
		return result;

	}

	/***
	 *
	 * 権限取得
	 *
	 * @return
	 */
	private String getPrivilege() {

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		dao = new JsonDataDaoImpl();

		String defValue = AuthorityDiv.EDITABLE;
		boolean adminDisp = false;
		String targetFunctionCd = "delivery";

		// 得意先別担当者権限リスト取得
		List<Map<String, Object>> userSystemAuthorityList = tm.required(() -> {
			return dao.getUserSystemAuthority(tenantId, defValue, adminDisp, userChargeId);
		});

		// 利用者権限マスタ利用者機能コード（Delivery）縛りデータ取得
		if (!CollectionUtils.isEmpty(userSystemAuthorityList)) {
			for (Map<String, Object> mp : userSystemAuthorityList) {
				if (targetFunctionCd.equals((String)mp.get("user_function_cd"))) {
					return (String)mp.get("authority_div");
				}
			}
		}

		return null;

	}

	/**
	 * 自得意先を取得し、それを納品先として扱う
	 *
	 * @return 納品先情報
	 *
	 * */
	private Map<String, Object> getDeliveryFromUserCd() {

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		dao = new JsonDataDaoImpl();

		Map<String, Object> userInfo = tm.required(() -> {
			return dao.getUserByUserCdForUserSite(tenantId, userCd, null);
		});

		Map<String, Object> deliveryInfo = null;

		if (!MapUtils.isEmpty(userInfo)) {
			deliveryInfo = new HashMap<String, Object>();
			deliveryInfo.put("user_delivery_cd", null);
			deliveryInfo.put("user_delivery_name1", (String)userInfo.get("user_name1"));
			deliveryInfo.put("user_delivery_name2", (String)userInfo.get("user_name2"));
			deliveryInfo.put("user_delivery_name",
					StringUtils.defaultIfBlank((String)userInfo.get("user_name1"), "") +
					StringUtils.defaultIfBlank((String)userInfo.get("user_name2"), "")
			);
			deliveryInfo.put("zip", (String)userInfo.get("zip"));
			deliveryInfo.put("city_name1", (String)userInfo.get("city_name1"));
			deliveryInfo.put("city_name2", (String)userInfo.get("city_name2"));
			deliveryInfo.put("city_name3", (String)userInfo.get("city_name3"));
			deliveryInfo.put("tel", (String)userInfo.get("tel"));
			deliveryInfo.put("fax", (String)userInfo.get("fax"));
		}

		return deliveryInfo;

	}

	/**
	 *
	 * 得意先担当者情報テーブル 登録or更新
	 *
	 * @param userDelivery 納品先情報
	 *
	 * */
	private void addUserChargeInfo(ModuleContext context, Map<String, Object> userDelivery)
			throws PhotonModuleException {

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		SqlSelectQueryFactory selectFactory = SqlSelectQueryFactory.newInstance();
		dao = new JsonDataDaoImpl();

		Map<String, Object> userChargeInfo = tm.required(() -> {
			return dao.getUserChargeInfoByUserCdUserChargeId(tenantId, userCd, userChargeId);
		});

		boolean isMyCompany = (StringUtils.isEmpty((String)userDelivery.get("user_delivery_cd")));

		if (userChargeInfo == null) {
			userChargeInfo = new HashMap<String, Object>();
		}

		// true:新規登録 false:既存データの更新
		boolean newRegist = MapUtils.isEmpty(userChargeInfo);

		userChargeInfo.put("user_cd", userCd);
		userChargeInfo.put("user_charge_id", userChargeId);

		if (!isMyCompany) {
			userChargeInfo.put("user_delivery_cd", (String)userDelivery.get("user_delivery_cd"));
		} else {
			userChargeInfo.put("user_delivery_cd", null);
		}

		if (!StringUtils.isEmpty((String)userDelivery.get("user_delivery_name"))) {
			// お届け先名の入力フォームの値を30バイトで分割
			String[] name = StringUtil.lengthSplitBytes((String)userDelivery.get("user_delivery_name"), 30, "Windows-31J");
			userChargeInfo.put("user_delivery_name1", name[0]);
			if (name.length >= 2) {
				userChargeInfo.put("user_delivery_name2", name[1]);
			} else {
				userChargeInfo.put("user_delivery_name2", "");
			}
		}

		userChargeInfo.put("zip", (String)userDelivery.get("zip"));
		userChargeInfo.put("city_name1", (String)userDelivery.get("city_name1"));
		userChargeInfo.put("city_name2", (String)userDelivery.get("city_name2"));
		userChargeInfo.put("city_name3", (String)userDelivery.get("city_name3"));
		userChargeInfo.put("tel", (String)userDelivery.get("tel"));
		userChargeInfo.put("fax", (String)userDelivery.get("fax"));

		if (!isMyCompany) {

			// 納品先情報にすでに登録されているデータの場合、当該レコードの納品先コードを取得して設定する
			HashMap<String, Object> checkParams = createDeliveryParams(userDelivery);
			SqlSelectQuery checkQuery = selectFactory.createSelectQueryFromFile(
					checkParams, context.getDto(),
					sqlFileDirPath, "aec20/user_delivery/getCheckData.sql");
			checkQuery.prepare();

			// 検索処理を実行
			List<Map<String, Object>> checkDeliveryList = tm.required(() -> {
				SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
						checkQuery,
						new MapResultListHandler(MapKeyNamingType.NONE));
				return command.execute();
			});

			if (checkDeliveryList != null && checkDeliveryList.size() == 1) {
				Map<String, Object> deliveryData = checkDeliveryList.get(0);
				userChargeInfo.put("user_delivery_cd", (String)deliveryData.get("user_delivery_cd"));
			}

		}

		// ◆INSERT、またはUPDATEで使用するentity_data.valの部分を作成する
		String data = JsonUtil.mapToJson(userChargeInfo);

		// 担当者情報テーブルに情報が存在有無により、登録・更新を行う
		int returnCd = tm.required(() -> {

			Map<String, Object> ret = null;

			if ( newRegist ) {
				// 挿入
				ret = dao.insertEditTableData(tenantId, "userchargeinfo", userChargeName, data);
			} else {
				// 更新
				ret = dao.updateEditTableData(tenantId, "userchargeinfo", userChargeName, data);
			}

			Integer retVal = (Integer)ret.get("ret");
			if (retVal != 0) {
				return -1;
			}
			return 0;

		});

	}

	private Map<String, Object> toSubMap(String val){
		 Map<String, Object> subMap = new  LinkedHashMap<String, Object>();
		 subMap.put("type", "static");
		 subMap.put("val", val);
		 return subMap;
	}

	/** お届け先存在チェックSQLパラメータ作成 */
	private HashMap<String, Object> createDeliveryParams(Map<String, Object> userDelivery) {
		HashMap<String, Object> paramMap = new HashMap<String, Object>();
		paramMap.put("tenantId", toSubMap(tenantId));
		paramMap.put("userCd", toSubMap(userCd));
		paramMap.put("name", toSubMap((String) userDelivery.get("user_delivery_name")));
		paramMap.put("zip", toSubMap((String) userDelivery.get("zip")));
		paramMap.put("cityName1", toSubMap((String) userDelivery.get("city_name1")));
		paramMap.put("cityName2", toSubMap((String) userDelivery.get("city_name2")));
		paramMap.put("cityName3", toSubMap((String) userDelivery.get("city_name3")));
		paramMap.put("tel", toSubMap((String) userDelivery.get("tel")));
		paramMap.put("fax", toSubMap((String) userDelivery.get("fax")));

		return paramMap;
	}

	/**
	 * tenantIdを取得します。
	 * @return tenantId
	 */
	public String getTenantId() {
		return tenantId;
	}

	/**
	 * tenantIdを設定します。
	 * @param tenantId
	 */
	public void setTenantId(String tenantId) {
		this.tenantId = tenantId;
	}

	/**
	 * userChargeIdを取得します。
	 * @return userChargeId
	 */
	public String getUserChargeId() {
		return userChargeId;
	}

	/**
	 * userChargeIdを設定します。
	 * @param userChargeId
	 */
	public void setUserChargeId(String userChargeId) {
		this.userChargeId = userChargeId;
	}

	/**
	 * userChargeNameを取得します。
	 * @return userChargeName
	 */
	public String getUserChargeName() {
		return userChargeName;
	}

	/**
	 * userChargeNameを設定します。
	 * @param userChargeName
	 */
	public void setUserChargeName(String userChargeName) {
		this.userChargeName = userChargeName;
	}

	/**
	 * userCdを取得します。
	 * @return userCd
	 */
	public String getUserCd() {
		return userCd;
	}

	/**
	 * userCdを設定します。
	 * @param userCd
	 */
	public void setUserCd(String userCd) {
		this.userCd = userCd;
	}

	/**
	 * deliveryInfoを取得します。
	 * @return deliveryInfo
	 */
	public Map<String, Object> getDeliveryInfo() {
		return deliveryInfo;
	}

	/**
	 * deliveryInfoを設定します。
	 * @param deliveryInfo
	 */
	public void setDeliveryInfo(Map<String, Object> deliveryInfo) {
		this.deliveryInfo = deliveryInfo;
	}

	/**
	 * deliveryPrivilegeを取得します。
	 * @return deliveryPrivilege
	 */
	public String getDeliveryPrivilege() {
		return deliveryPrivilege;
	}

	/**
	 * deliveryPrivilegeを設定します。
	 * @param deliveryPrivilege
	 */
	public void setDeliveryPrivilege(String deliveryPrivilege) {
		this.deliveryPrivilege = deliveryPrivilege;
	}

	/**
	 * sqlFileDirPathを取得します。
	 * @return sqlFileDirPath
	 */
	public String getSqlFileDirPath() {
		return sqlFileDirPath;
	}

	/**
	 * sqlFileDirPathを設定します。
	 * @param sqlFileDirPath
	 */
	public void setSqlFileDirPath(String sqlFileDirPath) {
		this.sqlFileDirPath = sqlFileDirPath;
	}

}
