package jp.ill.photon.module.tax;

import java.math.BigDecimal;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.model.AoSetting;
import jp.ill.photon.model.CartItem;
import jp.ill.photon.model.Item;
import jp.ill.photon.model.Tax;
import jp.ill.photon.model.Tax.TaxCalcDiv;
import jp.ill.photon.model.Tax.TaxDiv;
import jp.ill.photon.model.TaxSetting;
import jp.ill.photon.model.User;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.AoUtil;
import jp.ill.photon.util.StringUtil;
import jp.ill.photon.util.UtilTools;

public class CalcTaxModule implements PhotonModule {
	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "cart.list")
	private List<Map<String, Object>> detailList;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "got_session.userForm.user_cd")
	private String userCd;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.aladdinsetting")
	private AoSetting aoSetting;

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "dto", transferVal = "delivery_fee.delivery_fee")
	private Map<String, Object> deliveryFee;

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "dto", transferVal = "charge_fee.charge_fee")
	private Map<String, Object> chargeFee;

	/** メッセージID（SMBC通信結果エラー） */
    public static final String CART_ERROR_MSG_ID = "calcTaxErrMes";

    private List<String> ngItemList;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "user.first")
    private Map<String, Object> userForm;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.verNo.note")
	private String verNo;

	/** 単価端数処理区分 */
	private int unitPriceFractionDiv;

	/** 金額端数処理区分 */
	private int priceFractionDiv;

    public Map<String, Object> getUserForm() {
        return userForm;
    }

    public void setUserForm(Map<String, Object> userForm) {
        this.userForm = userForm;
    }

	public Map<String, Object> getChargeFee() {
		return chargeFee;
	}

	public void setChargeFee(Map<String, Object> chargeFee) {
		this.chargeFee = chargeFee;
	}

	public Map<String, Object> getDeliveryFee() {
		return deliveryFee;
	}

	public void setDeliveryFee(Map<String, Object> deliveryFee) {
		this.deliveryFee = deliveryFee;
	}

	public AoSetting getAoSetting() {
		return aoSetting;
	}

	public void setAoSetting(AoSetting aoSetting) {
		this.aoSetting = aoSetting;
	}

	public String getUserCd() {
		return userCd;
	}

	public void setUserCd(String userCd) {
		this.userCd = userCd;
	}

	public List<Map<String, Object>> getDetailList() {
		return detailList;
	}

	public void setDetailList(List<Map<String, Object>> detailList) {
		this.detailList = detailList;
	}

	public static boolean isVer12(String verNo) {
		return "1.2notax".equals(verNo);
	}

	/**
	 * AO12 軽減税率非対応版判定
	 */
	public boolean isVer12() {
		return isVer12(getVerNo());
	}

	public static boolean isVer12ReducedRate(String verNo) {
		return "1.2intax".equals(verNo);
	}

	/**
	 * AO12 軽減税率対応版判定
	 */
	public boolean isVer12ReducedRate() {
		return isVer12ReducedRate(getVerNo());
	}

	/**
	 * verNoを取得します。
	 * @return verNo
	 */
	public String getVerNo() {
		return verNo == null ? "1.3" : verNo;
	}

	/**
	 * verNoを設定します。
	 * @param verNo verNo
	 */
	public void setVerNo(String verNo) {
		this.verNo = verNo;
	}

	/**
	 * 単価端数処理区分を取得します。
	 * @return 単価端数処理区分
	 */
	public int getUnitPriceFractionDiv() {
		return unitPriceFractionDiv;
	}

	/**
	 * 単価端数処理区分を設定します。
	 * @param unitPriceFractionDiv 単価端数処理区分
	 */
	public void setUnitPriceFractionDiv(int unitPriceFractionDiv) {
		this.unitPriceFractionDiv = unitPriceFractionDiv;
	}

	/**
	 * 単価端数処理区分を設定します。
	 * @param unitPriceFractionDiv 単価端数処理区分
	 */
	public void setUnitPriceFractionDiv(User user) {
		this.unitPriceFractionDiv = (user != null) ? user.getUnitPriceFractionDiv() : null;
	}

	/**
	 * 金額端数処理区分を取得します。
	 * @return 金額端数処理区分
	 */
	public int getPriceFractionDiv() {
		return priceFractionDiv;
	}

	/**
	 * 金額端数処理区分を設定します。
	 * @param priceFractionDiv 金額端数処理区分
	 */
	public void setPriceFractionDiv(int priceFractionDiv) {
		this.priceFractionDiv = priceFractionDiv;
	}

	/**
	 * 金額端数処理区分を設定します。
	 * @param priceFractionDiv 金額端数処理区分
	 */
	public void setPriceFractionDiv(User user) {
		this.priceFractionDiv = (user != null) ? user.getPriceFractionDiv() : null;
	}

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		ModuleResult result = new ModuleResult();

		ngItemList = new ArrayList<>();

		try {
		    User user = User.getUser(context.getDto(), userCd);
			String taxPriceSum = getTaxPrice(context, user, result);
	        String taxPriceSumFormat = AoUtil.convAoRoundingSalesPrice(taxPriceSum,
	                aoSetting.getSalesPriceDigitNum(),
					StringUtil.defaultString(user.getPriceFractionDiv(), "0"));

	        // 配送料
	        if (deliveryFee != null) {
	            // 配送料は、新しい結果としてPUT。（detailListは更新しているのに、整合性がない？）
                result.getReturnData().put("delivery_fee", deliveryFee);
	        }
	        // 手数料
            if (chargeFee != null) {
                // 手数料は、新しい結果としてPUT。
                result.getReturnData().put("charge_fee", chargeFee);
            }

	        result.getReturnData().put("total_tax_price",
	                taxPriceSumFormat.replace(",", ""));

	        if (!TaxDiv.INCLUDING.equals(StringUtil.defaultString(user.getTaxDiv(), ""))) {
	            result.getReturnData().put("total_tax_price_for_sum",
	                    taxPriceSumFormat);
	        } else {
	            result.getReturnData().put("total_tax_price_for_sum", "0");
	        }

		} catch (PhotonModuleException e) {
		    result.setResultCode("error");

		    List<ActionDtoMessage> errorMessages = new ArrayList<ActionDtoMessage>();
		    ActionDtoMessage msg = new ActionDtoMessage();
            msg.setMessageId(CART_ERROR_MSG_ID);
            errorMessages.add(msg);

		    result.getMessages().put("error_page", errorMessages);
		}

		result.getReturnData().put("ng_item_list", ngItemList);

		return result;
	}

	public String getTaxPrice(
								ModuleContext context,
									User user,
									ModuleResult result)
			throws PhotonModuleException {
		JsonDataDao dao = new JsonDataDaoImpl();
		Timestamp nowDate = new Timestamp(System.currentTimeMillis());

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		String taxPriceSum = "0";

		if (TaxCalcDiv.ORDER_DETAIL
				.equals(String.valueOf(user.getTaxCalcDiv()))) {

			for (int i = 0; i < detailList.size(); i++) {
				Map<String, Object> detailMap = detailList.get(i);

				try {
					Map<String, Object> detailItem = calcTaxPrice(context, detailMap, user, nowDate);

					taxPriceSum = UtilTools.addBigDecimal(taxPriceSum, String.valueOf(detailItem.get("tax_price")));

					// カートのアイテムを更新
					detailList.set(i, detailItem);

				} catch (PhotonModuleException e) {
					ngItemList.add(String.valueOf(detailMap.get("item_cd")));
					throw e;
				}
			}

			// 配送料
			if (deliveryFee != null) {
				Map<String, Object> detailItem = calcTaxPrice(context, deliveryFee, user, nowDate);
				taxPriceSum = UtilTools.addBigDecimal(taxPriceSum, String.valueOf(detailItem.get("tax_price")));
			}

			// 手数料
			if (chargeFee != null) {
				Map<String, Object> detailItem = calcTaxPrice(context, chargeFee, user, nowDate);
				taxPriceSum = UtilTools.addBigDecimal(taxPriceSum, String.valueOf(detailItem.get("tax_price")));
			}

		} else if (TaxCalcDiv.ORDER_TOTAL
				.equals(String.valueOf(user.getTaxCalcDiv()))) {
			String taxRate = "";
			List<TaxSetting> taxList = new ArrayList<>();

			List<Map<String, Object>> taxListMap = null;
			if (isVer12()) {
				taxListMap = tm.required(() -> {
					return dao.getTaxRateByDate(
							context.getDto().getTenantId(),
							nowDate);
				});

			} else {
				taxListMap = tm.required(() -> {
					return dao.getTaxReducedRateByDate(
							context.getDto().getTenantId(),
							nowDate);
				});

			}

			for (Map<String, Object> taxMap : taxListMap) {
				taxList.add(TaxSetting.valueOf(taxMap));
			}

			Map<String, String> noTaxPriceMap = new HashMap<String, String>();
			for (int i = 0; i < taxList.size(); i++) {
				noTaxPriceMap.put(taxList.get(i).getTaxRate().toString(), "0");
			}

			for (int i = 0; i < detailList.size(); i++) {
				Map<String, Object> detailMap = detailList.get(i);

				Item item = tm.required(() -> {
					List<Map<String, Object>> itemMap = dao.getItemForOrder(
							context.getDto().getTenantId(),
							String.valueOf(detailMap.get("item_cd")),
							user.getUserCd());
					if (itemMap == null || itemMap.isEmpty()) {
						return null;
					} else {
						return Item.valueOf(itemMap.get(0));
					}
				});

				if (item == null)
					throw new PhotonModuleException("item is invalid", null);

				BigDecimal subTotalPrice = BigDecimal.ZERO;
				if (!((isVer12() || isVer12ReducedRate()) && Tax.TaxExemptDiv.ON.equals(String.valueOf(item.getTaxExemptDiv())))) {
					subTotalPrice = CartItem
							.getSubTotalPriceFormat(aoSetting, detailMap, userForm);

				}
				taxRate = getTaxRateForOrderSum(
						context,
						detailMap,
						user,
						nowDate);

				for (TaxSetting rate : taxList) {
					if (rate.getTaxRate().toString().equals(taxRate) || isVer12()) {
						noTaxPriceMap.put(rate.getTaxRate().toString(),
								UtilTools.addBigDecimal(
										noTaxPriceMap.get(
												rate.getTaxRate().toString()),
										subTotalPrice.toString()));
					}
				}

				detailMap.put("tax_price", new BigDecimal("0"));
				detailMap.put("tax_rate", Double.valueOf(taxRate));

				// カートのアイテムを更新
				detailList.set(i, detailMap);
			}

			// 配送料
			if (deliveryFee != null) {
				BigDecimal subTotalPrice = CartItem
						.getSubTotalPriceFormat(aoSetting, deliveryFee, userForm);
				taxRate = getTaxRateForOrderSum(context, deliveryFee, user,
						nowDate);

				for (TaxSetting rate : taxList) {
					if (rate.getTaxRate().toString().equals(taxRate) || isVer12()) {
						noTaxPriceMap.put(rate.getTaxRate().toString(),
								UtilTools.addBigDecimal(
										noTaxPriceMap.get(
												rate.getTaxRate().toString()),
										subTotalPrice.toString()));
					}
				}

				deliveryFee.put("tax_price", new BigDecimal("0"));
				deliveryFee.put("tax_rate", Double.valueOf(taxRate));
			}

			// 手数料
			if (chargeFee != null) {
				BigDecimal subTotalPrice = CartItem
						.getSubTotalPriceFormat(aoSetting, chargeFee, userForm);

				taxRate = getTaxRateForOrderSum(context, chargeFee, user,
						nowDate);

				for (TaxSetting rate : taxList) {
					if (rate.getTaxRate().toString().equals(taxRate) || isVer12()) {
						noTaxPriceMap.put(rate.getTaxRate().toString(),
								UtilTools.addBigDecimal(
										noTaxPriceMap.get(
												rate.getTaxRate().toString()),
										subTotalPrice.toString()));
					}
				}

				chargeFee.put("tax_price", new BigDecimal("0"));
				chargeFee.put("tax_rate", Double.valueOf(taxRate));
			}

			// 合計消費税
			for (TaxSetting rate : taxList) {
				taxPriceSum = UtilTools.addBigDecimal(taxPriceSum,
						Tax.taxCalcPrice(
								noTaxPriceMap.get(rate.getTaxRate().toString()),
								rate.getTaxRate().toString(),
								String.valueOf(user.getTaxDiv()),
								String.valueOf(user.getTaxFractionDiv()),
								String.valueOf(user.getTaxFractionDigitDiv())));
			}
		}
		// 消費税算出区分(請求合計、他)
		else {
		}

		return taxPriceSum;
	}

	/**
	 * 指定されてた商品コードの税率を取得する（内税のときのキャンペーン・手数料・送料の消費税分加算用）
	 * @param context
	 * @param itemCd
	 * @param userCd
	 * @param verNo
	 * @return
	 * @throws PhotonModuleException
	 */
	public static Double getTaxRateFromItemCd(ModuleContext context, String itemCd, String userCd, String verNo) throws PhotonModuleException {

		JsonDataDao dao = new JsonDataDaoImpl();
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		Timestamp nowDate = new Timestamp(System.currentTimeMillis());

		Item item = tm.required(() -> {
			List<Map<String, Object>> itemMap = dao.getItemForOrder(
					context.getDto().getTenantId(),
					itemCd, userCd);
			if (itemMap == null || itemMap.isEmpty()) {
				return null;
			} else {
				return Item.valueOf(itemMap.get(0));
			}
		});

		if (item == null)
			throw new PhotonModuleException("item is invalid", null);

		if (isVer12(verNo)) {

			// 非課税区分が"1"(非課税)の場合、消費税を０円とする
			if (Tax.TaxExemptDiv.ON.equals(String.valueOf(item.getTaxExemptDiv()))) {
				return 0.0;

			}

			TaxSetting taxSetting = tm.required(() -> {
				List<Map<String, Object>> taxMap = dao.getTaxRateByDate(
						context.getDto().getTenantId(),
						nowDate);
				if (taxMap == null || taxMap.isEmpty()) {
					return null;
				} else {
					if (taxMap.size() > 1) {
						return null;
					} else {
						return TaxSetting.valueOf(taxMap.get(0));
					}
				}
			});

			if (taxSetting == null)
				throw new PhotonModuleException("tax seting is not found", null);

			return taxSetting.getTaxRate();

		} else {
			// AO12 消費税対応版かつ、非課税区分が"1"(非課税)の場合、消費税を０円とする
			if (isVer12ReducedRate(verNo) && Tax.TaxExemptDiv.ON.equals(String.valueOf(item.getTaxExemptDiv()))) {
				return 0.0;
			}

			TaxSetting taxSetting = tm.required(() -> {
				List<Map<String, Object>> taxMap = dao.getTaxSettingByTaxDivCd(
						context.getDto().getTenantId(),
						String.valueOf(item.getTaxBracketCd()),
						nowDate);
				if (taxMap == null || taxMap.isEmpty()) {
					return null;
				} else {
					if (taxMap.size() > 1) {
						return null;
					} else {
						return TaxSetting.valueOf(taxMap.get(0));
					}
				}
			});

			if (taxSetting == null)
				throw new PhotonModuleException("tax seting is not found", null);

			return taxSetting.getTaxRate();
		}
	}

	/**
	 * detailItemには、 item_cd wholesale_price order_quantity tax_price tax_rate
	 * があることを前提
	 *
	 * @param context
	 * @param detailItem
	 * @param user
	 * @param nowDate
	 * @return
	 * @throws PhotonModuleException
	 */
	private Map<String, Object> calcTaxPrice(
												ModuleContext context,
												Map<String, Object> detailItem,
												User user,
												Timestamp nowDate) throws PhotonModuleException {
		Map<String, Object> rtnItem = null;
		if (isVer12()) {
			rtnItem = calcTaxPriceForOrderDetail(context, detailItem, user, nowDate);

		} else {
			rtnItem = calcTaxPriceForOrderDetailReducedRate(context, detailItem, user, nowDate);

		}

		return rtnItem;
	}

	/**
	 * 【軽減税率 非 対応版】<br />
	 * detailItemには、 item_cd wholesale_price order_quantity tax_price tax_rate
	 * があることを前提
	 *
	 * @param context
	 * @param detailItem
	 * @param user
	 * @param nowDate
	 * @return
	 * @throws PhotonModuleException
	 */
	private Map<String, Object> calcTaxPriceForOrderDetail(
															ModuleContext context,
															Map<String, Object> detailItem,
															User user,
															Timestamp nowDate)
			throws PhotonModuleException {
		JsonDataDao dao = new JsonDataDaoImpl();
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		Item item = tm.required(() -> {
			List<Map<String, Object>> itemMap = dao.getItemForOrder(
					context.getDto().getTenantId(),
					String.valueOf(detailItem.get("item_cd")),
					user.getUserCd());
			if (itemMap == null || itemMap.isEmpty()) {
				return null;
			} else {
				return Item.valueOf(itemMap.get(0));
			}
		});

		if (item == null)
			throw new PhotonModuleException("item is invalid", null);

		TaxSetting taxSetting = tm.required(() -> {
			List<Map<String, Object>> taxMap = dao.getTaxRateByDate(
					context.getDto().getTenantId(),
					nowDate);
			if (taxMap == null || taxMap.isEmpty()) {
				return null;
			} else {
				if (taxMap.size() > 1) {
					return null;
				} else {
					return TaxSetting.valueOf(taxMap.get(0));
				}
			}
		});

		if (taxSetting == null)
			throw new PhotonModuleException("tax seting is not found", null);

		detailItem.put("tax_rate", taxSetting.getTaxRate());

		// 非課税区分が"1"(非課税)の場合、消費税を０円とする
		if (Tax.TaxExemptDiv.ON.equals(String.valueOf(item.getTaxExemptDiv()))) {
			detailItem.put("tax_price", BigDecimal.ZERO);

		} else {
			// 消費税算出
			String taxPrice = getTaxPrice(detailItem, taxSetting, user);
			detailItem.put("tax_price", new BigDecimal(taxPrice));

		}
		return detailItem;
	}

	/**
	 * 【軽減税率 対応版】<br />
	 * detailItemには、 item_cd wholesale_price order_quantity tax_price tax_rate
	 * があることを前提
	 *
	 * @param context
	 * @param detailItem
	 * @param user
	 * @param nowDate
	 * @return
	 * @throws PhotonModuleException
	 */
	private Map<String, Object> calcTaxPriceForOrderDetailReducedRate(
																		ModuleContext context,
																		Map<String, Object> detailItem,
																		User user,
																		Timestamp nowDate)
			throws PhotonModuleException {
		JsonDataDao dao = new JsonDataDaoImpl();
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		Item item = tm.required(() -> {
			List<Map<String, Object>> itemMap = dao.getItemForOrder(
					context.getDto().getTenantId(),
					String.valueOf(detailItem.get("item_cd")),
					user.getUserCd());
			if (itemMap == null || itemMap.isEmpty()) {
				return null;
			} else {
				return Item.valueOf(itemMap.get(0));
			}
		});

		if (item == null)
			throw new PhotonModuleException("item is invalid", null);

		TaxSetting taxSetting = tm.required(() -> {
			List<Map<String, Object>> taxMap = dao.getTaxSettingByTaxDivCd(
					context.getDto().getTenantId(),
					String.valueOf(item.getTaxBracketCd()),
					nowDate);
			if (taxMap == null || taxMap.isEmpty()) {
				return null;
			} else {
				if (taxMap.size() > 1) {
					return null;
				} else {
					return TaxSetting.valueOf(taxMap.get(0));
				}
			}
		});

		if (taxSetting == null)
			throw new PhotonModuleException("tax seting is not found", null);

		detailItem.put("tax_rate", taxSetting.getTaxRate());

		// AO12 消費税対応版かつ、非課税区分が"1"(非課税)の場合、消費税を０円とする
		if (isVer12ReducedRate() && Tax.TaxExemptDiv.ON.equals(String.valueOf(item.getTaxExemptDiv()))) {
			detailItem.put("tax_price", BigDecimal.ZERO);

		} else {
			// 消費税算出
			String taxPrice = getTaxPrice(detailItem, taxSetting, user);
			detailItem.put("tax_price", new BigDecimal(taxPrice));

		}

		return detailItem;
	}

	private String getTaxRateForOrderSum(	ModuleContext context,
											Map<String, Object> detailItem,
											User user,
											Timestamp nowDate)
			throws PhotonModuleException {
		JsonDataDao dao = new JsonDataDaoImpl();
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		Item item = tm.required(() -> {
			List<Map<String, Object>> itemMap = dao.getItemForOrder(
					context.getDto().getTenantId(),
					String.valueOf(detailItem.get("item_cd")),
					user.getUserCd());
			if (itemMap == null || itemMap.isEmpty()) {
				return null;
			} else {
				return Item.valueOf(itemMap.get(0));
			}
		});
		if (item == null) {
			return "0";
		}

		TaxSetting taxSetting = tm.required(() -> {
			List<Map<String, Object>> taxMap = dao.getTaxSettingByTaxDivCd(
					context.getDto().getTenantId(),
					String.valueOf(item.getTaxBracketCd()), nowDate);
			if (taxMap == null || taxMap.isEmpty()) {
				return null;
			} else {
			    if (taxMap.size() > 1) {
                    return null;
                } else {
                    return TaxSetting.valueOf(taxMap.get(0));
                }
			}
		});

		if (taxSetting == null)
            throw new PhotonModuleException("tax seting is not found", null);

		return taxSetting.getTaxRate().toString();
	}

	/**
	 * 単価を取得します
	 *
	 * @param detailItem 商品情報
	 */
	private String getCalcWholeSalePrice(Map<String, Object> detailItem) {
		return AoUtil.convAoRoundingSalesUnitPrice(
				StringUtil.defaultString(detailItem.get("wholesale_price"), "0"),
				AoUtil.DispPattern.PATTERN_NO_COMMA,
				aoSetting.getSalesUnitPriceDigitNum(),
				aoSetting.getSalesUnitPriceDecimalPointNum(),
				StringUtil.defaultString(getUnitPriceFractionDiv(), "0"));
	}

	/**
	 * 小計を取得します
	 *
	 * @param detailItem 商品情報
	 */
	private String getSubTotalPrice(Map<String, Object> detailItem) {
		return AoUtil.convAoRoundingSalesPrice(
				getCalcWholeSalePrice(detailItem).toString(),
				StringUtil.defaultString(detailItem.get("order_quantity"), "0"),
				AoUtil.DispPattern.PATTERN_NO_COMMA,
				aoSetting.getSalesPriceDigitNum(),
				StringUtil.defaultString(getPriceFractionDiv(), "0"));
	}

	/**
	 * 消費税を取得します
	 *
	 * @param detailItem 商品情報
	 * @param taxSetting 消費税率設定ファイル情報
	 * @param user 得意先情報
	 */
	private String getTaxPrice(Map<String, Object> detailItem, TaxSetting taxSetting, User user) {
		return Tax.taxCalcPrice(
				getSubTotalPrice(detailItem).toString(),
				taxSetting.getTaxRate().toString(),
				StringUtil.defaultString(user.getTaxDiv()),
				StringUtil.defaultString(user.getTaxFractionDiv()),
				StringUtil.defaultString(user.getTaxFractionDigitDiv()));
	}
}
