package jp.ill.photon.module.output;

import java.io.ByteArrayInputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.Map;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.annotation.ModuleVersion;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.JsonUtil;


@ModuleVersion("1.0.0")
public class XMLOutputModule implements PhotonModule {

	static final String ENCODE =  "UTF-8";

	@ModuleParam(required=false)
	private String resultMapKey;

	public String getResultMapKey() {
		return resultMapKey;
	}

	public void setResultMapKey(String resultMapKey) {
		this.resultMapKey = resultMapKey;
	}

	@ModuleParam(required=false)
	private String resultListKey;

	public String getResultListKey() {
		return resultListKey;
	}

	public void setResultListKey(String resultListKey) {
		this.resultListKey = resultListKey;
	}


	@Override
	public ModuleResult execute( ModuleContext context ) throws PhotonModuleException {
		ModuleResult result = new ModuleResult();

		// 前モジュールで格納していた結果リスト文字列を取得する
		@SuppressWarnings("unchecked")
		String shohinJson = JsonUtil.mapToJson(((Map<String, Object>)((Map<String, Object>) context.getDto().getDataMap()).get(resultMapKey)).get(resultListKey));

		try {
			// （URLでの）呼び出しAPI名取得
			String requestURL = context.getDto().getParams().get("_init.url");
			String apiName = new String();
			if(requestURL.lastIndexOf("/")==-1 || requestURL.lastIndexOf("/")==requestURL.length()){
				apiName = "";
			}else{
				apiName = requestURL.substring(requestURL.lastIndexOf("/")+1);
			}

			// XML出力
			outputXml(apiName,result,shohinJson);

		} catch (PhotonModuleException ex) {
			throw new PhotonModuleException(" putオペレーションがこのマップでサポートされない場合のエラー", ex);
		}

		return result;
	}


	/**
	 * XML出力
	 *
	 * @param apiName API名称
	 * @param result モジュール結果設定用変数。ModuleResult
	 * @param shohinJson 「[{ID:VALUE,ID:VALUE},{ID:VALUE,ID:VALUE},{ID:VALUE,・・・}]」文字列
	 * @throws PhotonModuleException getBytesメソッドにて指定された文字セットがサポートされていない場合 - UnsupportedEncodingException
	 */
	private void outputXml(String apiName,ModuleResult result,String shohinJson) throws PhotonModuleException {

		try{

			// 他モジュールのエラーメッセージ取得＆格納用
			String[] errorMsg = new String[0];
			// 返却フォーマットのResultStatus部分を作成
			String xmlResultStatusData = makeResultStatus(errorMsg);

			// 「[{ID:VALUE,ID:VALUE},{ID:VALUE,ID:VALUE},{ID:VALUE,・・・}]」文字列を、[{ }] でくくられたデータ毎に分割した文字配列にする
			String[] individualMap = deleteBracketArray(shohinJson);
			// 返却フォーマットのResult部分を作成
			String xmlResultData = makeResult(individualMap);

			// 出力用のXMLを作成
			String shohinListXml = makeXml(apiName,individualMap,xmlResultStatusData,xmlResultData);

			result.getReturnData().put("input_stream", new ByteArrayInputStream(shohinListXml.getBytes(ENCODE)));
			result.getReturnData().put("content_length", shohinListXml.getBytes(Charset.forName(ENCODE)).length);
			result.getReturnData().put("content_type", "application/xml; charset="+ENCODE);


		} catch (UnsupportedEncodingException ex) {
			throw new PhotonModuleException(" putオペレーションがこのマップでサポートされない場合のエラー", ex);
		}
	}



	/**
	 * 出力用XMLを作成する
	 *
	 * @param apiName 呼び出しAPI名
	 * @param individualDataMap データ毎の配列
	 * @param errorString Error部分の文字列
	 * @param resultString Result部分の文字列
	 * @return 出力用XML
	 */
	private String makeXml(String apiName,String[] individualDataMap,String resultStatusString,String resultString) {
		StringBuffer tempXmlBuffer = new StringBuffer();
		tempXmlBuffer = tempXmlBuffer.append("<?xml version='1.0' encoding='" + ENCODE + "'?>");
		tempXmlBuffer = tempXmlBuffer.append("<" + apiName + " version='1.0'>");
			tempXmlBuffer = tempXmlBuffer.append("<ResultSet TotalResult='"+individualDataMap.length+"'>");
				tempXmlBuffer = tempXmlBuffer.append(resultStatusString);
				tempXmlBuffer = tempXmlBuffer.append(resultString);
			tempXmlBuffer = tempXmlBuffer.append("</ResultSet>");
		tempXmlBuffer = tempXmlBuffer.append("</" + apiName + ">");


		return tempXmlBuffer.toString();
	}



	/**
	 * 返却フォーマットのResultStatus部分を作成
	 *
	 * @param individualMap エラーデータ結果文字配列
	 * @return XML形式のエラー文字列
	 */
	private String makeResultStatus(String[] errorMsgMap) {

		StringBuffer xmlStatusStBuffer = new StringBuffer();

		xmlStatusStBuffer = xmlStatusStBuffer.append("<ResultStatus>");
		if(errorMsgMap!=null && errorMsgMap.length==0){
			xmlStatusStBuffer = xmlStatusStBuffer.append("<UpdStatus>success</UpdStatus>");
			xmlStatusStBuffer = xmlStatusStBuffer.append("<Error No='1'>");
				xmlStatusStBuffer = xmlStatusStBuffer.append("<Code></Code>");
				xmlStatusStBuffer = xmlStatusStBuffer.append("<Message></Message>");
			xmlStatusStBuffer = xmlStatusStBuffer.append("</Error>");
		}else{
			xmlStatusStBuffer = xmlStatusStBuffer.append("<UpdStatus>error</UpdStatus>");
			for(int errorIndividualCnt=0 ; errorIndividualCnt<errorMsgMap.length ; errorIndividualCnt++){

				xmlStatusStBuffer = xmlStatusStBuffer.append("<Error No='" + errorIndividualCnt+1 + "'>");
					xmlStatusStBuffer = xmlStatusStBuffer.append("<Code></Code>");
					xmlStatusStBuffer = xmlStatusStBuffer.append("<Message>" + errorMsgMap[errorIndividualCnt] + "</Message>");
				xmlStatusStBuffer = xmlStatusStBuffer.append("</Error>");
			}
		}
		xmlStatusStBuffer = xmlStatusStBuffer.append("</ResultStatus>");

		return xmlStatusStBuffer.toString();

	}

	/**
	 * 返却フォーマットのResult部分を作成
	 *
	 * @param individualMap データ毎に分割した文字配列（String[0]=[ID:VALUE,ID:VALUE],String[1]=[ID:VALUE,ID:VALUE]・・・）
	 * @return 「<id>VALUE</id>」形式のXML文字列
	 */
	private String makeResult(String[] individualMap) {

		// JSONのデリミタ文字列定義。あれば削除、無ければ何もしない制御。
		final String JSON_DELIMITER =  ",\"";
		String[] jsonMap = null;
		StringBuffer xmlDataStBuffer = new StringBuffer();

		for(int individualCnt=0 ; individualCnt<individualMap.length ; individualCnt++){

			// デリミタ「,」で分割すると、VALUEにも「,」が含まれている場合があるので、それだけで分割するのは不可能。
			// よって「ID:VALUE , ID:VALUE , ・・・」文字列のIDは「"」で囲まれており、VALUEは「"」で囲まれているものとそうでないものが存在する。
			// この事からデリミタ「,"」で分割すればVALUEに「,」が入っていた場合、分割されるのを防ぐ事ができる。
			// その代りIDの初めの「"」も消えているので注意が必要。
			jsonMap = individualMap[individualCnt].split(JSON_DELIMITER);

			// 「ID:VALUE」配列からXMLを作成する
			xmlDataStBuffer.append(makeXmlFormat(individualCnt+1, jsonMap));
		}

		return xmlDataStBuffer.toString();
	}

	/**
	 * 「[{ID:VALUE,ID:VALUE},{ID:VALUE,ID:VALUE},{ID:VALUE,・・・}]」文字列を受け取り、
	 * [{ }] でくくられたデータ毎に分割した文字配列にする
	 *
	 * @param idValueMap 「[{ID:VALUE,ID:VALUE},{ID:VALUE,ID:VALUE},{ID:VALUE,・・・}]」文字列
	 * @return [{ }] で分割した文字配列。（String[0]=[ID:VALUE,ID:VALUE],String[1]=[ID:VALUE,ID:VALUE]・・・）。  [{ }] で囲まれていない場合はnull
	 */
	private String[] deleteBracketArray(String bracketSt) {

		// JSONの開始・終了・区切り文字列定義。あれば削除、無ければ何もしない制御。
		final String JSON_BEGIN =  "[{";
		final String JSON_END =  "}]";
		final String JSON_BETWEEN =  "\\},\\{";
		String[] bracketMap = null;

		// JSON定義の [{ }] で囲まれた文字列の場合は [{ }] を削除する
		if(bracketSt.indexOf(JSON_BEGIN)==0 && bracketSt.length()>=JSON_END.length() && bracketSt.lastIndexOf(JSON_END)==bracketSt.length()-JSON_END.length()){
			bracketSt = bracketSt.substring(JSON_BEGIN.length());
			bracketSt = bracketSt.substring(0,bracketSt.length()-JSON_END.length());

			// １（グループ）データずつ分割する
			// （[ },{ ] で１グループが分けられているので、その部分で分割）
			bracketMap = bracketSt.split(JSON_BETWEEN);

		}

		return bracketMap;

	}

	/**
	 * 「ID:VALUE」配列からXMLを作成する
	 *
	 * @param dataNum データの数
	 * @param idValueMap 「ID:VALUE」配列
	 * @return 「<id>VALUE</id>」形式のXML文字列
	 */
	private String makeXmlFormat(int dataNum, String[] idValueMap) {

		// JSONのキーとデータの区切り文字列定義
		final String KEY_DATA_SPLIT =  ":";
		// JSONのキーとデータの引用符定義
		final String QUOTATION_MARKS =  "\"";
		// 全て（IDとVALUE）をCADATAで囲む（ID用）（対象：指定したものとIDが部分一致するもの）
		final String[] ID_INDEXOF_CDATA_ALL = {"<",">","&","'","\"","?","!","#","$","%","(",")",";",":","[","]","/"};
		// 全て（IDとVALUE）をCADATAで囲む（ID用）（対象：指定したものとIDが前方一致するもの）
		final String[] ID_STARTSWITH_CDATA_ALL = {"1","2","3","4","5","6","7","8","9","0"};
		// XML形式のエスケープ対象文字（VALUE用）（対象：指定したものとIDが部分一致するもの）
		final String[] VALUE_INDEXOF_CDATA_VALUE = {"<","&"};

		StringBuffer returnStBuffer = new StringBuffer();
		StringBuffer xmlId = new StringBuffer();
		StringBuffer xmlValue = new StringBuffer();

		// XML形式のエスケープ文字処理ループのカウンター
		int idNum=0;
		int idNum2=0;
		int valueNum=0;

		returnStBuffer.append("<Result No='"+dataNum+"'>");
		// 引数で取得した配列全てのデータを処理するループ
		for(int mapIndex=0 ; mapIndex<idValueMap.length ; mapIndex++){

			// キーとデータの区切り文字列がある場合
			if(idValueMap[mapIndex].indexOf(KEY_DATA_SPLIT) != -1){

				// キーとデータを分離する
				xmlId = new StringBuffer(idValueMap[mapIndex].substring(0, idValueMap[mapIndex].indexOf(KEY_DATA_SPLIT)).trim());
				xmlValue = new StringBuffer(idValueMap[mapIndex].substring(idValueMap[mapIndex].indexOf(KEY_DATA_SPLIT) + KEY_DATA_SPLIT.length()));
				// キーとデータに引用符がついている場合は削除する
				if(xmlId.indexOf(QUOTATION_MARKS)==0 && xmlId.length()>=QUOTATION_MARKS.length() && xmlId.lastIndexOf(QUOTATION_MARKS)==xmlId.length()-QUOTATION_MARKS.length()){
					xmlId = new StringBuffer(xmlId.toString().substring(QUOTATION_MARKS.length()));
					xmlId = new StringBuffer(xmlId.substring(0,xmlId.length()-QUOTATION_MARKS.length()));
				}else if(xmlId.length()>=QUOTATION_MARKS.length() && xmlId.lastIndexOf(QUOTATION_MARKS)==xmlId.length()-QUOTATION_MARKS.length()){
					xmlId = new StringBuffer(xmlId.substring(0,xmlId.length()-QUOTATION_MARKS.length()));
				}

				if(xmlValue.indexOf(QUOTATION_MARKS)==0 && xmlValue.length()>=QUOTATION_MARKS.length() && xmlValue.lastIndexOf(QUOTATION_MARKS)==xmlValue.length()-QUOTATION_MARKS.length()){
					xmlValue = new StringBuffer(xmlValue.toString().substring(QUOTATION_MARKS.length()));
					xmlValue = new StringBuffer(xmlValue.substring(0,xmlValue.length()-QUOTATION_MARKS.length()));
				}

				// XML形式の(ID)エスケープ対象文字の数だけループ
				for(idNum=0 ; idNum<ID_INDEXOF_CDATA_ALL.length ; idNum++){
					// 全て（IDとVALUE）をCADATAで囲む（対象：指定したエスケープ対象文字とIDが部分一致するもの）
					if(xmlId!=null && (xmlId.toString()).indexOf(ID_INDEXOF_CDATA_ALL[idNum]) != -1){
						// XMLデータ形式に整形
						returnStBuffer.append("<![CDATA[<" + xmlId + ">" + xmlValue + "</" + xmlId + ">]]>");
						break;
					}
				}
				// 上記の条件と合致しない場合
				if(idNum==ID_INDEXOF_CDATA_ALL.length){
					for(idNum2=0 ; idNum2<ID_STARTSWITH_CDATA_ALL.length ; idNum2++){
						// 全て（IDとVALUE）をCADATAで囲む（対象：指定したエスケープ対象文字とIDが前方一致するもの）
						if(xmlId!=null && (xmlId.toString()).startsWith(ID_STARTSWITH_CDATA_ALL[idNum2])){
							// XMLデータ形式に整形
							returnStBuffer.append("<![CDATA[<" + xmlId + ">" + xmlValue + "</" + xmlId + ">]]>");
							break;
						}
					}
					// 上記の条件と合致しない場合
					if(idNum2==ID_STARTSWITH_CDATA_ALL.length){
						// XML形式の(VALUE)エスケープ対象文字の数だけループ
						for(valueNum=0 ; valueNum<VALUE_INDEXOF_CDATA_VALUE.length ; valueNum++){
							// VALUEをCADATAで囲む（対象：指定したエスケープ対象文字とVALUEが部分一致するもの）
							if(xmlValue!=null && !("").equals(xmlValue.toString()) && (xmlValue.toString()).indexOf(VALUE_INDEXOF_CDATA_VALUE[valueNum]) != -1){
								// XMLデータ形式に整形
								returnStBuffer.append("<" + xmlId + "><![CDATA[" + xmlValue + "]]></" + xmlId + ">");
								break;
							}
						}
						// 上記の条件と合致しない場合
						if(valueNum==VALUE_INDEXOF_CDATA_VALUE.length){
							// XMLデータ形式に整形
							returnStBuffer.append("<" + xmlId + ">" + xmlValue + "</" + xmlId + ">");
						}
					}
				}
			}
		}
		returnStBuffer.append("</Result>");

		return returnStBuffer.toString();
	}
}
