package jp.ill.photon.module.order;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.model.AoSetting;
import jp.ill.photon.model.CartItem;
import jp.ill.photon.model.Item;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.module.cart.CartInModule;
import jp.ill.photon.util.AoUtil;
import jp.ill.photon.util.UtilTools;

import org.apache.commons.lang.StringUtils;
import org.seasar.doma.jdbc.tx.TransactionManager;

public class OrderUpdatePreModule implements PhotonModule {

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "param", transferVal = "order_no")
    private String orderNo;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "param_multi", transferVal = "record_no")
    private String[] recordNo;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "param_multi", transferVal = "del_record_no")
    private String[] delRecordNo;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "param_multi", transferVal = "item_cd")
    private String[] itemCd;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "param_multi", transferVal = "order.quantity")
    private String[] orderQuantity;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "param_multi", transferVal = "order_case_quantity")
    private String[] orderCaseQuantity;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "got_session.userForm.user_cd")
    private String userCd;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "got_session_charge.userChargeForm.user_charge_id")
    private int userChargeId;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType="dto", transferVal="common.aladdinsetting")
    private AoSetting aoSetting;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.caseNumDiv.note")
    private String caseNumDiv;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.serviceChargeCd.note")
    private String chargeFeeItemCd;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.deliveryFeeItemCd.note")
    private String deliveryFeeItemCd;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.webOrderCampaignItemCd.note")
    private String webOrderCampaignCd;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "user.first")
    private Map<String, Object> userForm;

    @ModuleParam(required = true)
    private List<Map<String, Object>> itemList;

    public static final String ERROR_MSG_INVALID_ITEM_ID = "inputErrMesItemNotExists";
    private static String ERR_NUMBER_INVALID = "number invalid";

	/** メッセージID（入力値無効） */
	public static final String INPUT_ERR_NUMBER = "inputErrMesCartNumber";

    public List<Map<String, Object>> getItemList() {
        return itemList;
    }

    public void setItemList(List<Map<String, Object>> itemList) {
        this.itemList = itemList;
    }

    public Map<String, Object> getUserForm() {
        return userForm;
    }

    public void setUserForm(Map<String, Object> userForm) {
        this.userForm = userForm;
    }

    public String getChargeFeeItemCd() {
        return chargeFeeItemCd;
    }

    public void setChargeFeeItemCd(String chargeFeeItemCd) {
        this.chargeFeeItemCd = chargeFeeItemCd;
    }

    public String getDeliveryFeeItemCd() {
        return deliveryFeeItemCd;
    }

    public void setDeliveryFeeItemCd(String deliveryFeeItemCd) {
        this.deliveryFeeItemCd = deliveryFeeItemCd;
    }

    public String getWebOrderCampaignCd() {
        return webOrderCampaignCd;
    }

    public void setWebOrderCampaignCd(String webOrderCampaignCd) {
        this.webOrderCampaignCd = webOrderCampaignCd;
    }

    public String[] getDelRecordNo() {
        return delRecordNo;
    }

    public void setDelRecordNo(String[] delRecordNo) {
        this.delRecordNo = delRecordNo;
    }

    public String getOrderNo() {
        return orderNo;
    }

    public void setOrderNo(String orderNo) {
        this.orderNo = orderNo;
    }

    public String[] getRecordNo() {
        return recordNo;
    }

    public void setRecordNo(String[] recordNo) {
        this.recordNo = recordNo;
    }

    public String[] getItemCd() {
        return itemCd;
    }

    public void setItemCd(String[] itemCd) {
        this.itemCd = itemCd;
    }

    public String[] getOrderQuantity() {
        return orderQuantity;
    }

    public void setOrderQuantity(String[] orderQuantity) {
        this.orderQuantity = orderQuantity;
    }

    public String[] getOrderCaseQuantity() {
        return orderCaseQuantity;
    }

    public void setOrderCaseQuantity(String[] orderCaseQuantity) {
        this.orderCaseQuantity = orderCaseQuantity;
    }

    public String getUserCd() {
        return userCd;
    }

    public void setUserCd(String userCd) {
        this.userCd = userCd;
    }

    public int getUserChargeId() {
        return userChargeId;
    }

    public void setUserChargeId(int userChargeId) {
        this.userChargeId = userChargeId;
    }

    public AoSetting getAoSetting() {
        return aoSetting;
    }

    public void setAoSetting(AoSetting aoSetting) {
        this.aoSetting = aoSetting;
    }

    public String getCaseNumDiv() {
        return caseNumDiv;
    }

    public void setCaseNumDiv(String caseNumDiv) {
        this.caseNumDiv = caseNumDiv;
    }

    @Override
    public ModuleResult execute(ModuleContext context) throws PhotonModuleException {
        ModuleResult result = new ModuleResult();

        List<String> errorMessage = new ArrayList<String>();

        TransactionManager tm = DomaConfig.singleton().getTransactionManager();
        JsonDataDao dao = new JsonDataDaoImpl();

        List<Map<String, Object>> cartList = new ArrayList<Map<String, Object>>();

        String totalRetailSalePrice = "0";
        String totalWholeSalePrice = "0";
        String totalQuantity = "0";

        List<ActionDtoMessage> tmpErrorMessage = new ArrayList<ActionDtoMessage>();
		Map<String, Object> retrunMessageListMap = new HashMap<String, Object>();
		Map<String, List<String>> errorMessageListMap = new HashMap<String, List<String>>();
		Map<String, String> errorMessageMap = new HashMap<String, String>();

        if (userForm == null) {
            return result;
        }

        Arrays.sort(delRecordNo);

        if (itemCd.length != 0) {

	        for (int i=0; i < recordNo.length; i++) {

	            String itemCdStr = itemCd[i];
	            String recordNoStr = recordNo[i];

	            int search = Arrays.binarySearch(delRecordNo, recordNoStr);

	            if (search < 0) {

	                if (OrderHisListModule.isSpecialDetail(itemCdStr, chargeFeeItemCd, deliveryFeeItemCd, webOrderCampaignCd)) {
	                    continue;
	                }

	                String caseQuantityStr = orderCaseQuantity[i];
	                String quantityStr = orderQuantity[i];

	                Item item = getItemData(itemCdStr);

	                if (item == null) {
	                	// エラー・メッセージを積む
	                	errorMessageMap.put(itemCdStr, ERROR_MSG_INVALID_ITEM_ID);
						addErrorMessage(errorMessageListMap, itemCdStr, errorMessage, result, tmpErrorMessage, ERROR_MSG_INVALID_ITEM_ID);
						continue;
	                }

	                // 入力フォーマットチェック
	                boolean[] checkResult = CartInModule.checkCartInputFormat(item, quantityStr, caseQuantityStr, caseNumDiv, aoSetting, errorMessage, userForm);

	                if (errorMessage.size() > 0) {

	                	if (ERR_NUMBER_INVALID.equals(errorMessage.get(0))) {
							errorMessageMap.put(itemCdStr, INPUT_ERR_NUMBER);
							addErrorMessage(errorMessageListMap, itemCdStr, errorMessage, result, tmpErrorMessage, INPUT_ERR_NUMBER);
							continue;
						} else {
							// エラー・メッセージを積む
		                	errorMessageMap.put(itemCdStr, ERROR_MSG_INVALID_ITEM_ID);
							addErrorMessage(errorMessageListMap, itemCdStr, errorMessage, result, tmpErrorMessage, ERROR_MSG_INVALID_ITEM_ID);
							continue;
						}
	                }

	                boolean orderQuantityIsBlank = checkResult[0];
	                boolean orderCaseQuantityIsBlank = checkResult[1];

	                // TODO 単価は注文時のものを使用したほうがいいかも。
	                CartItem cartItem = null;

	                if (!orderCaseQuantityIsBlank) {
	                    // カート更新（ケース）
	                    cartItem = makeCartData(context, item, quantityStr, caseQuantityStr, orderNo, recordNoStr);
	                }
	                else if (!orderQuantityIsBlank) {
	                    // カート更新（バラ）
	                    cartItem = makeCartData(context, item, quantityStr, null, orderNo, recordNoStr);
	                }

	                if (cartItem != null) {
	                    BigDecimal retailSalesPrice = cartItem.getRetailSalesPriceFormat(aoSetting, userForm);
	                    BigDecimal subTotalPrice = cartItem.getSubTotalPriceFormat(aoSetting, userForm);

	                    totalRetailSalePrice = UtilTools.addBigDecimal(totalRetailSalePrice, retailSalesPrice.toString());
	                    totalWholeSalePrice = UtilTools.addBigDecimal(totalWholeSalePrice, subTotalPrice.toString());
	                    totalQuantity = UtilTools.addBigDecimal(totalQuantity, cartItem.getOrderQuantity().toString());

	                    // cartList.add(cartItem.toMap());
	                    cartList.add(cartItem.toMap(aoSetting, userForm));
	                }
	            }
	        }
        }

        retrunMessageListMap.put("error_field", tmpErrorMessage);
		result.getReturnData().put("messages", errorMessageListMap);
		result.getReturnData().put("err_messages", errorMessageMap); // 暫定
		result.setMessages(retrunMessageListMap);

        result.getReturnData().put("list", cartList);
        result.getReturnData().put("total_retail_sale_price", totalRetailSalePrice);
        result.getReturnData().put("total_whole_sale_price", totalWholeSalePrice);
        result.getReturnData().put("total_quantity", AoUtil.convAoRoundingSalesQuantity(totalQuantity,
                aoSetting.getSalesQuantityDigitNum(), aoSetting.getSalesQuantityDecimalPointNum(), aoSetting.getSalesQuantityRoundingDiv()));

        return result;
    }

    private void addErrorMessage(Map<String, List<String>> errorMessageListMap,
			String itemCdStr, List<String> errorMessage, ModuleResult result, List<ActionDtoMessage> tmpErrorMessage, String errorMsg) {
		errorMessageListMap.put(itemCdStr, errorMessage);
		result.setResultCode("error");
		result.getReturnData().put("result_code", "error");

		ActionDtoMessage dtoMessage = new ActionDtoMessage();
		dtoMessage.setFormFieldCd(itemCdStr);
		dtoMessage.setMessageId(errorMsg);
		tmpErrorMessage.add(dtoMessage);
	}

    private CartItem makeCartData(ModuleContext context, Item item, String quantityStr, String caseQuantityStr, String orderNo, String recordNo) {
        boolean caseFlg = false;
        if (!StringUtils.isEmpty(caseQuantityStr)) {
            caseFlg = true;
        }

        CartItem cart = new CartItem();
        cart.setItemCd(item.getItemCd());
        cart.setUserChargeId(String.valueOf(userChargeId));
        cart.setUserCd(userCd);

        cart.setOrderQuantity(new BigDecimal(quantityStr));
        if (caseFlg) {
            cart.setOrderCaseQuantity(new BigDecimal(caseQuantityStr));
        } else {
            cart.setOrderCaseQuantity(new BigDecimal(0));
        }

        // この辺りはセットしなくなるはず
        cart.setWholesalePrice(item.getWholeSalePrice());
        cart.setRetailPrice(new BigDecimal(0)); // 暫定

        cart.setItemName1(item.getItemName1());
        cart.setItemName2(item.getItemName2());
        cart.setQuantityPerCase(item.getQuantityPerCase());

        if (item.getUnit() != null) {
            cart.setUnit(item.getUnit().toString());
        }

        cart.setOrderNo(orderNo);
        cart.setRecordNo(recordNo);

        return cart;
    }

    /**
     * itemListを設定します。
     * @param itemList itemList
     */
    public Item getItemData(String itemCd) {
        Item rtnItem = null;
        for (Map<String, Object> itemData : itemList) {
            if (!itemCd.equals(itemData.get("item_cd"))) {
                continue;
            }
            rtnItem = new Item();
            rtnItem.setItemCd(String.valueOf(itemData.get("item_cd")));
            rtnItem.setWholeSalePrice(new BigDecimal(String.valueOf(itemData.get("price") == null ? "0" : itemData.get("price"))));
            rtnItem.setItemDispFlg(1); // 取得SQLの条件句がdisp_flg = 1 のため1固定でセット
            //rtnItem.setCartDispFlg(Integer.valueOf(String.valueOf(itemData.get("cart_disp_flg") == null ? "0" : itemData.get("cart_disp_flg"))));
            rtnItem.setCartDispFlg(1); // cartDispFlgも現在のマスタ値にかかわらず、有効で上書きする。 // 2017/08/30
            rtnItem.setQuantityPerCase(
                    new BigDecimal(String.valueOf(itemData.get("quantity_per_case") == null ? "0" : itemData.get("quantity_per_case"))));
            rtnItem.setItemName1(String.valueOf(itemData.get("item_name1")));
            rtnItem.setItemName2(String.valueOf(itemData.get("item_name2")));
            rtnItem.setUnit(String.valueOf(itemData.get("unit")));
            break;
        }

        return rtnItem;
    }


}
