package jp.ill.photon.module.order;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.model.AoSetting;
import jp.ill.photon.model.CartItem;
import jp.ill.photon.model.Item;
import jp.ill.photon.model.OrderDetail;
import jp.ill.photon.model.OrderSummary;
import jp.ill.photon.model.User;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.StringUtil;

public class OrderUpdateModule implements PhotonModule {

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "param", transferVal = "order_no")
	private String orderNo;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "param_multi", transferVal = "del_record_no")
	private String[] delRecordNo;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "got_session.userForm.user_cd")
	private String userCd;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "got_session_charge.userChargeForm.user_charge_id")
	private int userChargeId;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "cart.list")
	private List<Map<String, Object>> cartList;

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "dto", transferVal = "tax.total_tax_price")
	private String totalTaxPrice;

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "dto", transferVal = "tax.total_tax_price_for_sum")
	private String totalTaxPriceForSum;

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "dto", transferVal = "tax.delivery_fee")
	private Map<String, Object> deliveryFee;

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "dto", transferVal = "tax.charge_fee")
	private Map<String, Object> chargeFee;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.caseNumDiv.note")
	private String caseNumDiv;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.serviceChargeCd.note")
	private String chargeFeeItemCd;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.deliveryFeeItemCd.note")
	private String deliveryFeeItemCd;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.webOrderCampaignItemCd.note")
	private String webOrderCampaignCd;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.aladdinsetting")
	private AoSetting aoSetting;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "user.first")
	private Map<String, Object> userForm;

	/** メッセージID（注文削除メッセージ） */
	public static final String CHUMON_DELETE_MSG_ID = "DelOrder2";

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "param", transferVal = "importPermitDiv")
	private String importPermitDiv;

	public Map<String, Object> getUserForm() {
		return userForm;
	}

	public void setUserForm(Map<String, Object> userForm) {
		this.userForm = userForm;
	}

	public String[] getDelRecordNo() {
		return delRecordNo;
	}

	public void setDelRecordNo(String[] delRecordNo) {
		this.delRecordNo = delRecordNo;
	}

	public AoSetting getAoSetting() {
		return aoSetting;
	}

	public void setAoSetting(AoSetting aoSetting) {
		this.aoSetting = aoSetting;
	}

	public String getOrderNo() {
		return orderNo;
	}

	public void setOrderNo(String orderNo) {
		this.orderNo = orderNo;
	}

	public String getChargeFeeItemCd() {
		return chargeFeeItemCd;
	}

	public void setChargeFeeItemCd(String chargeFeeItemCd) {
		this.chargeFeeItemCd = chargeFeeItemCd;
	}

	public String getDeliveryFeeItemCd() {
		return deliveryFeeItemCd;
	}

	public void setDeliveryFeeItemCd(String deliveryFeeItemCd) {
		this.deliveryFeeItemCd = deliveryFeeItemCd;
	}

	public String getWebOrderCampaignCd() {
		return webOrderCampaignCd;
	}

	public void setWebOrderCampaignCd(String webOrderCampaignCd) {
		this.webOrderCampaignCd = webOrderCampaignCd;
	}

	public String getCaseNumDiv() {
		return caseNumDiv;
	}

	public void setCaseNumDiv(String caseNumDiv) {
		this.caseNumDiv = caseNumDiv;
	}

	public String getUserCd() {
		return userCd;
	}

	public void setUserCd(String userCd) {
		this.userCd = userCd;
	}

	public int getUserChargeId() {
		return userChargeId;
	}

	public void setUserChargeId(int userChargeId) {
		this.userChargeId = userChargeId;
	}

	public List<Map<String, Object>> getCartList() {
		return cartList;
	}

	public void setCartList(List<Map<String, Object>> cartList) {
		this.cartList = cartList;
	}

	public String getTotalTaxPrice() {
		return totalTaxPrice;
	}

	public void setTotalTaxPrice(String totalTaxPrice) {
		this.totalTaxPrice = totalTaxPrice;
	}

	public String getTotalTaxPriceForSum() {
		return totalTaxPriceForSum;
	}

	public void setTotalTaxPriceForSum(String totalTaxPriceForSum) {
		this.totalTaxPriceForSum = totalTaxPriceForSum;
	}

	public Map<String, Object> getDeliveryFee() {
		return deliveryFee;
	}

	public void setDeliveryFee(Map<String, Object> deliveryFee) {
		this.deliveryFee = deliveryFee;
	}

	public Map<String, Object> getChargeFee() {
		return chargeFee;
	}

	public void setChargeFee(Map<String, Object> chargeFee) {
		this.chargeFee = chargeFee;
	}

	public String getImportPermitDiv() {
		return importPermitDiv;
	}

	public void setImportPermitDiv(String importPermitDiv) {
		this.importPermitDiv = importPermitDiv;
	}

	private boolean isCartListEmpty() {
		if (cartList.isEmpty()) {
			return true;
		}

		boolean result = true;
		for (Map<String, Object> paramCart : cartList) {
			if (paramCart.get("item_cd") != null) {
				String itemCd = String.valueOf(paramCart.get("item_cd"));

				if (!itemCd.equals(chargeFeeItemCd)
						|| !itemCd.equals(deliveryFeeItemCd)
						|| !itemCd.equals(webOrderCampaignCd)) {
					result = false;
				}
			}
		}

		return result;
	}

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		ModuleResult result = new ModuleResult();
		result.getReturnData().put("order_deleted", "");

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		JsonDataDao dao = new JsonDataDaoImpl();

		// TODO トランザクションに入れたほうがいいか？
		User user = User.getUser(context.getDto(), userCd);

		int r = tm.required(() -> {

			// 削除する明細を削除
			for (String delRecordNoStr : delRecordNo) {
				dao.deleteOrderDetailByRecordNo(context.getDto().getTenantId(),
						orderNo, delRecordNoStr);
			}

			// 先にキャンペーン、配送料、手数料の明細を削除
			dao.deleteOrderSpecialDetail(context.getDto().getTenantId(), userCd,
					orderNo, chargeFeeItemCd, deliveryFeeItemCd,
					webOrderCampaignCd);

			if (isCartListEmpty()) {
				// 明細全件削除時は、見出から削除する
				dao.deleteOrder(context.getDto().getTenantId(), orderNo);
				result.setResultCode("delete");

				List<ActionDtoMessage> errorMessages = new ArrayList<ActionDtoMessage>();
				ActionDtoMessage msg = new ActionDtoMessage();
				msg.setMessageId(CHUMON_DELETE_MSG_ID);
				errorMessages.add(msg);
				result.getMessages().put("info_page", errorMessages);
				result.getReturnData().put("order_deleted", "1");

				return 0;
			}

			OrderSummary summary = new OrderSummary();

			int maxRecordNo = 0;
			CartItem campaignCartItem = null;
			Item campaignItem = null;

			List<OrderDetail> specialDetails = new ArrayList<OrderDetail>();

			for (Map<String, Object> paramCart : cartList) {
				CartItem cart = CartItem.valueOf(paramCart);

				Item item = summary.sumOrderDetail(context, dao, cart, userCd,
						aoSetting, null, null, userForm, webOrderCampaignCd.equals(cart.getItemCd()));

				BigDecimal taxPrice = (cart.getTaxPrice() == null)
						? BigDecimal.ZERO : cart.getTaxPrice();
				Double taxRate = (cart.getTaxRate() == null) ? 0.0
						: cart.getTaxRate();

				if (webOrderCampaignCd != null
						&& !webOrderCampaignCd.equals(cart.getItemCd())) {
					maxRecordNo = Math.max(maxRecordNo,
							Integer.valueOf(cart.getRecordNo()));

					// 明細を更新
					Map<String, Object> map = new HashMap<String, Object>();
					map.put("item_cd", cart.getItemCd());
					map.put("order_quantity", StringUtil.defaultString(CartItem.getOrderQuantityFormat(cart.getOrderQuantity(), aoSetting), "0.0"));
					map.put("order_case_quantity", StringUtil.defaultString(CartItem.getOrderCaseQuantityFormat(cart.getOrderCaseQuantity(), aoSetting), "0.0") );
					map.put("shipment_quantity", StringUtil.defaultString(CartItem.getOrderQuantityFormat( new BigDecimal(0.0), aoSetting), "0.0"));
					map.put("shipment_case_quantity", StringUtil.defaultString(CartItem.getOrderCaseQuantityFormat( new BigDecimal(0.0), aoSetting), "0.0"));
					map.put("unshipped_quantity", StringUtil.defaultString(CartItem.getOrderQuantityFormat(cart.getOrderQuantity(), aoSetting), "0.0"));
					map.put("unshipped_case_quantity", StringUtil.defaultString(CartItem.getOrderCaseQuantityFormat(cart.getOrderCaseQuantity(), aoSetting), "0.0"));
					map.put("quantity_per_case", StringUtil.defaultString(item.getQuantityPerCase(), "0.0"));
					map.put("wholesale_price", StringUtil.defaultString(CartItem.getWholeSalePriceFormat(aoSetting, cart.getWholesalePrice(), userForm), "0.0"));
					map.put("tax_price", StringUtil.defaultString(StringUtil.toBigDecimal(String.valueOf(taxPrice), new BigDecimal(0.0)), "0.0"));
					map.put("tax_rate", taxRate);
					map.put("tax_calc_div", user.getTaxCalcDiv());
					map.put("tax_fraction_digit_div",
							user.getTaxFractionDigitDiv());
					map.put("tax_fraction_div", user.getTaxFractionDiv());
					map.put("tax_exempt_div", item.getTaxExemptDiv());
					map.put("tax_bracket_cd", item.getTaxBracketCd());
					map.put("order_no", cart.getOrderNo());
					map.put("record_no", cart.getRecordNo());

					String data = JsonUtil.mapToJson(map);
					dao.updateEditTableData(context.getDto().getTenantId(),
							"orderdetail", "system", data);

					// dao.updateOrderDetail(context.getDto().getTenantId(),
					// userCd, cart.getOrderNo(),
					// Integer.valueOf(cart.getRecordNo()), cart.getItemCd(),
					// cart.getOrderQuantity(), cart.getOrderCaseQuantity(),
					// item.getQuantityPerCase(),
					// cart.getWholesalePrice(), taxPrice, taxRate,
					// user.getTaxCalcDiv(),
					// user.getTaxFractionDigitDiv(), user.getTaxFractionDiv(),
					// item.getTaxExemptDiv(), item.getTaxBracketCd());
				}

				if (webOrderCampaignCd != null
						&& webOrderCampaignCd.equals(cart.getItemCd())) {
					campaignCartItem = cart;
					campaignItem = item;
				}
			}

			// キャンペーン
			if (campaignCartItem != null) {
				OrderDetail detail = new OrderDetail(orderNo, ++maxRecordNo,
						campaignCartItem, campaignItem, user, aoSetting,
						userForm);
				detail.setWebPriceUseDiv(1);
				specialDetails.add(detail);
			}

			// 配送料
			if (deliveryFee != null) {
				CartItem deliveryFeeItem = CartItem.valueOf(deliveryFee);
				Item item = summary.sumOrderDetail(context, dao,
						deliveryFeeItem, userCd, aoSetting, null, null,
						userForm, true);
				OrderDetail detail = new OrderDetail(orderNo, ++maxRecordNo,
						deliveryFeeItem, item, user, aoSetting, userForm);
				detail.setWebPriceUseDiv(1);
				specialDetails.add(detail);
			}

			// 手数料
			if (chargeFee != null) {
				CartItem chargeFeeItem = CartItem.valueOf(chargeFee);
				Item item = summary.sumOrderDetail(context, dao, chargeFeeItem,
						userCd, aoSetting, null, null, userForm, true);
				OrderDetail detail = new OrderDetail(orderNo, ++maxRecordNo,
						chargeFeeItem, item, user, aoSetting, userForm);
				detail.setWebPriceUseDiv(1);
				specialDetails.add(detail);
			}

			// 特殊明細登録
			for (OrderDetail detail : specialDetails) {
				try {
					dao.insertEditTableData(context.getDto().getTenantId(),
							"orderdetail", user.getUserCd(),
							JsonUtil.mapToJsonSnakeCaseKey(detail));
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}

			// 受注見出し更新
			Map<String, Object> map = new HashMap<String, Object>();
			map.put("tax_price", new BigDecimal(totalTaxPrice));
			map.put("wholesale_price",
					new BigDecimal(summary.getSumWholesalePrice()));
			map.put("retail_price",
					new BigDecimal(summary.getSumRetailPrice()));
			map.put("total_order_quantity",
					CartItem.getOrderQuantityFormat( new BigDecimal(summary.getTotalQuantity()), aoSetting) );
			map.put("total_shipment_quantity", CartItem.getOrderQuantityFormat( new BigDecimal(0.0), aoSetting) );
			map.put("total_unshipped_quantity",
					CartItem.getOrderQuantityFormat( new BigDecimal(summary.getTotalQuantity()), aoSetting) );
			map.put("order_no", orderNo);

			if (importPermitDiv != null && importPermitDiv != "") {
				map.put("import_permit_div", importPermitDiv);
			}

			String data = JsonUtil.mapToJson(map);
			dao.updateEditTableData(context.getDto().getTenantId(), "order",
					"system", data);

			// dao.updateOrder(context.getDto().getTenantId(), userCd, orderNo,
			// importPermitDiv, new BigDecimal(totalTaxPrice),
			// new BigDecimal(summary.getSumWholesalePrice()), new
			// BigDecimal(summary.getSumRetailPrice()), new
			// BigDecimal(summary.getTotalQuantity()));

			result.setResultCode("success");
			return 0;
		});

		return result;
	}

}
