package jp.ill.photon.module.order;

import java.math.BigDecimal;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.model.AoSetting;
import jp.ill.photon.model.CartItem;
import jp.ill.photon.model.OrderCampaign;
import jp.ill.photon.model.Tax;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.module.tax.CalcTaxModule;
import jp.ill.photon.util.AoUtil;
import jp.ill.photon.util.DateUtil;
import jp.ill.photon.util.StringUtil;
import jp.ill.photon.util.UtilTools;

public class CalcWebCampaignModule implements PhotonModule {

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.webOrderCampaignItemCd.note")
	private String webOrderCampaignCd;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "cart.total_whole_sale_price")
	private String totalWholeSalePrice;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.aladdinsetting")
	private AoSetting aoSetting;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "cart.list")
	private List<Map<String, Object>> cartList;

	@ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "user.first")
    private Map<String, Object> userForm;

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "dto", transferVal = "pre_update_order_data.list")
	private List<Map<String, Object>> orderData;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.verNo.note")
	private String verNo;

	public static final String ERROR_MSG_ID = "calcWebCampaignErrMes";

    public String getVerNo() {
		return verNo;
	}

	public void setVerNo(String verNo) {
		this.verNo = verNo;
	}

	public Map<String, Object> getUserForm() {
        return userForm;
    }

    public void setUserForm(Map<String, Object> userForm) {
        this.userForm = userForm;
    }

	public List<Map<String, Object>> getOrderData() {
		return orderData;
	}

	public void setOrderData(List<Map<String, Object>> orderData) {
		this.orderData = orderData;
	}

	public List<Map<String, Object>> getCartList() {
		return cartList;
	}

	public void setCartList(List<Map<String, Object>> cartList) {
		this.cartList = cartList;
	}

	public String getWebOrderCampaignCd() {
		return webOrderCampaignCd;
	}

	public void setWebOrderCampaignCd(String webOrderCampaignCd) {
		this.webOrderCampaignCd = webOrderCampaignCd;
	}

	public String getTotalWholeSalePrice() {
		return totalWholeSalePrice;
	}

	public void setTotalWholeSalePrice(String totalWholeSalePrice) {
		this.totalWholeSalePrice = totalWholeSalePrice;
	}

	public AoSetting getAoSetting() {
		return aoSetting;
	}

	public void setAoSetting(AoSetting aoSetting) {
		this.aoSetting = aoSetting;
	}

	@Override
	public ModuleResult execute(ModuleContext context) throws PhotonModuleException {
		ModuleResult result = new ModuleResult();

		boolean isCalc = true;
		if (webOrderCampaignCd == null) {
			isCalc = false;
		}

		TransactionManager tm = DomaConfig.singleton()
				.getTransactionManager();
		JsonDataDao dao = new JsonDataDaoImpl();

		// キャンペーン値引きが既に追加されていれば追加しない
		if (isCalc) {
			for (Map<String, Object> cartMap : cartList) {
				CartItem cart = CartItem.valueOf(cartMap);
				if (webOrderCampaignCd.equals(cart.getItemCd())) {
					isCalc = false;
				}
			}

		}

		if (isCalc) {

			OrderCampaign orderCampaign = null;
			if(orderData == null){
				// 受注データがない場合、”現在日時”が開始日時～終了日時の間にあるキャンペーンを取得
				orderCampaign = tm.required(() -> {
					List<Map<String, Object>> campaignMap = dao
							.getWebCampaignList(context.getDto().getTenantId());
					if (campaignMap == null || campaignMap.isEmpty()) {
						return null;
					} else {
						return OrderCampaign.valueOf(campaignMap.get(0));
					}
				});
			}else{
				// 受注データがある場合、”注文日時”が開始日時～終了日時の間にあるキャンペーンを取得
				orderCampaign = tm.required(() -> {
					Timestamp orderDate = null;
					if( orderData != null && orderData.size() > 0 ){
						orderDate = DateUtil.toSqlTimestamp((String)orderData.get(0).get("order_date"));
					}
					List<Map<String, Object>> campaignMap = dao
							.getWebCampaignListForOrderHisInfo(context.getDto().getTenantId(), orderDate);
					if (campaignMap == null || campaignMap.isEmpty()) {
						return null;
					} else {
						return OrderCampaign.valueOf(campaignMap.get(0));
					}
				});
			}

			String discountPrice = "0";
			if (orderCampaign != null) {

				List<Map<String, Object>> itemMap = dao.getItemForOrder(context.getDto().getTenantId(), webOrderCampaignCd, String.valueOf(userForm.get("user_cd")));
		        if (itemMap == null || itemMap.isEmpty()) {
		        	result.setResultCode("error");

				    List<ActionDtoMessage> errorMessages = new ArrayList<ActionDtoMessage>();
				    ActionDtoMessage msg = new ActionDtoMessage();
		            msg.setMessageId(ERROR_MSG_ID);
		            errorMessages.add(msg);
				    result.getMessages().put("error_page", errorMessages);
		        }

				Double taxRate = 1.0;
				if (Tax.TaxDiv.INCLUDING.equals(StringUtil.defaultString(userForm.get("tax_div"), ""))) {
					taxRate = CalcTaxModule.getTaxRateFromItemCd(context, webOrderCampaignCd, String.valueOf(userForm.get("user_cd")), verNo) + 1.0;
				}

				discountPrice = getDiscountPrice(orderCampaign, taxRate, StringUtil.defaultString(userForm.get("tax_div"), ""));

				// 内税の場合は消費税分を加算
//				if (Tax.TaxDiv.INCLUDING.equals(StringUtil.defaultString(userForm.get("tax_div"), ""))) {
//					discountPrice = UtilTools.multiplyBigDecimal(discountPrice, String.valueOf(taxRate));
//				}

				if (new BigDecimal(discountPrice)
						.compareTo(BigDecimal.ZERO) != 0) {
					discountPrice = AoUtil.convAoRoundingSalesPrice(
							discountPrice, AoUtil.DispPattern.PATTERN_NO_COMMA,
							aoSetting.getSalesPriceDigitNum(),
							StringUtil.defaultString(userForm.get("price_fraction_div"), "0") );
					discountPrice = UtilTools.multiplyBigDecimal(discountPrice,
							"-1");

					// 値引き明細の作成
		            CartItem webCampaignItem = new CartItem();
		            webCampaignItem.setItemCd(webOrderCampaignCd);
		            webCampaignItem.setOrderQuantity(new BigDecimal(1.0));
		            webCampaignItem.setOrderCaseQuantity(new BigDecimal(0.0));
		            webCampaignItem.setRetailPrice(new BigDecimal(0.0));
		            webCampaignItem.setWholesalePrice(new BigDecimal(discountPrice));
		            webCampaignItem.setQuantityPerCase(new BigDecimal(0.0));
		            webCampaignItem.setIsDisplay(false);
		            webCampaignItem.setCartId(-1); // 特殊明細は負のcartIdを設定

		            // キャンペーン値引きはカートリストに直接追加
		            cartList.add(webCampaignItem.toMap());
		            // 画面表示用に単体でもセット
		            result.getReturnData().put("web_campaign", webCampaignItem.toMap());
		            result.getReturnData().put("web_campaign_plane_data", orderCampaign.toMap());
				}
			}
		}

		// カート合計金額を再計算して再セット
		String totalRetailSalePrice = "0";
		String totalWholeSalePrice = "0";
		String totalQuantity = "0";

		for (int i = 0; i < cartList.size(); i++) {
			Map<String, Object> cartMap = cartList.get(i);
			CartItem cartItem = CartItem.valueOf(cartMap);

			BigDecimal retailSalesPrice = cartItem
					.getRetailSalesPriceFormat(aoSetting, userForm);
			BigDecimal subTotalPrice = cartItem
					.getSubTotalPriceFormat(aoSetting, userForm);

			totalRetailSalePrice = UtilTools.addBigDecimal(totalRetailSalePrice,
					retailSalesPrice.toString());
			totalWholeSalePrice = UtilTools.addBigDecimal(totalWholeSalePrice,
					subTotalPrice.toString());
			totalQuantity = UtilTools.addBigDecimal(totalQuantity,
					cartItem.getOrderQuantity().toString());
		}

		result.getReturnData().put("total_retail_sale_price",
				totalRetailSalePrice);
		result.getReturnData().put("total_whole_sale_price",
				totalWholeSalePrice);
		result.getReturnData().put("total_quantity",
				AoUtil.convAoRoundingSalesQuantity(totalQuantity,
						aoSetting.getSalesQuantityDigitNum(),
						aoSetting.getSalesQuantityDecimalPointNum(),
						aoSetting.getSalesQuantityRoundingDiv()));

		return result;
	}

	public String getDiscountPrice(OrderCampaign campaign, Double taxRate, String taxDiv) {
		// 値引金額
		String discountPrice = "0";
		BigDecimal bdTotalPrice = new BigDecimal(totalWholeSalePrice);

		if (DiscountDiv.RATE
				.equals(String.valueOf(campaign.getDiscountDiv()))) {
			// 割合値引
			if (campaign.getDiscountRate().compareTo(BigDecimal.ZERO) == 0) {
				discountPrice = "0";
			} else if (campaign.getDiscountAmountBorderIntax(taxRate, taxDiv)
					.compareTo(bdTotalPrice) > 0) {
				discountPrice = "0";
			} else {
				String discountRate = campaign.getDiscountRate()
						.multiply(new BigDecimal("0.01")).toString();
				discountPrice = UtilTools
						.multiplyBigDecimal(totalWholeSalePrice, discountRate);
			}
		} else if (DiscountDiv.FIX
				.equals(String.valueOf(campaign.getDiscountDiv()))) {
			// 固定値引
			discountPrice = String.valueOf(campaign.getFixedDiscountPrice());

			if (Tax.TaxDiv.INCLUDING.equals(taxDiv)) {
				discountPrice = UtilTools.multiplyBigDecimal(discountPrice, String.valueOf(taxRate));
	    	}

			if (new BigDecimal(discountPrice).compareTo(bdTotalPrice) > 0) {
				discountPrice = totalWholeSalePrice;
			}
			if (campaign.getDiscountAmountBorderIntax(taxRate, taxDiv)
					.compareTo(bdTotalPrice) > 0) {
				discountPrice = "0";
			}
		}
		return discountPrice;
	}

	// Aladdin-EC ３次開発_04 20141104 ADD-E
	// Aladdin-EC ３次開発_05 20141106 ADD-S
	/** 割引区分 */
	public static class DiscountDiv {
		/** 割合値引 */
		public static final String RATE = "1";
		/** 固定値引 */
		public static final String FIX = "2";
		/** マップ */
		public static final LinkedHashMap<String, String> DISCOUNT_DIV_MAP = new LinkedHashMap<String, String>();
		static {
			DISCOUNT_DIV_MAP.put(RATE, "割引率を指定");
			DISCOUNT_DIV_MAP.put(FIX, "割引金額を指定");
		}
	}

}
