package jp.ill.photon.module.order;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.model.AoSetting;
import jp.ill.photon.model.CartItem;
import jp.ill.photon.model.Delivery;
import jp.ill.photon.model.DeliveryFee;
import jp.ill.photon.model.Tax;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.module.tax.CalcTaxModule;
import jp.ill.photon.util.AoUtil;
import jp.ill.photon.util.CheckUtil;
import jp.ill.photon.util.StringUtil;
import jp.ill.photon.util.UtilTools;

public class CalcDeliveryFeeModule implements PhotonModule {

	@ModuleParam(required = true, paramGroup = true)
	private Delivery delivery = new Delivery();

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "cart.total_whole_sale_price")
	private String totalWholeSalePrice;

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.postageFreeThreshold.note")
	private String postageFreeThreshold = null;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.deliveryFeeItemCd.note")
	private String deliveryFeeItemCd;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.aladdinsetting")
	private AoSetting aoSetting;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "got_session.userForm.user_cd")
	private String userCd;

	@ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "user.first")
    private Map<String, Object> userForm;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.verNo.note")
	private String verNo;

	public static final String ERROR_MSG_ID = "calcDeliveryFeeErrMes";

    public String getVerNo() {
		return verNo;
	}

	public void setVerNo(String verNo) {
		this.verNo = verNo;
	}

	public Map<String, Object> getUserForm() {
        return userForm;
    }

    public void setUserForm(Map<String, Object> userForm) {
        this.userForm = userForm;
    }

	public String getUserCd() {
		return userCd;
	}

	public void setUserCd(String userCd) {
		this.userCd = userCd;
	}

	public AoSetting getAoSetting() {
		return aoSetting;
	}

	public void setAoSetting(AoSetting aoSetting) {
		this.aoSetting = aoSetting;
	}

	public String getPostageFreeThreshold() {
		return postageFreeThreshold;
	}

	public void setPostageFreeThreshold(String postageFreeThreshold) {
		this.postageFreeThreshold = postageFreeThreshold;
	}

	public String getTotalWholeSalePrice() {
		return totalWholeSalePrice;
	}

	public void setTotalWholeSalePrice(String totalWholeSalePrice) {
		this.totalWholeSalePrice = totalWholeSalePrice;
	}

	public Delivery getDelivery() {
		return delivery;
	}

	public void setDelivery(Delivery delivery) {
		this.delivery = delivery;
	}

	public String getDeliveryFeeItemCd() {
		return deliveryFeeItemCd;
	}

	public void setDeliveryFeeItemCd(String deliveryFeeItemCd) {
		this.deliveryFeeItemCd = deliveryFeeItemCd;
	}

	@Override
	public ModuleResult execute(ModuleContext context) throws PhotonModuleException {
		ModuleResult result = new ModuleResult();

		// 注文合計金額が無料なら送料は発生しない＜仕様から削除
//		if (new BigDecimal(0.0).equals(new BigDecimal(totalWholeSalePrice))) {
//			return result;
//		}

		// 配送料地域の特定 TODO Ver2.0では郵便番号（と重量？）から取得する方式にいずれ置き換える
		DeliveryFee deliveryFee = getMatchingDeliveryFeeCd(context,
				delivery.getCityName1(), delivery.getCityName2(),
				delivery.getCityName3());

		//コードの存在チェック
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		JsonDataDao dao = new JsonDataDaoImpl();
		List<Map<String, Object>> itemMap = dao.getItemForOrder(context.getDto().getTenantId(), deliveryFeeItemCd, userCd);
        if (itemMap == null || itemMap.isEmpty()) {
        	result.getReturnData().put("area_name", deliveryFee.getAreaName());

        	result.setResultCode("error");

		    List<ActionDtoMessage> errorMessages = new ArrayList<ActionDtoMessage>();
		    ActionDtoMessage msg = new ActionDtoMessage();
            msg.setMessageId(ERROR_MSG_ID);
            errorMessages.add(msg);
		    result.getMessages().put("error_page", errorMessages);

            return result;
        }

		if (!CheckUtil.isEmpty(postageFreeThreshold)
				&& !CheckUtil.isEmpty(totalWholeSalePrice)) {

			// 内税の場合はしきい値に消費税分を加算
			if (Tax.TaxDiv.INCLUDING.equals(StringUtil.defaultString(userForm.get("tax_div"), ""))) {
				Double taxRate = CalcTaxModule.getTaxRateFromItemCd(context, deliveryFeeItemCd, String.valueOf(userForm.get("user_cd")), verNo) + 1.0;

				postageFreeThreshold = UtilTools.multiplyBigDecimal(postageFreeThreshold, String.valueOf(taxRate));
			}

			if (new BigDecimal(totalWholeSalePrice)
					.compareTo(new BigDecimal(postageFreeThreshold)) >= 0) {
				// 合計金額が配送料無料注文金額閾値以上の場合は配送料はなし

			    result.getReturnData().put("area_name", deliveryFee.getAreaName());
			    result.getReturnData().put("delivery_fee_cd", deliveryFee.getDeliveryFeeCd());
				return result;
			}
		}

		String deliveryCharge = "0";
		if (deliveryFee != null) {

			deliveryCharge = deliveryFee.getDeliveryCharge().toString();

			// 内税の場合は消費税分を加算
			if (Tax.TaxDiv.INCLUDING.equals(StringUtil.defaultString(userForm.get("tax_div"), ""))) {
				Double taxRate = CalcTaxModule.getTaxRateFromItemCd(context, deliveryFeeItemCd, String.valueOf(userForm.get("user_cd")), verNo) + 1.0;

				deliveryCharge = UtilTools.multiplyBigDecimal(deliveryCharge, String.valueOf(taxRate));
			}

			deliveryCharge = AoUtil.convAoRoundingSalesPrice(
					deliveryCharge,
					AoUtil.DispPattern.PATTERN_NO_COMMA,
					aoSetting.getSalesPriceDigitNum(),
					StringUtil.defaultString(userForm.get("price_fraction_div"), "0") );
		}

		if (new BigDecimal(deliveryCharge).compareTo(BigDecimal.ZERO) > 0) {
			// 配送料明細を作る
			CartItem deliveryFeeItem = new CartItem();
			deliveryFeeItem.setItemCd(deliveryFeeItemCd);
			deliveryFeeItem.setOrderQuantity(new BigDecimal(1.0));
			deliveryFeeItem.setOrderCaseQuantity(new BigDecimal(0.0));
			deliveryFeeItem.setRetailPrice(new BigDecimal(0.0));
			deliveryFeeItem.setWholesalePrice(new BigDecimal(deliveryCharge));
			deliveryFeeItem.setQuantityPerCase(new BigDecimal(0.0));

			result.getReturnData().put("delivery_fee", deliveryFeeItem.toMap());
		}

		result.getReturnData().put("delivery_fee_cd", deliveryFee.getDeliveryFeeCd());
		result.getReturnData().put("area_name", deliveryFee.getAreaName());

		return result;
	}

	/**
	 *
	 * @param context
	 * @return
	 */
	public static DeliveryFee getMatchingDeliveryFeeCd(	ModuleContext context,
														String cityName1,
														String cityName2,
														String cityName3) {

		StringBuffer addressBuf = new StringBuffer();
		if (cityName1 != null) {
			addressBuf.append(cityName1);
		}
		if (cityName2 != null) {
			addressBuf.append(cityName2);
		}
		if (cityName3 != null) {
			addressBuf.append(cityName3);
		}
		String address = addressBuf.toString();

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		JsonDataDao dao = new JsonDataDaoImpl();

		DeliveryFee deliveryFee = tm.required(() -> {
			List<Map<String, Object>> deliveryFeeMap = dao
					.getMatchingDeliveryFee(context.getDto().getTenantId(),
							address);
			if (deliveryFeeMap == null || deliveryFeeMap.isEmpty()) {
				return null;
			} else {
				return DeliveryFee.valueOf(deliveryFeeMap.get(0));
			}
		});

		return deliveryFee;
	}

}
