package jp.ill.photon.module.order;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.model.AoSetting;
import jp.ill.photon.model.CartItem;
import jp.ill.photon.model.ChargeFee;
import jp.ill.photon.model.Payment.AccountWayDiv;
import jp.ill.photon.model.Tax;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.module.tax.CalcTaxModule;
import jp.ill.photon.util.AoUtil;
import jp.ill.photon.util.CheckUtil;
import jp.ill.photon.util.StringUtil;
import jp.ill.photon.util.UtilTools;

public class CalcChargeFeeModule implements PhotonModule {

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "cart.total_whole_sale_price")
    private String totalWholeSalePrice;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.serviceChargeCd.note")
    private String chargeFeeItemCd;

    @ModuleParam(required = false)
    @DefaultParamSetting(transferType = "param", transferVal = "selected_payment_cd")
    private String selectedPaymentCd;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType="dto", transferVal="common.aladdinsetting")
    private AoSetting aoSetting;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "user.first")
    private Map<String, Object> userForm;

    @ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.verNo.note")
	private String verNo;

    @ModuleParam(required = false)
    @DefaultParamSetting(transferType = "dto", transferVal = "got_payment_session.selectedPaymentCd")
    private String sessionSelectedPaymentCd = null;

    public String getSessionSelectedPaymentCd() {
		return sessionSelectedPaymentCd;
	}

	public void setSessionSelectedPaymentCd(String sessionSelectedPaymentCd) {
		this.sessionSelectedPaymentCd = sessionSelectedPaymentCd;
	}

	public static final String ERROR_MSG_ID = "calcChargeFeeErrMes";

    public String getVerNo() {
		return verNo;
	}

	public void setVerNo(String verNo) {
		this.verNo = verNo;
	}


    public Map<String, Object> getUserForm() {
        return userForm;
    }

    public void setUserForm(Map<String, Object> userForm) {
        this.userForm = userForm;
    }

    public AoSetting getAoSetting() {
        return aoSetting;
    }

    public void setAoSetting(AoSetting aoSetting) {
        this.aoSetting = aoSetting;
    }

    public String getSelectedPaymentCd() {
        return selectedPaymentCd;
    }

    public void setSelectedPaymentCd(String selectedPaymentCd) {
        this.selectedPaymentCd = selectedPaymentCd;
    }

    public String getChargeFeeItemCd() {
        return chargeFeeItemCd;
    }

    public void setChargeFeeItemCd(String chargeFeeItemCd) {
        this.chargeFeeItemCd = chargeFeeItemCd;
    }

    public String getTotalWholeSalePrice() {
        return totalWholeSalePrice;
    }

    public void setTotalWholeSalePrice(String totalWholeSalePrice) {
        this.totalWholeSalePrice = totalWholeSalePrice;
    }

    @Override
    public ModuleResult execute(ModuleContext context) throws PhotonModuleException {
        ModuleResult result = new ModuleResult();

        if (CheckUtil.isEmpty(chargeFeeItemCd) || CheckUtil.isEmpty(totalWholeSalePrice)) {
            return result;
        }

        String paymentCd = CheckUtil.isEmpty(selectedPaymentCd) ? sessionSelectedPaymentCd : selectedPaymentCd;

        // 手数料を表示するのは代引のみ
        if (!AccountWayDiv.COD.equals(paymentCd)) {
            return result;
        }

        TransactionManager tm = DomaConfig.singleton().getTransactionManager();
        JsonDataDao dao = new JsonDataDaoImpl();

        List<Map<String, Object>> itemMap = dao.getItemForOrder(context.getDto().getTenantId(), chargeFeeItemCd, String.valueOf(userForm.get("user_cd")));
        if (itemMap == null || itemMap.isEmpty()) {
        	result.setResultCode("error");

		    List<ActionDtoMessage> errorMessages = new ArrayList<ActionDtoMessage>();
		    ActionDtoMessage msg = new ActionDtoMessage();
            msg.setMessageId(ERROR_MSG_ID);
            errorMessages.add(msg);
		    result.getMessages().put("error_page", errorMessages);

            return result;
        }

        Double taxRate = 1.0;
		if (Tax.TaxDiv.INCLUDING.equals(StringUtil.defaultString(userForm.get("tax_div"), ""))) {
			taxRate = CalcTaxModule.getTaxRateFromItemCd(context, chargeFeeItemCd, String.valueOf(userForm.get("user_cd")), verNo) + 1.0;
		}

		final Double taxRateConst = new Double(taxRate);
        List<ChargeFee> chargeFeeList = tm.required(() -> {
            List<ChargeFee> list = new ArrayList<ChargeFee>();
            List<Map<String, Object>> chargeFeeListMap = dao.getChargeFeeSettingList(context.getDto().getTenantId());

            for (Map<String, Object> codMap : chargeFeeListMap) {
            	ChargeFee fee = ChargeFee.valueOf(codMap);

            	// 内税の場合はしきい値も加算
            	if (Tax.TaxDiv.INCLUDING.equals(StringUtil.defaultString(userForm.get("tax_div"), ""))) {
            		fee.setThreshold(new BigDecimal(UtilTools.multiplyBigDecimal(fee.getThreshold().toString(), String.valueOf(taxRateConst))));
            	}

                list.add(fee);
            }

            return list;
        });

        // 手数料を取得
        String chargePrice = "0";
        BigDecimal totalPrice = new BigDecimal(totalWholeSalePrice);
        for (ChargeFee chargeFee : chargeFeeList) {
            if (totalPrice.compareTo(chargeFee.getThreshold()) >= 0) {
                BigDecimal bdFee = BigDecimal.ZERO;
                if (chargeFee.getFee() != null) {
                    bdFee = chargeFee.getFee().setScale(0,
                            BigDecimal.ROUND_DOWN);
                }

                chargePrice = chargeFee.getFee().toString();

                 // 内税の場合は消費税分を加算
    			if (Tax.TaxDiv.INCLUDING.equals(StringUtil.defaultString(userForm.get("tax_div"), ""))) {
    				chargePrice = UtilTools.multiplyBigDecimal(chargePrice, String.valueOf(taxRate));
    			}

                chargePrice = AoUtil.convAoRoundingSalesPrice(chargePrice,
                        AoUtil.DispPattern.PATTERN_NO_COMMA, aoSetting.getSalesPriceDigitNum(), StringUtil.defaultString(userForm.get("price_fraction_div"), "0") );
            }
        }

        // 手数料明細を作成
        CartItem chargeFeeItem = new CartItem();
        chargeFeeItem.setItemCd(chargeFeeItemCd);
        chargeFeeItem.setOrderQuantity(new BigDecimal(1.0));
        chargeFeeItem.setOrderCaseQuantity(new BigDecimal(0.0));
        chargeFeeItem.setRetailPrice(new BigDecimal(0.0));
        chargeFeeItem.setWholesalePrice(new BigDecimal(chargePrice));
        chargeFeeItem.setQuantityPerCase(new BigDecimal(0.0));

        result.getReturnData().put("charge_fee", chargeFeeItem.toMap());

        return result;
    }

}
