package jp.ill.photon.module.mail;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.doma.SqlSelectQueryFactory;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.SearchForm;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.MapUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.SQLUtil;
import jp.ill.photon.util.StringUtil;
import jp.ill.photon.util.mail.MailUtil;
import jp.ill.photon.util.mail.SmtpMailSender;

import org.apache.commons.mail.EmailException;
import org.seasar.doma.MapKeyNamingType;
import org.seasar.doma.internal.jdbc.command.MapResultListHandler;
import org.seasar.doma.jdbc.command.SelectCommand;
import org.seasar.doma.jdbc.query.SqlSelectQuery;
import org.seasar.doma.jdbc.tx.TransactionManager;

/**
 * メールキューテーブルからメールを一括送信するモジュール
 *
 * @author h_tanaka
 *
 */
public class SmtpSendFromMailQueueModule implements PhotonModule {

	/** メール送信サーバアドレス */
	@ModuleParam(required = true)
	private String smtpHost;

	/** メール送信サーバポート */
	@ModuleParam(required = true)
	private String smtpPort;

	/** メール送信認証用ユーザー名 */
	@ModuleParam(required = false)
	private String smtpUserName;

	/** メール送信認証用パスワード */
	@ModuleParam(required = false)
	private String smtpPassword;

	@ModuleParam(required = true)
	private SearchForm searchForm;

	@ModuleParam(required = true)
	private String sqlFileDirPath;

	@ModuleParam(required = true)
	private String sqlFilePath;

	@ModuleParam(required = false)
	private String afterRowSqlFilePath;

	@ModuleParam(required = false)
	private Map<String, Object> additionalMailParams;

	@ModuleParam(required = true)
	private int queryLimit;

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(
			SmtpSendFromMailQueueModule.class);

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		// メール送信オブジェクト準備
		int port = Integer.parseInt(getSmtpPort());
		SmtpMailSender sender = new SmtpMailSender(getSmtpHost(), port,
				getSmtpUserName(), getSmtpPassword());

		ModuleResult result = new ModuleResult();

		// DB検索・更新用オブジェクト準備
		SqlSelectQueryFactory selectFactory = SqlSelectQueryFactory
				.newInstance();
		SqlSelectQuery selectQuery = selectFactory.createSelectQueryFromFile(
				getSearchForm().getParamMap(), context.getDto(),
				getSqlFileDirPath(), getSqlFilePath());

		SqlSelectQuery afterRowQuery = null;
		if (StringUtil.defaultString(getAfterRowSqlFilePath(), "") != "") {
			afterRowQuery = selectFactory.createSelectQueryFromFile(
					new HashMap<String, Object>(), context.getDto(),
					getSqlFileDirPath(), getAfterRowSqlFilePath());
		}

		Map<String, Object> additionalParams = createParamData(
				getAdditionalMailParams(), context.getDto());

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		// 送信対象メールキュー取得処理（queryLimit件ずつ取得してメール送信する）
		int cnt = -1;
		int limit = getQueryLimit();
		int offset = 0;
		while (cnt != 0) {
			selectQuery.addParameter("limit", Integer.class, limit);
			selectQuery.addParameter("offset", Integer.class, offset);
			selectQuery.prepare();

			// 検索処理を実行
			List<Map<String, Object>> resultList = tm.required(() -> {
				SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
						selectQuery,
						new MapResultListHandler(MapKeyNamingType.NONE));
				return command.execute();
			});
			SQLUtil.convertDbValues(resultList);

			// 検索結果が空ではない場合
			if (resultList != null && !resultList.isEmpty()) {

				Map<String, Object> updateFields = null;
				List<String> toAddressList = null;
				List<String> ccAddressList = null;
				List<String> bccAddressList = null;
				Map<String, Object> mailSetting = null;
				Map<String, Object> mailQueue = null;
				Map<String, Object> mailParams = null;
				String mailTitle = null;
				String mailContent = null;
				for (Map<String, Object> mailInfo : resultList) {

					mailSetting = (Map<String, Object>) mailInfo.get("mail");
					if (mailSetting == null) {
						logger.error("メール設定が取得できませんでした: (guid)"
								+ mailInfo.getOrDefault("guid", ""));
						continue;
					}

					mailQueue = (Map<String, Object>) mailInfo.get("queue");
					if (mailQueue == null) {
						logger.error("メールキュー情報が取得できませんでした: (guid)"
								+ mailInfo.getOrDefault("guid", ""));
						continue;
					}

					toAddressList = (List<String>) mailInfo
							.getOrDefault("address_to", new ArrayList<>());
					if (toAddressList == null || toAddressList.isEmpty()) {
						logger.error("メール送信先が取得できませんでした: (queue_id)"
								+ mailQueue.getOrDefault("queue_id", ""));
						continue;
					}

					ccAddressList = (List<String>) mailInfo
							.getOrDefault("address_cc", new ArrayList<>());

					bccAddressList = (List<String>) mailInfo
							.getOrDefault("address_bcc", new ArrayList<>());

					mailParams = MapUtil.mergeMaps(additionalParams,
							(Map<String, Object>) mailInfo
									.getOrDefault("params", new HashMap<>()));

					mailTitle = MailUtil.processMailTemplate(
							StringUtil.defaultString(
									mailSetting.getOrDefault("title", "タイトルなし"), ""),
							mailParams);

					mailContent = MailUtil.processMailTemplate(
							StringUtil.defaultString(
									mailSetting.getOrDefault("text", ""), ""),
							mailParams);

					updateFields = new HashMap<>();
					updateFields.put("queue_id",
							mailQueue.getOrDefault("queue_id", ""));
					updateFields.put("sended_flg", "0");

					try {
						sender.sendMail(
								MailUtil.addressListToMap(toAddressList),
								MailUtil.addressListToMap(ccAddressList),
								MailUtil.addressListToMap(bccAddressList),
								mailTitle,
								mailContent,
								StringUtil.defaultString(mailSetting
										.getOrDefault("charset", "UTF-8"), ""),
								StringUtil.defaultString(mailSetting
										.getOrDefault("re_mail", "UTF-8"), ""),
								StringUtil.defaultString(mailSetting
										.getOrDefault("re_mail_name", "UTF-8"), ""),// 送信元ユーザー名
								StringUtil.defaultString(mailSetting
										.getOrDefault("re_mail", "UTF-8"), ""));
						updateFields.put("sended_flg", "1");
					} catch (EmailException e) {
						logger.error("メール送信失敗", e);
						updateFields.put("sended_flg", "9");
					}

					updateAfterRow(afterRowQuery, context.getDto(),
							updateFields, tm);
				}

				// クエリの結果件数を設定
				cnt = resultList.size();
			} else {
				cnt = 0;
			}

			offset += limit;
		}

		// 結果オブジェクト生成
		result.setResultCode("success");
		result.getReturnData().put("result_code", "success");

		return result;
	}

	/**
	 * SQLファイルのパラメータデータを生成する
	 *
	 * @param srcParams
	 * @param context
	 * @return
	 */
	@SuppressWarnings({ "rawtypes" })
	protected Map<String, Object> createParamData(	Map<String, Object> srcParams,
													ActionDto dto) {
		// パラメータデータを取得
		Map<String, Object> params = new LinkedHashMap<String, Object>();
		for (Map.Entry<String, Object> param : srcParams.entrySet()) {

			String type = String.valueOf(((Map) param.getValue()).get("type"));
			params.put((String) param.getKey(), ParamUtil.getParamValueByType(
					type, ((Map) param.getValue()).get("val"), dto));
		}

		return params;
	}

	/**
	 * 行処理後クエリが指定されている場合は実行する
	 *
	 * @param factory
	 * @param sql
	 * @param dto
	 * @param line
	 * @param convertedLine
	 * @throws PhotonModuleException
	 */
	protected List<Map<String, Object>> updateAfterRow(	SqlSelectQuery selectQuery,
														ActionDto dto,
														Map<String, Object> fields,
														TransactionManager tm)
			throws PhotonModuleException {

		if (selectQuery == null) {
			return new ArrayList<Map<String, Object>>();
		}

		selectQuery.addParameter("tenantId", String.class, dto.getTenantId());
		selectQuery.addParameter("fields", Map.class, fields);
		selectQuery.prepare();

		// 更新クエリ実行
		return tm.required(() -> {
			SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
					selectQuery,
					new MapResultListHandler(MapKeyNamingType.NONE));
			return command.execute();
		});
	}

	public String getSmtpHost() {
		return smtpHost;
	}

	public void setSmtpHost(String smtpHost) {
		this.smtpHost = smtpHost;
	}

	public String getSmtpPort() {
		return smtpPort;
	}

	public void setSmtpPort(String smtpPort) {
		this.smtpPort = smtpPort;
	}

	public String getSmtpUserName() {
		return smtpUserName;
	}

	public void setSmtpUserName(String smtpUserName) {
		this.smtpUserName = smtpUserName;
	}

	public String getSmtpPassword() {
		return smtpPassword;
	}

	public void setSmtpPassword(String smtpPassword) {
		this.smtpPassword = smtpPassword;
	}

	public SearchForm getSearchForm() {
		return searchForm;
	}

	public void setSearchForm(SearchForm searchForm) {
		this.searchForm = searchForm;
	}

	public String getSqlFileDirPath() {
		return sqlFileDirPath;
	}

	public void setSqlFileDirPath(String sqlFileDirPath) {
		this.sqlFileDirPath = sqlFileDirPath;
	}

	public String getSqlFilePath() {
		return sqlFilePath;
	}

	public void setSqlFilePath(String sqlFilePath) {
		this.sqlFilePath = sqlFilePath;
	}

	public String getAfterRowSqlFilePath() {
		return afterRowSqlFilePath;
	}

	public void setAfterRowSqlFilePath(String afterRowSqlFilePath) {
		this.afterRowSqlFilePath = afterRowSqlFilePath;
	}

	public Map<String, Object> getAdditionalMailParams() {
		return additionalMailParams;
	}

	public void setAdditionalMailParams(Map<String, Object> additionalMailParams) {
		this.additionalMailParams = additionalMailParams;
	}

	public int getQueryLimit() {
		return queryLimit;
	}

	public void setQueryLimit(int queryLimit) {
		this.queryLimit = queryLimit;
	}

}
