package jp.ill.photon.module.mail;

import java.util.LinkedHashMap;
import java.util.Map;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.mail.SmtpMailSender;

import org.apache.commons.mail.EmailException;

/**
 * SMTPメール送信モジュール
 *
 * @author h_tanaka
 *
 */
public class SmtpMailSendModule implements PhotonModule {

	/** メール送信サーバアドレス */
	@ModuleParam(required = true)
	private String smtpHost;

	/** メール送信サーバポート */
	@ModuleParam(required = true)
	private String smtpPort;

	/** メール送信認証用ユーザー名 */
	@ModuleParam(required = false)
	private String smtpUserName;

	/** メール送信認証用パスワード */
	@ModuleParam(required = false)
	private String smtpPassword;

	/** TO送信先メールアドレス（複数の場合はカンマ区切） */
	@ModuleParam(required = true)
	private String toAddresses;

	/** CC送信先メールアドレス（複数の場合はカンマ区切） */
	@ModuleParam(required = false)
	private String ccAddresses;

	/** BCC送信先メールアドレス（複数の場合はカンマ区切） */
	@ModuleParam(required = false)
	private String bccAddresses;

	/** 送信メール件名 */
	@ModuleParam(required = true)
	private String subject;

	/** 送信メール内容 */
	@ModuleParam(required = true)
	private String contents;

	/** 送信メールキャラクタセット */
	@ModuleParam(required = false)
	private String charSet;

	/** 送信元メールアドレス */
	@ModuleParam(required = true)
	private String fromAddress;

	/** 送信者名 */
	@ModuleParam(required = false)
	private String fromAddressName;

	/** 返信用メールアドレス */
	@ModuleParam(required = true)
	private String replyAddress;

	public static final String MAIL_ERROR_MSG_ID = "mailSendErrMes";

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(SmtpMailSendModule.class);

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		int port = Integer.parseInt(getSmtpPort());
		SmtpMailSender sender = new SmtpMailSender(getSmtpHost(), port,
				getSmtpUserName(), getSmtpPassword());

		String[] toAddressList = null;
		if (getToAddresses() != null && getToAddresses().length() != 0) {
			toAddressList = getToAddresses().split(",");
		}
		Map<String, String> toAddressMap = addressesToMap(toAddressList);

		String[] ccAddressList = null;
		if (getCcAddresses() != null && getCcAddresses().length() != 0) {
			ccAddressList = getCcAddresses().split(",");
		}
		Map<String, String> ccAddressMap = addressesToMap(ccAddressList);

		String[] bccAddressList = null;
		if (getBccAddresses() != null && getBccAddresses().length() != 0) {
			bccAddressList = getBccAddresses().split(",");
		}
		Map<String, String> bccAddressMap = addressesToMap(bccAddressList);

		ModuleResult result = new ModuleResult();
		try {
			sender.sendMail(toAddressMap, ccAddressMap, bccAddressMap,
					getSubject(), getContents(), getCharSet(), getFromAddress(),
					getFromAddressName(), getReplyAddress());
		} catch (EmailException e) {
			logger.error("メール送信失敗", e);
			result.setResultCode("error");
			result.getReturnData().put("result_code", "error");
			return result;
		}

		// 結果オブジェクト生成
		result.setResultCode("success");
		result.getReturnData().put("result_code", "success");

		return result;
	}

	/**
	 * メールアドレスのリストをMailSender用のMapに変換する
	 *
	 * @param addressList
	 * @return
	 */
	protected Map<String, String> addressesToMap(String[] addressList) {
		Map<String, String> addressMap = new LinkedHashMap<>();
		if (addressList == null) {
			return addressMap;
		}

		for (String address : addressList) {
			addressMap.put(address, null);
		}

		return addressMap;
	}

	public String getSmtpHost() {
		return smtpHost;
	}

	public void setSmtpHost(String smtpHost) {
		this.smtpHost = smtpHost;
	}

	public String getSmtpPort() {
		return smtpPort;
	}

	public void setSmtpPort(String smtpPort) {
		this.smtpPort = smtpPort;
	}

	public String getSmtpUserName() {
		return smtpUserName;
	}

	public void setSmtpUserName(String smtpUserName) {
		this.smtpUserName = smtpUserName;
	}

	public String getSmtpPassword() {
		return smtpPassword;
	}

	public void setSmtpPassword(String smtpPassword) {
		this.smtpPassword = smtpPassword;
	}

	public String getSubject() {
		return subject;
	}

	public void setSubject(String subject) {
		this.subject = subject;
	}

	public String getContents() {
		return contents;
	}

	public void setContents(String contents) {
		this.contents = contents;
	}

	public String getCharSet() {
		return charSet;
	}

	public void setCharSet(String charSet) {
		this.charSet = charSet;
	}

	public String getFromAddress() {
		return fromAddress;
	}

	public void setFromAddress(String fromAddress) {
		this.fromAddress = fromAddress;
	}

	public String getFromAddressName() {
		return fromAddressName;
	}

	public void setFromAddressName(String fromAddressName) {
		this.fromAddressName = fromAddressName;
	}

	public String getReplyAddress() {
		return replyAddress;
	}

	public void setReplyAddress(String replyAddress) {
		this.replyAddress = replyAddress;
	}

	public String getToAddresses() {
		return toAddresses;
	}

	public void setToAddresses(String toAddresses) {
		this.toAddresses = toAddresses;
	}

	public String getCcAddresses() {
		return ccAddresses;
	}

	public void setCcAddresses(String ccAddresses) {
		this.ccAddresses = ccAddresses;
	}

	public String getBccAddresses() {
		return bccAddresses;
	}

	public void setBccAddresses(String bccAddresses) {
		this.bccAddresses = bccAddresses;
	}

}
