package jp.ill.photon.module.itemcategory;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.seasar.doma.FetchType;
import org.seasar.doma.MapKeyNamingType;
import org.seasar.doma.internal.jdbc.command.MapResultListHandler;
import org.seasar.doma.internal.jdbc.sql.SqlParser;
import org.seasar.doma.jdbc.SqlLogType;
import org.seasar.doma.jdbc.command.SelectCommand;
import org.seasar.doma.jdbc.query.SqlSelectQuery;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.SystemSetting;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ParamUtil;

/**
 * [drill_down_cat] 商品分類リスト一括取得モジュール.
 * 
 * <p>
 * 商品分類1～10から商品マスタで使用されている商品分類の一覧を取得する。<br/>
 * モジュールパラメータとして選択中の商品分類を受取り、フィルタした結果を返す。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 * <dl>
 * <dt>values</dt>
 * <dd>「item_category_*」をキー、各商品分類1～10の選択値（モジュールパラメータcategoriesで渡された値）を値として持つマップ。</dd>
 * <dt>list</dt>
 * <dd>「item_category_*」をキー、各商品分類1～10の検索結果リスト(List&lt;Map&lt;String,
 * Object&gt;&gt;)を値として持つマップ。システム設定上非表示になっている商品分類は値が空リストとなる。</dd>
 * <dt>item_category_cnt</dt>
 * <dd>商品分類マスタ数。10。</dd>
 * <dt>disp_item_category_cnt</dt>
 * <dd>システム設定上表示状態になっている商品分類マスタ数。</dd>
 * </dl>
 * </p>
 * 
 * @author m_fukukawa
 *
 */
public class DrillDownSearchModule implements PhotonModule {

	/**
	 * システム設定用モジュールパラメータ.
	 * 
	 * <p>
	 * 通常はDTOの「common.systemsetting」を指定する。<br/>
	 * システム設定の商品分類関連設定を読み込み、処理を行う。
	 * </p>
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>dto</dd>
	 * <dt>transfer_val</dt>
	 * <dd>common.systemsetting</dd>
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = true)
	private SystemSetting common;

	/**
	 * SQLファイルのフォルダパス指定用モジュールパラメータ.
	 * 
	 * <p>
	 * 通常はDTOの「common.systemsetting.exSqlFileDir.note」を指定することで、<br/>
	 * システム設定のSQL保存ディレクトリからSQLファイルを読み込むことができる。
	 * </p>
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>dto</dd>
	 * <dt>transfer_val</dt>
	 * <dd>common.systemsetting.exSqlFileDir.note</dd>
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = true)
	private String sqlFileDirPath;

	/**
	 * SQLファイルパス指定用モジュールパラメータ.
	 * 
	 * <p>
	 * sqlFileDirPathからの相対パスを指定する。
	 * </p>
	 */
	@ModuleParam(required = true)
	private String sqlFilePath;

	/**
	 * テナントID用モジュールパラメータ.
	 * 
	 * <p>
	 * テナントIDを指定する。<br/>
	 * 通常はparamの「_init.tenant_id」を指定する。
	 * </p>
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>param</dd>
	 * <dt>transfer_val</dt>
	 * <dd>_init.tenant_id</dd>
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = true)
	private String tenantId;

	/**
	 * 表示フィルタ用モジュールパラメータ.
	 * 
	 * <p>
	 * sql_file_pathで指定したSQLファイル内で使用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String visibleOnly;

	/**
	 * 展示会コード用モジュールパラメータ.
	 * 
	 * <p>
	 * sql_file_pathで指定したSQLファイル内で使用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String exhibitCd;

	/**
	 * 得意先別有効商品区分用モジュールパラメータ.
	 * 
	 * <p>
	 * sql_file_pathで指定したSQLファイル内で使用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "static", transferVal = "1")
	private String userActiveItemFlg;

	/**
	 * 得意先コード用モジュールパラメータ.
	 * 
	 * <p>
	 * sql_file_pathで指定したSQLファイル内で使用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String userCd;

	/**
	 * カタログコード用モジュールパラメータ.
	 * 
	 * <p>
	 * sql_file_pathで指定したSQLファイル内で使用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String catalogCd;

	/**
	 * 商品分類別選択値用モジュールパラメータ.
	 * 
	 * <p>
	 * 商品分類1～10でそれぞれ選択中の値のマップを受け取る。
	 * </p>
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 * 
	 * <pre>
	 * {
	 *   "item_category_cd1": {
	 *     "type": "param",
	 *     "val": "category1"
	 *   },
	 *   ...
	 *   "item_category_cd10": {
	 *     "type": "param",
	 *     "val": "category10"
	 *   }
	 * }
	 * </pre>
	 * 
	 * </dd>
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = false)
	private Map<String, Object> categories;

	private static int MAX_ITEM_CATEGORY_CNT = 10;
	private static String FMT_ITEM_CATEGORY_DISP_SORT = "itemCategoryDispSort";
	private static String FMT_ITEM_CATEGORY_DISP_FLG = "itemCategory%sDispFlg";

	private ActionDto dto;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		dto = context.getDto();
		int dispItemcategoryCnt = 0;

		ModuleResult result = new ModuleResult();

		Map<String, Object> listMap = new HashMap<String, Object>();
		Map<String, Object> valMap = new HashMap<String, Object>();
		// 展示会
		// (展示会モードONのとき)
		if (SystemSetting.ExhibitFlg.ON.equals(common.get("exhibitFlg"))) {
			String key = "exhibit";
			List<Map<String, Object>> list = null; // TODO:展示会リスト
			listMap.put(key, list);
			valMap.put(key, exhibitCd);
		}

		// 商品分類
		for (int i = 1; i <= MAX_ITEM_CATEGORY_CNT; i++) {

			List<Map<String, Object>> catList = null;
			String itemCategoryDiv = common
					.get(FMT_ITEM_CATEGORY_DISP_SORT + i);
			String cat = "item_category_" + itemCategoryDiv;

			String catCd = "";

			if (!StringUtils.isEmpty(itemCategoryDiv)) {

				catCd = this.getSettingValueFromCategories(itemCategoryDiv);

				String dispFlg = common.get(String
						.format(FMT_ITEM_CATEGORY_DISP_FLG, itemCategoryDiv));
				if (SystemSetting.DispFlg.YES.equals(dispFlg)) {
					dispItemcategoryCnt++;
					catList = getSearchCategory(i);
				} else {
					catList = new ArrayList<Map<String, Object>>();
				}

			} else {
				catList = new ArrayList<Map<String, Object>>();
			}

			// 戻り値セット
			listMap.put(cat, catList);
			valMap.put(cat, catCd);

		}

		result.getReturnData().put("list", listMap);
		result.getReturnData().put("values", valMap);
		result.getReturnData().put("item_category_cnt", MAX_ITEM_CATEGORY_CNT);
		result.getReturnData().put("disp_item_category_cnt",
				dispItemcategoryCnt);

		return result;

	}

	/**
	 * 値取得
	 *
	 * @param 商品分類番号
	 * @return 値
	 *
	 */
	@SuppressWarnings("rawtypes")
	private String getSettingValueFromCategories(String itemCategoryDiv) {
		// Map map = (Map)categories.get("item_category_cd" +
		// Integer.toString(num));
		Map map = (Map) categories.get("item_category_cd" + itemCategoryDiv);
		Object val = ParamUtil.getParamObjectValueByType(map, dto);
		return (val == null) ? null : (String) val;
	}

	/**
	 * 商品分類取得
	 *
	 * @param sortPos ソート位置
	 * @return リスト
	 * @throws PhotonModuleException
	 */
	private List<Map<String, Object>> getSearchCategory(int sortPos)
			throws PhotonModuleException {

		// 商品分類区分の取得
		String itemCategory = common.get(FMT_ITEM_CATEGORY_DISP_SORT + sortPos);
		// カテゴリリスト
		List<Map<String, Object>> ret = new ArrayList<Map<String, Object>>();
		if (StringUtils.isEmpty(itemCategory)) {
			return ret;
		}

		// 上位商品分類の条件の判定(下位商品分類の条件は使用しない)
		List<String> categoryList = new ArrayList<String>() {
			{
				for (int i = 1; i <= MAX_ITEM_CATEGORY_CNT; i++) {
					add("");
				}
			}
		};
		for (int i = 1; i <= MAX_ITEM_CATEGORY_CNT; i++) {
			String itemCategoryDiv = common
					.get(FMT_ITEM_CATEGORY_DISP_SORT + i);
			String catCd = "";
			Integer idx = null;
			if (!StringUtils.isEmpty(itemCategoryDiv)) {
				catCd = this.getSettingValueFromCategories(itemCategoryDiv);
				idx = Integer.parseInt(itemCategoryDiv) - 1;
			}
			if (idx != null && sortPos > i) {
				categoryList.set(idx, catCd);
			}
		}

		// クエリ実行
		List<Map<String, Object>> tmpList = this.executeQuery(itemCategory,
				categoryList);
		if (tmpList == null) {
			return ret;
		}
		ret = tmpList;

		return ret;

	}

	/**
	 * クエリ実行
	 *
	 * @param itemCategory 商品分類区分
	 * @param itemCatDiv 商品分類区分
	 * @return リスト
	 * @throws PhotonModuleException
	 */
	private List<Map<String, Object>> executeQuery(	String itemCategory,
													List<String> categoryList)
			throws PhotonModuleException {

		// SQLファイル存在チェック
		Path path = Paths.get(getSqlFileDirPath(), getSqlFilePath());
		if (!Files.exists(path)) {
			throw new PhotonModuleException("SQLファイルが存在しません:" + path.toString(),
					null);
		}

		// SQLを読み込んでDomaのSQLParserで解析
		SqlParser parser;
		try {
			parser = new SqlParser(
					Files.readAllLines(path, Charset.forName("UTF-8")).stream()
							.collect(Collectors.joining("\n")));
		} catch (IOException e) {
			throw new PhotonModuleException("SQLの解析に失敗しました:" + path.toString(),
					e);
		}

		// DomaのクエリオブジェクトにSQLをセットする
		SqlSelectQuery selectQuery = new SqlSelectQuery();
		selectQuery.setConfig(DomaConfig.singleton());
		selectQuery.setCallerClassName(getClass().getName());
		selectQuery.setCallerMethodName("execute");
		selectQuery.setFetchType(FetchType.LAZY);
		selectQuery.setSqlLogType(SqlLogType.FORMATTED);
		selectQuery.setSqlNode(parser.parse());

		selectQuery.addParameter("tenantId", String.class, tenantId);
		selectQuery.addParameter("visibleOnly", String.class, visibleOnly);
		selectQuery.addParameter("itemCategoryDiv", String.class, itemCategory);
		selectQuery.addParameter("exhibitCd", String.class, exhibitCd);
		selectQuery.addParameter("catalogCd", String.class, catalogCd);
		selectQuery.addParameter("userCd", String.class, userCd);
		selectQuery.addParameter("userActiveItemFlg", String.class,
				userActiveItemFlg);
		selectQuery.addParameter("itemCategories", List.class, categoryList);

		selectQuery.prepare();

		// 検索処理を実行
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		List<Map<String, Object>> resultList = tm.required(() -> {
			SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
					selectQuery,
					new MapResultListHandler(MapKeyNamingType.NONE));
			return command.execute();
		});

		return resultList;

	}

	/**
	 * sqlFileDirPathを取得します。
	 * 
	 * @return sqlFileDirPath
	 */
	public String getSqlFileDirPath() {
		if (sqlFileDirPath == null) {
			sqlFileDirPath = "";
		}
		return sqlFileDirPath;
	}

	/**
	 * sqlFileDirPathを設定します。
	 * 
	 * @param sqlFileDirPath
	 */
	public void setSqlFileDirPath(String sqlFileDirPath) {
		this.sqlFileDirPath = sqlFileDirPath;
	}

	/**
	 * sqlFilePathを取得します。
	 * 
	 * @return sqlFilePath
	 */
	public String getSqlFilePath() {
		return sqlFilePath;
	}

	/**
	 * sqlFilePathを設定します。
	 * 
	 * @param sqlFilePath
	 */
	public void setSqlFilePath(String sqlFilePath) {
		this.sqlFilePath = sqlFilePath;
	}

	/**
	 * tenantIdを取得します。
	 * 
	 * @return tenantId
	 */
	public String getTenantId() {
		return tenantId;
	}

	/**
	 * tenantIdを設定します。
	 * 
	 * @param tenantId
	 */
	public void setTenantId(String tenantId) {
		this.tenantId = tenantId;
	}

	/**
	 * visibleOnlyを取得します。
	 * 
	 * @return visibleOnly
	 */
	public String getVisibleOnly() {
		return visibleOnly;
	}

	/**
	 * visibleOnlyを設定します。
	 * 
	 * @param visibleOnly
	 */
	public void setVisibleOnly(String visibleOnly) {
		this.visibleOnly = visibleOnly;
	}

	/**
	 * categoriesを取得します。
	 * 
	 * @return categories
	 */
	public Map<String, Object> getCategories() {
		return categories;
	}

	/**
	 * categoriesを設定します。
	 * 
	 * @param categories
	 */
	public void setCategories(Map<String, Object> categories) {
		this.categories = categories;
	}

	/**
	 * exhibitCdを取得します。
	 * 
	 * @return exhibitCd
	 */
	public String getExhibitCd() {
		return exhibitCd;
	}

	/**
	 * exhibitCdを設定します。
	 * 
	 * @param exhibitCd
	 */
	public void setExhibitCd(String exhibitCd) {
		this.exhibitCd = exhibitCd;
	}

	/**
	 * commonを取得します。
	 * 
	 * @return common
	 */
	public SystemSetting getCommon() {
		return common;
	}

	/**
	 * commonを設定します。
	 * 
	 * @param common
	 */
	public void setCommon(SystemSetting common) {
		this.common = common;
	}

	/**
	 * userActiveItemFlgを取得します。
	 * 
	 * @return userActiveItemFlg
	 */
	public String getUserActiveItemFlg() {
		return userActiveItemFlg;
	}

	/**
	 * userActiveItemFlgを設定します。
	 * 
	 * @param userActiveItemFlg
	 */
	public void setUserActiveItemFlg(String userActiveItemFlg) {
		this.userActiveItemFlg = userActiveItemFlg;
	}

	/**
	 * userCdを取得します。
	 * 
	 * @return userCd
	 */
	public String getUserCd() {
		return userCd;
	}

	/**
	 * userCdを設定します。
	 * 
	 * @param userCd
	 */
	public void setUserCd(String userCd) {
		this.userCd = userCd;
	}

	/**
	 * catalogCdを取得します。
	 * 
	 * @return catalogCd
	 */
	public String getCatalogCd() {
		return catalogCd;
	}

	/**
	 * catalogCdを設定します。
	 * 
	 * @param catalogCd
	 */
	public void setCatalogCd(String catalogCd) {
		this.catalogCd = catalogCd;
	}

}
