package jp.ill.photon.module.item;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.seasar.doma.MapKeyNamingType;
import org.seasar.doma.internal.jdbc.command.MapResultListHandler;
import org.seasar.doma.jdbc.command.SelectCommand;
import org.seasar.doma.jdbc.query.SqlSelectQuery;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.doma.SqlSelectQueryFactory;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.MapParam;
import jp.ill.photon.model.SearchForm;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.JudgeExecutableModuleUtil;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.SQLUtil;
import jp.ill.photon.util.StringUtil;

/**
 * [user_item_select] 商品検索処理を実行するモジュール.
 * 
 * <p>
 * ユーザーのログイン状態と実行するSQLの情報をパラメータとして受け取り、検索結果を返すモジュール。
 * sql_file_selectモジュールと同じ値を作成して返す。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 * <dl>
 * <dt>cnt</dt>
 * <dd>limit前のデータ件数。(SQLから"count"という名称で返された値を使用する)</dd>
 * <dt>list</dt>
 * <dd>検索結果リスト。MapのListを返却する。</dd>
 * <dt>first</dt>
 * <dd>検索結果の最初の1データ。Mapを返却する。</dd>
 * </dl>
 * </p>
 * 
 * @author h_tanaka
 * @see jp.ill.photon.module.db.SqlFileSelectModule
 *
 */
public class UserItemSelectModule implements PhotonModule {

	/**
	 * SQLタイプ定義
	 * 
	 * @author h_tanaka
	 *
	 */
	public static class SqlTypes {
		public static final String TEXT = "text";
		public static final String FILE = "file";
	}

	/**
	 * SQLファイルで使用するパラメータマップ用モジュールパラメータ.
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 * 
	 * <pre>
	 * {
	 *   "tenantId": {
	 *     "type": "param",
	 *     "val": "_init.tenant_id"
	 *   },
	 *   "formCd": {
	 *     "type": "static",
	 *     "val": "test_val"
	 *   },
	 *   "check_list": {
	 *     "type": "param",
	 *     "val": "checks",
	 *     "data_type": "object_list"
	 *   }
	 * }
	 * </pre>
	 * 
	 * </dd>
	 * </dl>
	 * </p>
	 * 
	 * <p>
	 * 上記の内容を指定すると、SQLファイル内で「tenantId」「formCd」「check_list」というパラメータを使用することができる。<br/>
	 * 特に指定しない場合、パラメータはString型として解釈されるが、「data_type」キーでデータ型を指定することでマップやリストとして<br/>
	 * 使用できるようになる。
	 * </p>
	 * 
	 * <p>
	 * data_typeで指定可能な値：<br/>
	 * <dl>
	 * <dt>text</dt>
	 * <dd>文字列</dd>
	 * <dt>number</dt>
	 * <dd>数値</dd>
	 * <dt>text_list</dt>
	 * <dd>List&lt;String&gt;型</dd>
	 * <dt>object_list</dt>
	 * <dd>List&lt;Object&gt;型</dd>
	 * <dt>text_map</dt>
	 * <dd>Map&lt;String, String&gt;型</dd>
	 * <dt>object_map</dt>
	 * <dd>Map&lt;String, Object&gt;型</dd>
	 * </dl>
	 * </p>
	 * 
	 */
	@ModuleParam(required = true, domainObject = true)
	private SearchForm searchForm;

	/**
	 * SQLファイルのフォルダパス指定用モジュールパラメータ.
	 * 
	 * <p>
	 * 通常はDTOの「common.systemsetting.exSqlFileDir.note」を指定することで、<br/>
	 * システム設定のSQL保存ディレクトリからSQLファイルを読み込むことができる。
	 * </p>
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>dto</dd>
	 * <dt>transfer_val</dt>
	 * <dd>common.systemsetting.exSqlFileDir.note</dd>
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = false)
	private String sqlFileDirPath;

	/**
	 * 対象ファイルの文字コード指定用モジュールパラメータ.
	 * 
	 * <p>
	 * 指定しなかった場合は「UTF-8」で読み込む。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String encoding;

	/**
	 * 対象ファイルの改行コード指定用モジュールパラメータ.
	 * 
	 * <p>
	 * 指定しなかった場合は「\n」で読み込む。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String lineBreak;

	/**
	 * ログイン済用メインSQLタイプ指定用モジュールパラメータ.
	 * 
	 * <p>
	 * メインSQL指定を「テキスト（text）」で行うか、「ファイルパス（file）」で行うかを指定する。<br/>
	 * 指定されない場合はデフォルトで「テキスト」を使用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String mainSqlTypeForLogin;

	/**
	 * ログイン済用メインSQLファイルパス指定用モジュールパラメータ.
	 * 
	 * <p>
	 * SQLテキストまたはsqlFileDirPathからの相対パスを指定する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String mainSqlForLogin;

	/**
	 * 未ログイン用メインSQLファイルタイプ指定用モジュールパラメータ.
	 * 
	 * <p>
	 * メインSQL指定を「テキスト（text）」で行うか、「ファイルパス（file）」で行うかを指定する。<br/>
	 * 指定されない場合はデフォルトで「テキスト」を使用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String mainSqlTypeForNoLogin;

	/**
	 * 未ログイン用メインSQLファイルパス指定用モジュールパラメータ.
	 * 
	 * <p>
	 * SQLテキストまたはsqlFileDirPathからの相対パスを指定する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String mainSqlForNoLogin;

	/**
	 * メインSQLに組み込むサブSQL指定用モジュールパラメータ.
	 * 
	 * <p>
	 * 置換用テキストをキーとして持つJSONマップを受け取る。<br/>
	 * 置換用テキストがメインSQLに含まれる場合、サブSQLで置換用テキストが置換される。<br/>
	 * サブSQLはファイルパスまたはテキストのどちらかで指定できる。
	 * </p>
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 * 
	 * <pre>
	 * {
	 *   "/xx shohin_price_sql  x/": {
	 *     "type": "file",
	 *     "val": "shohin_price.sql"
	 *   },
	 *   "/xx category_sql x/": {
	 *     "type": "text",
	 *     "val": "select category from item_category"
	 *   }
	 * }
	 * </pre>
	 * 
	 * </dd>
	 * </dl>
	 * </p>
	 * 
	 */
	@ModuleParam(required = false)
	private MapParam subSqlMap;

	/**
	 * ログイン済みかどうかを判断するための式を受け取る.
	 * 
	 * <p>
	 * この値が空もしくは式の結果がtrueの場合、ログイン済みSQLが実行される。<br/>
	 * それ以外の場合は未ログインSQLが実行される。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String authCheckCondLayout;

	/**
	 * ログイン済みかどうかを判断するための式で使用するパレメータノマップを受け取る.
	 * 
	 */
	@ModuleParam(required = false)
	private MapParam authCheckTargetValues;

	/**
	 * 新規トランザクション開始指定用モジュールパラメータ.
	 * 
	 * <p>
	 * このSQLを実行する際にrequiredNewで別トランザクションを開始するかを指定する<br/>
	 * （1:開始する、1以外:開始しない）
	 * </p>
	 */
	@ModuleParam(required = false)
	private String requireNewTransaction;

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(UserItemSelectModule.class);

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		ModuleResult result = new ModuleResult();
		List<Map<String, Object>> resultList = new ArrayList<>();

		// パラメータを解析し、実行するSQLを生成する。

		// ログイン済みかどうかの判定

		boolean isLoggedIn = true;
		String authCheckCondLayout = getAuthCheckCondLayout();
		if (authCheckCondLayout != null && authCheckCondLayout.length() > 0) {
			Map<String, Object> authCheckTargetValues = getAuthCheckTargetValues()
					.getParamMap();
			isLoggedIn = JudgeExecutableModuleUtil.isExecutable(
					authCheckCondLayout, authCheckTargetValues,
					context.getDto());
		}

		String mainSqlText = getMainSqlText(isLoggedIn);

		if (StringUtil.defaultString(mainSqlText, "") != "") {

			mainSqlText = mergeSubSql(mainSqlText, getSubSqlMap().getParamMap(),
					context.getDto());

			SqlSelectQuery selectQuery = SqlSelectQueryFactory.newInstance()
					.createSelectQuery(getSearchForm().getParamMap(),
							context.getDto(), mainSqlText);

			selectQuery.prepare();

			// 検索処理を実行
			if (getRequireNewTransaction().equals("1")) {
				TransactionManager tm = DomaConfig.singleton()
						.getTransactionManager();
				resultList = tm.requiresNew(() -> {
					SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
							selectQuery,
							new MapResultListHandler(MapKeyNamingType.NONE));
					return command.execute();
				});
			} else {
				SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
						selectQuery,
						new MapResultListHandler(MapKeyNamingType.NONE));
				resultList = command.execute();
			}

			// 値のコンバート
			SQLUtil.convertDbValues(resultList);
		}

		// 結果オブジェクト生成
		result.getReturnData().put("list", resultList);
		if (resultList.size() > 0) {
			result.getReturnData().put("first", resultList.get(0));
		} else {
			result.getReturnData().put("first", null);
		}

		int cnt = 0;
		if (resultList != null && !resultList.isEmpty()
				&& resultList.get(0).get("count") != null) {
			cnt = Integer
					.parseInt(String.valueOf(resultList.get(0).get("count")));
		}

		logger.info(String.format("---->[count]:[%d]", ((cnt == 0)
				? ((resultList != null) ? resultList.size() : 0) : cnt)));

		result.getReturnData().put("cnt", cnt);

		return result;
	}

	/**
	 * メインSQLを取得する.
	 * 
	 * @param isLoggedIn
	 * @return
	 */
	protected String getMainSqlText(boolean isLoggedIn) {
		String mainSql = null;
		String mainSqlType = null;
		if (isLoggedIn) {
			mainSql = getMainSqlForLogin();
			mainSqlType = getMainSqlTypeForLogin();
		} else {
			mainSql = getMainSqlForNoLogin();
			mainSqlType = getMainSqlTypeForNoLogin();
		}

		// メインSQLテキスト取得
		String mainSqlText = null;
		if (mainSqlType != null && mainSqlType.length() > 0
				&& mainSqlType.equals(SqlTypes.FILE) && mainSql != null
				&& mainSql.length() > 0) {
			mainSqlText = slurpFile(getSqlFileDirPath(), mainSql);
		} else {
			mainSqlText = mainSql;
		}

		return mainSqlText;
	}

	/**
	 * メインSQLとサブSQLをマージして返す.
	 * 
	 * @param mainSqlText
	 * @param subSqlMap
	 * @return
	 */
	protected String mergeSubSql(	String mainSqlText,
									Map<String, Object> subSqlMap,
									ActionDto dto) {

		if (subSqlMap == null || subSqlMap.size() == 0) {
			return mainSqlText;
		}

		Map<String, Object> mp = null;
		String replaceValue = null;
		for (Map.Entry<String, Object> entry : subSqlMap.entrySet()) {
			mp = (Map<String, Object>) entry.getValue();
			if (mp.getOrDefault("data_type", SqlTypes.TEXT)
					.equals(SqlTypes.FILE)) {
				replaceValue = slurpFile(getSqlFileDirPath(),
						StringUtil.defaultString(
								ParamUtil.getParamObjectValueByType(mp, dto),
								""));
			} else {
				replaceValue = StringUtil.defaultString(
						ParamUtil.getParamObjectValueByType(mp, dto), "");
			}
			mainSqlText = mainSqlText.replace(entry.getKey(), replaceValue);
		}

		return mainSqlText;
	}

	/**
	 * ファイル読み込み
	 * 
	 * @param dirPath
	 * @param filePath
	 * @return
	 */
	protected String slurpFile(String dirPath, String filePath) {
		if (dirPath == null || filePath == null) {
			return null;
		}

		// ファイル存在チェック
		Path path = Paths.get(dirPath, filePath);
		if (!Files.exists(path)) {
			logger.error("ファイルが存在しません:" + path.toString(), null);
			return null;
		}

		String encoding = getEncoding();
		if (encoding == null || encoding.isEmpty()) {
			encoding = "UTF-8";
		}

		String lineBreak = getLineBreak();
		if (lineBreak == null || lineBreak.isEmpty()) {
			lineBreak = "\n";
		}

		// ファイルコンテンツを一括読み込み
		String contents;
		try {
			contents = Files.readAllLines(path, Charset.forName(encoding))
					.stream().collect(Collectors.joining(lineBreak));
		} catch (IOException e) {
			logger.error("ファイルの読込に失敗しました:" + path.toString(), e);
			return null;
		}

		return contents;
	}

	public SearchForm getSearchForm() {
		return searchForm;
	}

	public void setSearchForm(SearchForm searchForm) {
		this.searchForm = searchForm;
	}

	public String getSqlFileDirPath() {
		if (sqlFileDirPath == null) {
			sqlFileDirPath = "";
		}
		return sqlFileDirPath;
	}

	public void setSqlFileDirPath(String sqlFileDirPath) {
		this.sqlFileDirPath = sqlFileDirPath;
	}

	public String getRequireNewTransaction() {
		if (requireNewTransaction == null) {
			requireNewTransaction = "";
		}
		return requireNewTransaction;
	}

	public void setRequireNewTransaction(String requireNewTransaction) {
		this.requireNewTransaction = requireNewTransaction;
	}

	public MapParam getSubSqlMap() {
		if (subSqlMap == null) {
			subSqlMap = new MapParam();
		}
		return subSqlMap;
	}

	public void setSubSqlMap(MapParam subSqlMap) {
		this.subSqlMap = subSqlMap;
	}

	public String getAuthCheckCondLayout() {
		return authCheckCondLayout;
	}

	public void setAuthCheckCondLayout(String authCheckCondLayout) {
		this.authCheckCondLayout = authCheckCondLayout;
	}

	public MapParam getAuthCheckTargetValues() {
		if (authCheckTargetValues == null) {
			authCheckTargetValues = new MapParam();
		}
		return authCheckTargetValues;
	}

	public void setAuthCheckTargetValues(MapParam authCheckTargetValues) {
		this.authCheckTargetValues = authCheckTargetValues;
	}

	public String getMainSqlTypeForLogin() {
		return mainSqlTypeForLogin;
	}

	public void setMainSqlTypeForLogin(String mainSqlTypeForLogin) {
		this.mainSqlTypeForLogin = mainSqlTypeForLogin;
	}

	public String getMainSqlForLogin() {
		return mainSqlForLogin;
	}

	public void setMainSqlForLogin(String mainSqlForLogin) {
		this.mainSqlForLogin = mainSqlForLogin;
	}

	public String getMainSqlTypeForNoLogin() {
		return mainSqlTypeForNoLogin;
	}

	public void setMainSqlTypeForNoLogin(String mainSqlTypeForNoLogin) {
		this.mainSqlTypeForNoLogin = mainSqlTypeForNoLogin;
	}

	public String getMainSqlForNoLogin() {
		return mainSqlForNoLogin;
	}

	public void setMainSqlForNoLogin(String mainSqlForNoLogin) {
		this.mainSqlForNoLogin = mainSqlForNoLogin;
	}

	public String getEncoding() {
		return encoding;
	}

	public void setEncoding(String encoding) {
		this.encoding = encoding;
	}

	public String getLineBreak() {
		return lineBreak;
	}

	public void setLineBreak(String lineBreak) {
		this.lineBreak = lineBreak;
	}

}
