package jp.ill.photon.module.item;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.seasar.doma.MapKeyNamingType;
import org.seasar.doma.internal.jdbc.command.MapResultListHandler;
import org.seasar.doma.jdbc.command.SelectCommand;
import org.seasar.doma.jdbc.query.SqlSelectQuery;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.doma.SqlSelectQueryFactory;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.SearchForm;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.FileUtil;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringUtil;

/**
 * 検索対象の商品画像一括更新処理モジュール
 *
 * @author m_fukukawa
 *
 */
public class ItemImageLumpUploadBySearchModule implements PhotonModule {

	/* モジュールパラメータ */
	@ModuleParam(required = false)
	private Map<String, Object> common;

	@ModuleParam(required = true)
	private String tenantId; // テナントID

	@ModuleParam(required = true)
	private List<Map<String, Object>> fieldList; // カラム設定

	@ModuleParam(required = false)
	private String selectedItemCd; // 選択された商品

	@ModuleParam(required = true)
	private Map<String, Object> filePathInfoMap; // ファイルパス情報リスト

	@ModuleParam(required = false)
	private String updateUserName; // 登録/更新ユーザ名

	@ModuleParam(required = false)
	private Map<String, Object> after; // モジュール実行後の処理分岐設定

	@ModuleParam(required = false)
	private Map<String, Object> messages; // メッセージ

	@ModuleParam(required = true, domainObject = true)
	private SearchForm searchForm;

	@ModuleParam(required = true)
	private String sqlFileDirPath;

	@ModuleParam(required = true)
	private String sqlFilePath;

	/* クラス内の複数メソッドで使用する変数 */
	private List<String> keyColCdList = new LinkedList<String>() {
		{
			add("item_cd");
		}
	};
	private List<String> updateColCdList = new LinkedList<String>() {
		{
			add("item_image1");
			add("item_image2");
			add("item_image3");
			add("item_image4");
		}
	};

	private Map<String, Object> systemSetting;
	private Map<String, Object> errorMessage;

	private int sqlResult = -1;
	private Map<String, List<String>> messageListMap;
	private static String MSG_KEY = "db_modify";

	/** テーブルコード */
	private static String EDIT_TABLE_CD = "item";

	/** 削除フラグ */
	private static class DeleteFlg {
		/** 未削除 */
		public static final String OFF = "0";
		/** 削除済 */
		public static final String ON = "1";
	}

	@SuppressWarnings("unchecked")
	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		if (!MapUtils.isEmpty(common)) {
			systemSetting = (Map<String, Object>) common.get("systemsetting");
			errorMessage = (Map<String, Object>) common.get("mesg");
		}

		doModifyLump(getFilePathInfoMap(), context.getDto());

		ModuleResult result = new ModuleResult();

		result.getReturnData().put("sql-result", sqlResult);
		String resultCd = (sqlResult == -1) ? "failed" : "success";
		result.getReturnData().put("result_code", resultCd);
		if (sqlResult == 0) {
			result.setResultCode("success");
			// 完了メッセージセット
			if (!MapUtils.isEmpty(messages)) {
				// TODO: 追加するメッセージをmesgに入るように
				String mes = ParamUtil.getParamStrValueByType(
						messages.get(result.getResultCode()), context.getDto());
				if (!StringUtils.isEmpty(mes)) {
					addMessage(MSG_KEY, mes);
				}
			}
		} else {
			result.setResultCode("failed");
		}
		// モジュールのパラメータ"after"に値がセットされていたとき
		if (after != null) {
			Map<String, Object> mp = (Map<String, Object>) after
					.get(result.getResultCode());
			if (mp != null) {
				result.getReturnData().put("after_params", mp.get("params"));
				result.setNextPath((String) mp.get("next_path"));
				result.setResultType((String) mp.get("result_type"));
			}
		}

		if (!MapUtils.isEmpty(messageListMap)) {
			result.getReturnData().put("messages", messageListMap);
		}

		return result;

	}

	/***
	 *
	 * 画像情報更新（商品を選択して一括更新）
	 *
	 * @throws PhotonModuleException
	 *
	 */
	@SuppressWarnings("rawtypes")
	private void doModifyLump(	Map<String, Object> filePathInfoMap,
								ActionDto dto)
			throws PhotonModuleException {

		SqlSelectQueryFactory selectFactory = SqlSelectQueryFactory
				.newInstance();
		SqlSelectQuery selectQuery = selectFactory.createSelectQueryFromFile(
				getSearchForm().getParamMap(), dto, getSqlFileDirPath(),
				getSqlFilePath());

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		JsonDataDao dao = new JsonDataDaoImpl();
		// トランザクション開始
		sqlResult = tm.required(() -> {

			Map<String, String> baseMap = new HashMap<String, String>() {
				{
					put("item_cd", selectedItemCd);
				}
			};

			// ◆選択されている商品データをDBから取得
			String fields = getSelectFieldStr();
			String data = JsonUtil.mapToJson(baseMap);
			Map<String, Object> selectedItem = dao.getEditTableRow(tenantId,
					EDIT_TABLE_CD, fields, data);

			try {

				// ◆選択されている商品のファイル削除・更新
				deleteAndSaveFiles(selectedItem, filePathInfoMap, dao, dto);

				// ◆選択されている商品データをDBから再取得
				selectedItem = dao.getEditTableRow(tenantId, EDIT_TABLE_CD,
						fields, data);

				// ◆商品コード×商品画像項目数ループし、
				// 選択されている商品の画像項目と
				// リストの商品の画像項目の状況を比較し、下記のように処理する
				// 選択された商品の画像：登録されている / リストの商品：登録されていない → データの商品に基準商品の画像を登録する。
				// 選択された商品の画像：登録されている / リストの商品：登録されている → データの商品に基準商品の画像を登録する。
				// 選択された商品の画像：登録されていない / リストの商品：登録されていない → なにもしない。
				// 選択された商品の画像：登録されていない / リストの商品：登録されている → データの商品の画像を削除する。

				List<Map<String, Object>> fieldList = getFieldList();

				int cnt = -1;
				int limit = 100;
				int offset = 0;
				while (cnt != 0) {
					selectQuery.addParameter("limit", Integer.class, limit);
					selectQuery.addParameter("offset", Integer.class, offset);
					selectQuery.prepare();

					// 検索処理を実行
					SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
							selectQuery,
							new MapResultListHandler(MapKeyNamingType.NONE));
					List<Map<String, Object>> itemList = command.execute();

					// 検索結果が空ではない場合
					if (itemList != null && !itemList.isEmpty()) {
						String itemCd = null;
						for (Map<String, Object> item : itemList) {
							itemCd = StringUtil.defaultString(
									item.getOrDefault("item_cd", ""), "");

							if (itemCd.equals(selectedItemCd)) { // 選択されている商品は、ここでは更新の対象としない。
								continue;
							}

							if (!CollectionUtils.isEmpty(fieldList)) {

								for (Map<String, Object> col : fieldList) {

									// 元のファイルデータを取得

									String fileNameFromList = null;
									String name = (String) col.get("col_cd");
									String fileNameFromBase = (String) selectedItem
											.get(name);

									fileNameFromList = (String) item.get(name);

									if (!StringUtils.isEmpty(fileNameFromBase)
											&& StringUtils.isEmpty(
													fileNameFromList)) {

										// 選択された商品の画像：登録されている / リストの商品：登録されていない
										// データの商品に基準商品の画像を登録する。
										// 選択された商品の画像をコピー
										// ファイル名はリストの商品のものになるようにする
										copyAttachedFile(col, selectedItem,
												item, dao);

									} else if (!StringUtils
											.isEmpty(fileNameFromBase)
											&& !StringUtils.isEmpty(
													fileNameFromList)) {

										// 選択された商品の画像：登録されている / リストの商品：登録されている
										// データの商品に基準商品の画像を登録する。
										// 選択された商品の画像をコピー
										// ファイル名はリストの商品のものになるようにする
										copyAttachedFile(col, selectedItem,
												item, dao);

									} else if (StringUtils
											.isEmpty(fileNameFromBase)
											&& StringUtils.isEmpty(
													fileNameFromList)) {

										// 選択された商品の画像：登録されていない / リストの商品：登録されていない
										// なにもしない。

									} else if (StringUtils
											.isEmpty(fileNameFromBase)
											&& !StringUtils.isEmpty(
													fileNameFromList)) {

										// 選択された商品の画像：登録されていない / リストの商品：登録されている
										// データの商品の画像を削除する。
										deleteAttachedFile(col, item,
												fileNameFromList, dao);

									}

								}

							}

						}

						// クエリの結果件数を設定
						cnt = itemList.size();
					} else {
						cnt = 0;
					}

					offset += limit;
				}

			} catch (Exception e) {
				// メッセージ追加
				// TODO: 追加するメッセージをmesgに入るように
				String msg = null;
				if (!MapUtils.isEmpty(errorMessage) && !MapUtils
						.isEmpty((Map) errorMessage.get("commonErrMes11"))) {
					msg = (String) ((Map) errorMessage.get("commonErrMes11"))
							.get("format");
				}
				addMessage(MSG_KEY, msg);
				e.printStackTrace();
				tm.setRollbackOnly();
			}

			return 0;

		});

	}

	/**
	 *
	 * SELECTフィールドを作成する
	 *
	 * @return
	 */
	private String getSelectFieldStr() {
		StringJoiner sjField = new StringJoiner(",");
		for (String s : updateColCdList) {
			sjField.add(String.format("val->>$$%s$$ AS %s", s, s));
		}
		for (String s : keyColCdList) {
			sjField.add(String.format("val->>$$%s$$ AS %s", s, s));
		}
		return sjField.toString();
	}

	/**
	 *
	 * エラーメッセージリストにエラーメッセージ追加
	 *
	 * @param key
	 * @param message
	 *
	 */
	private void addMessage(String key, String message) {

		if (messageListMap == null) {
			messageListMap = new HashMap<String, List<String>>();
		}

		List<String> list = messageListMap.get(key);
		if (list == null) {
			list = new ArrayList<String>();
		}
		list.add(message);

		messageListMap.put(key, list);

	}

	/**
	 * ファイル削除＆登録
	 *
	 * @param item 商品マスタの元データ
	 *
	 * @throws Exception
	 *
	 */
	private void deleteAndSaveFiles(Map<String, Object> item,
									Map<String, Object> filePathInfoMap,
									JsonDataDao dao,
									ActionDto dto)
			throws Exception {

		List<Map<String, Object>> fieldList = getFieldList();
		if (!CollectionUtils.isEmpty(fieldList)) {

			String colCd = null;
			String paramKey = null;
			String fileNameFromDB = null;
			Map<String, String> filePathInfo = null;
			String itemCd = StringUtil
					.defaultString(item.getOrDefault("item_cd", ""), "");
			for (Map<String, Object> field : fieldList) {

				if (item != null) {
					colCd = StringUtil.defaultString(
							field.getOrDefault("col_cd", ""), "");
					paramKey = itemCd + "_" + colCd;
					filePathInfo = (Map<String, String>) filePathInfoMap
							.getOrDefault(paramKey, null);
					if (filePathInfo == null) {
						continue;
					}

					fileNameFromDB = StringUtil
							.defaultString(item.getOrDefault(colCd, ""), "");

					// ファイルの削除フラグが設定されている場合、ファイルを削除
					String delFlg = filePathInfo.getOrDefault("del_flg", "0");
					if (DeleteFlg.ON.equals(delFlg)) {
						deleteAttachedFile(field, item, fileNameFromDB, dao);
					}

					// アップロードされたファイルを保存する
					saveAttachedFile(itemCd, field, item, filePathInfo, dto,
							dao);
				}

			}

		}
	}

	/**
	 *
	 * データ1件の指定列に添付されているファイルを削除する
	 *
	 * @param col カラム定義
	 * @param item 商品マスタの元データ
	 * @param fileNameFromDB ファイル名
	 *
	 * @throws Exception
	 *
	 */
	private void deleteAttachedFile(Map<String, Object> col,
									Map<String, Object> item,
									String fileNameFromDB,
									JsonDataDao dao)
			throws Exception {

		// ファイルユーティリティクラスのインスタンス生成
		FileUtil fileUtil = new FileUtil();

		// ルートパスを取得
		String rootPath = getRootPathFromColumnInfo(col, item, dao);

		// ファイル名updateのクエリ作成
		// dataの文字列作成
		// 削除なので、項目名の値をクリアする固定のSQLを実行する
		Map<String, Object> mp = getKeyJson(item);
		mp.put((String) col.get("col_cd"), "");
		String data = JsonUtil.mapToJson(mp);
		dao.updateEditTableData(tenantId, EDIT_TABLE_CD, getUpdateUserName(),
				data);

		if (!StringUtils.isEmpty(rootPath)
				&& !StringUtils.isEmpty(fileNameFromDB)) {
			// 実ファイル削除
			fileUtil.delFile(rootPath, fileNameFromDB);
		}

	}

	/**
	 *
	 * ファイル名を生成する
	 *
	 * @param file 元ファイル名
	 * @param col カラム定義
	 * @param item 商品マスタの元データ
	 *
	 * @return
	 *
	 */
	private String getNewFileName(	String file,
									Map<String, Object> col,
									Map<String, Object> item,
									JsonDataDao dao) {

		String fileNameDir = "";
		String ret = file;

		String colCd = (String) col.get("col_cd");
		String fileNameSelectQuery = (String) col.get("file_name_select_query");

		// ファイル名を生成する。
		// ファイル名用のクエリが設定されているときは、SELECTした結果をファイル名にする。
		if (!StringUtils.isEmpty(fileNameSelectQuery)) {
			fileNameDir = getFileNameByColumnInfo(fileNameSelectQuery, colCd,
					item, dao);
			ret = fileNameDir + "." + getSuffix(file);
		}

		return ret;

	}

	/**
	 *
	 * 選択された商品に紐付いているファイルをよその商品にコピー
	 *
	 * @param col カラム定義
	 * @param orgData 商品マスタの元データ
	 * @param orgData 商品マスタの元データ
	 *
	 * @throws Exception
	 *
	 */
	private void copyAttachedFile(	Map<String, Object> col,
									Map<String, Object> srcData,
									Map<String, Object> dstData,
									JsonDataDao dao)
			throws Exception {

		// ルートパスを取得（元データの情報から）
		String rootPath = getRootPathFromColumnInfo(col, srcData, dao);
		String colCd = (String) col.get("col_cd");
		String srcFileName = (String) srcData.get(colCd);

		// 更新先の商品のファイル名生成
		String dstFileName = getNewFileName(srcFileName, col, dstData, dao);

		// DBにファイル名を登録
		Map<String, Object> mp = getKeyJson(dstData);
		mp.put(colCd, dstFileName);
		String data = JsonUtil.mapToJson(mp);
		dao.updateEditTableData(tenantId, EDIT_TABLE_CD, getUpdateUserName(),
				data);

		// コピー
		Path srcFilePath = Paths.get(rootPath + File.separator + srcFileName);
		Path dstFilePath = Paths.get(rootPath + File.separator + dstFileName);
		Files.copy(srcFilePath, dstFilePath,
				StandardCopyOption.REPLACE_EXISTING);

	}

	/**
	 *
	 * アップロードされたファイルを添付する
	 *
	 * @param itemCd 商品コード
	 * @param col カラム定義
	 * @param item 商品マスタの元データ
	 *
	 * @throws Exception
	 *
	 */
	private void saveAttachedFile(	String itemCd,
									Map<String, Object> col,
									Map<String, Object> item,
									Map<String, String> filePathInfo,
									ActionDto dto,
									JsonDataDao dao)
			throws Exception {

		// ルートパスを取得
		String rootPath = getRootPathFromColumnInfo(col, item, dao);

		String reqKey = (String) col.get("col_cd");
		String colCd = (String) col.get("col_cd");
		String dataSrcItemName = itemCd + "_" + reqKey;

		// オリジナルのファイル名取得
		String file = filePathInfo.getOrDefault("file_name", "");

		if (!StringUtils.isEmpty(file)) {

			String updateFileName = getNewFileName(file, col, item, dao);

			// DBにファイル名を登録
			Map<String, Object> mp = getKeyJson(item);
			mp.put(colCd, updateFileName);
			String data = JsonUtil.mapToJson(mp);
			dao.updateEditTableData(tenantId, EDIT_TABLE_CD,
					getUpdateUserName(), data);

			// ファイル本体を保存
			if (filePathInfo.getOrDefault("is_dnd", "0").equals("1")) {

				// 値から
				FileUtil.outputFileBase64(
						filePathInfo.getOrDefault("image", ""), rootPath,
						updateFileName);

			} else {

				// ファイルが設定されていれば実施する
				Path srcFilePath = getFileFromRequestParameter(
						filePathInfo.getOrDefault("path", ""), dto);
				if (srcFilePath != null) {

					// ファイルをコピーする
					Path dstFilePath = Paths
							.get(rootPath + File.separator + updateFileName);
					Files.copy(srcFilePath, dstFilePath,
							StandardCopyOption.REPLACE_EXISTING);

				}

			}
		}

	}

	/**
	 *
	 * キー情報を作成
	 *
	 * @param orgData
	 *
	 * @return キー情報
	 *
	 */
	private Map<String, Object> getKeyJson(Map<String, Object> orgData) {
		return new HashMap<String, Object>() {
			{
				for (String key : keyColCdList) {
					put(key, (String) orgData.get(key));
				}
			}
		};
	}

	/**
	 *
	 * カラム設定情報から、ファイルパスを取得
	 *
	 * @param col カラム定義
	 * @param item 商品マスタの元データ
	 *
	 * @return ファイルパス
	 *
	 */
	@SuppressWarnings("rawtypes")
	private String getRootPathFromColumnInfo(	Map<String, Object> col,
												Map<String, Object> item,
												JsonDataDao dao) {

		String dir = (String) col.get("file_store_path_system_setting_id");
		String select = (String) col.get("sub_dir_name_select_query");
		String fileDir = "";

		if (!StringUtils.isEmpty(select)) {
			fileDir = getFileNameByColumnInfo(select,
					(String) col.get("col_cd"), item, dao);
		}

		StringJoiner sj = new StringJoiner(File.separator);

		String value = null;
		if (!MapUtils.isEmpty(systemSetting)
				&& !MapUtils.isEmpty((Map) systemSetting.get(dir))) {
			value = (String) ((Map) systemSetting.get(dir)).get("note");
		}
		sj.add(value);

		if (!StringUtils.isEmpty(fileDir)) {
			sj.add(fileDir);
		}

		return sj.toString();

	}

	/**
	 * 対象データのcolNameに格納されているファイル名を取得する
	 *
	 * @param valName
	 * @param dispName
	 * @param item 商品マスタの元データ
	 * @return
	 */
	private String getFileNameByColumnInfo(	String valName,
											String dispName,
											Map<String, Object> item,
											JsonDataDao dao) {
		String selectQuery = valName + " AS " + dispName;
		Map<String, Object> key = getKeyJson(item);
		Map<String, Object> mp = dao.getEditTableRow(tenantId, EDIT_TABLE_CD,
				selectQuery, JsonUtil.mapToJson(key));
		if (mp != null && !mp.isEmpty()) {
			return (String) mp.get(dispName);
		}
		return null;
	}

	/**
	 *
	 * ファイル名から拡張子を返します。
	 *
	 * @param fileName ファイル名
	 *
	 * @return ファイルの拡張子
	 *
	 */
	private String getSuffix(String fileName) {
		if (fileName == null) {
			return null;
		}
		int point = fileName.lastIndexOf(".");
		if (point != -1) {
			return fileName.substring(point + 1);
		}
		return fileName;
	}

	/**
	 *
	 * リスエスト情報からファイルを取り出す
	 *
	 * @param key リクエスト情報のキー名
	 *
	 * @return ファイルパス
	 *
	 */
	private Path getFileFromRequestParameter(String file, ActionDto dto) {
		if (StringUtils.isEmpty(file)) {
			return null;
		}
		Path srcPath = Paths.get(file);
		if (!Files.exists(srcPath)) {
			return null;
		}
		return srcPath;
	}

	/**
	 * commonを取得します。
	 *
	 * @return common
	 */
	public Map<String, Object> getCommon() {
		return common;
	}

	/**
	 * commonを設定します。
	 *
	 * @param common
	 */
	public void setCommon(Map<String, Object> common) {
		this.common = common;
	}

	/**
	 * tenantIdを取得します。
	 *
	 * @return tenantId
	 */
	public String getTenantId() {
		return tenantId;
	}

	/**
	 * tenantIdを設定します。
	 *
	 * @param tenantId
	 */
	public void setTenantId(String tenantId) {
		this.tenantId = tenantId;
	}

	/**
	 * afterを取得します。
	 *
	 * @return after
	 */
	public Map<String, Object> getAfter() {
		return after;
	}

	/**
	 * afterを設定します。
	 *
	 * @param after
	 */
	public void setAfter(Map<String, Object> after) {
		this.after = after;
	}

	/**
	 * messagesを取得します。
	 *
	 * @return messages
	 */
	public Map<String, Object> getMessages() {
		return messages;
	}

	/**
	 * messagesを設定します。
	 *
	 * @param messages
	 */
	public void setMessages(Map<String, Object> messages) {
		this.messages = messages;
	}

	public List<Map<String, Object>> getFieldList() {
		return fieldList;
	}

	public void setFieldList(List<Map<String, Object>> fieldList) {
		this.fieldList = fieldList;
	}

	public String getUpdateUserName() {
		return updateUserName;
	}

	public void setUpdateUserName(String updateUserName) {
		this.updateUserName = updateUserName;
	}

	public Map<String, Object> getFilePathInfoMap() {
		return filePathInfoMap;
	}

	public void setFilePathInfoMap(Map<String, Object> filePathInfoMap) {
		this.filePathInfoMap = filePathInfoMap;
	}

	public String getSelectedItemCd() {
		return selectedItemCd;
	}

	public void setSelectedItemCd(String selectedItemCd) {
		this.selectedItemCd = selectedItemCd;
	}

	public SearchForm getSearchForm() {
		return searchForm;
	}

	public void setSearchForm(SearchForm searchForm) {
		this.searchForm = searchForm;
	}

	public String getSqlFileDirPath() {
		return sqlFileDirPath;
	}

	public void setSqlFileDirPath(String sqlFileDirPath) {
		this.sqlFileDirPath = sqlFileDirPath;
	}

	public String getSqlFilePath() {
		return sqlFilePath;
	}

	public void setSqlFilePath(String sqlFilePath) {
		this.sqlFilePath = sqlFilePath;
	}

}
