package jp.ill.photon.module.freepage;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.annotation.ModuleVersion;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.DateUtil;
import jp.ill.photon.util.JsonUtil;

/**
 * 
 * 入力値のフォルダが、自由ページのコンテンツとして登録されているか判定。<br />
 * 登録済みのコンテンツフォルダが存在しているかも判定する<br />
 * <br />
 * 入力: <br/>
 * freePageList: 登録されている自由ページの一覧<br/>
 * singleList: 個別用一覧(名前の一覧)(List{@literal<String>})<br/>
 * embedList: 埋め込み用一覧(名前の一覧)(List{@literal<String>})<br/>
 * <br/>
 * 戻り値: <br />
 * insertFreePageList: 新規追加自由ページ(ArrayList{@literal<Object>})<br />
 * deleteFreePageList: 削除対象自由ページ(ArrayList{@literal<Object>})
 */
@ModuleVersion("1.0.0")
public class CheckFreePageContentsModule implements PhotonModule {

	/** 登録済みの自由ページ一覧 */
	@ModuleParam(required = false)
	private List<Map<String, Object>> freePageList;

	/** 個別用一覧 */
	@ModuleParam(required = false)
	private List<String> singleList;

	/** 埋め込み用一覧 */
	@ModuleParam(required = false)
	private List<String> embedList;

	/* コンテンツフラグ */
	/** コンテンツフラグ 単体 */
	private final String CONTENTS_FLG_SINGLE = "0";
	/** コンテンツフラグ 埋め込み */
	private final String CONTENTS_FLG_EMBED = "1";

	/** デフォルト 表示区分 */
	private final String DEFAULT_DISP_FLG = "0"; // 0:非表示 1:表示

	/** デフォルト オープン区分 */
	private final String DEFAULT_OPEN_FLG = "0"; // 0:非オープン 1:オープン

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		// 返却用配列
		ArrayList<Object> insertFreePageList = new ArrayList<Object>();
		ArrayList<Object> deleteFreePageList = new ArrayList<Object>();

		/**
		 *  登録済みの自由ページ一覧の内容が、個別用・埋め込み用一覧に存在するか判定する
		 *  登録済みの自由ページ一覧に有り、個別用・埋め込み用一覧に無しの場合は削除対象
		 *  登録済みの自由ページ一覧に無し、個別用・埋め込み用一覧に有りの場合は新規登録対象とする
		 */
		// 削除対象作成
		for (Map<String, Object> freePage : freePageList) {
			if (CONTENTS_FLG_SINGLE.equals(freePage.get("contents_flg"))) {
				// 単体用リストから検索
				if (singleList != null && singleList.contains(freePage.get("contents_path"))) {
					// 一覧に存在する場合、一覧から削除
					singleList.remove(freePage.get("contents_path"));

				} else {
					// 一覧に存在しない場合、削除対象リストへ追加する
					deleteFreePageList.add(makeDeleteParam(freePage));

				}

			} else if (CONTENTS_FLG_EMBED.equals(freePage.get("contents_flg"))) {
				// 埋め込み用リストから検索
				if (embedList != null && embedList.contains(freePage.get("contents_path"))) {
					// 一覧に存在する場合、一覧から削除
					embedList.remove(freePage.get("contents_path"));

				} else {
					// 一覧に存在しない場合、削除対象リストへ追加する
					deleteFreePageList.add(makeDeleteParam(freePage));

				}
			}
		}

		// 新規追加対象作成
		if (singleList != null) {
			for (String contentsPath : singleList) {
				insertFreePageList.add(makeInsertParam(contentsPath, CONTENTS_FLG_SINGLE));
			}
		}
		if (embedList != null) {
			for (String contentsPath : embedList) {
				insertFreePageList.add(makeInsertParam(contentsPath, CONTENTS_FLG_EMBED));
			}
		}

		// 結果を返す
		ModuleResult result = new ModuleResult();
		result.getReturnData().put("insertFreePageList", insertFreePageList.size() > 0 ? insertFreePageList.toString() : null);
		result.getReturnData().put("deleteFreePageList", deleteFreePageList);

		return result;
	}

	private String makeDeleteParam(Map<String, Object> freePage) {
		/*
		Map<String, Object> tmpMap = new HashMap<String, Object>();
		tmpMap.put("free_page_id", (String) freePage.get("free_page_id").toString());
		return JsonUtil.mapToJson(tmpMap);
		*/
		return (String) freePage.get("free_page_id").toString();
	}

	private String makeInsertParam(String contentsPath, String contents_flg) {
		String key = contentsPath + contents_flg + DateUtil.getYYYYMMDDHHMIMS();

		Map<String, Object> tmpMap = new HashMap<String, Object>();
		tmpMap.put("free_page_id", key.hashCode());
		tmpMap.put("disp_flg", DEFAULT_DISP_FLG);
		tmpMap.put("open_div", DEFAULT_OPEN_FLG);
		tmpMap.put("contents_flg", contents_flg);
		tmpMap.put("contents_path", contentsPath);
		return JsonUtil.mapToJson(tmpMap);
	}

	/**
	 * 登録済みの自由ページ一覧を取得します。
	 * @return 登録済みの自由ページ一覧
	 */
	public List<Map<String,Object>> getFreePageList() {
	    return freePageList;
	}

	/**
	 * 登録済みの自由ページ一覧を設定します。
	 * @param freePageList 登録済みの自由ページ一覧
	 */
	public void setFreePageList(List<Map<String,Object>> freePageList) {
	    this.freePageList = freePageList;
	}

	/**
	 * 個別用一覧を取得します。
	 * @return 個別用一覧
	 */
	public List<String> getSingleList() {
	    return singleList;
	}

	/**
	 * 個別用一覧を設定します。
	 * @param singleList 個別用一覧
	 */
	public void setSingleList(List<String> singleList) {
	    this.singleList = singleList;
	}

	/**
	 * 埋め込み用一覧を取得します。
	 * @return 埋め込み用一覧
	 */
	public List<String> getEmbedList() {
	    return embedList;
	}

	/**
	 * 埋め込み用一覧を設定します。
	 * @param embedList 埋め込み用一覧
	 */
	public void setEmbedList(List<String> embedList) {
	    this.embedList = embedList;
	}
}
