package jp.ill.photon.module.form;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.thymeleaf.util.StringUtils;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.MapListParam;
import jp.ill.photon.model.MapParam;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringUtil;

/**
 * [form_process] フォーム定義を元にフォーム表示用データを生成するモジュール.
 *
 * <p>
 * フォーム定義、初期値、ユーザー入力値を元にフォーム表示用データを作成する。<br/>
 * 画面初期表示時は初期値を表示し、入力エラー時は入力値を表示する、といった切替を行うことができる。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 * <dl>
 * <dt>fields</dt>
 * <dd>フォームフィールドコードをキーとして、表示データを値（Object型）として持つマップ。</dd>
 * <dt>values</dt>
 * <dd>フォームフィールドコードをキーとして、表示データをString型に変換したものを値として持つマップ。</dd>
 * <dt>has_input</dt>
 * <dd>入力値がある場合は「1」を返し、ない場合は「0」を返す。<br/>
 * ただし、inputsで「not_input="1"」で渡されたパラメータは入力値としてカウントしない。</dd>
 * </dl>
 * </p>
 * 
 * @author h_tanaka
 *
 */
public class FormProcessModule implements PhotonModule {

	/**
	 * ユーザー入力値用モジュールパラメータ.
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 * 
	 * <pre>
	 * {
	 *   "user_name": {
	 *     "type": "param",
	 *     "val": "user_name",
	 *     "remarks": "ユーザーが入力したユーザー名"
	 *   },
	 *   "mail": {
	 *     "type": "param",
	 *     "val": "mail",
	 *     "remarks": "ユーザーが入力したメールアドレス"
	 *   }
	 * }
	 * </pre>
	 * 
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = false)
	private MapParam inputs;

	/**
	 * 初期値用モジュールパラメータ.
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 * 
	 * <pre>
	 * {
	 *   "user_name": {
	 *     "type": "static",
	 *     "val": "ユーザー名",
	 *     "remarks": "ユーザー名初期値"
	 *   },
	 *   "mail": {
	 *     "type": "static",
	 *     "val": "",
	 *     "remarks": "メールアドレス初期値"
	 *   }
	 * }
	 * </pre>
	 * 
	 * </dl>
	 * </p>
	 * 
	 * <p>
	 * inputsに同じキーで値の入ったパラメータがなければ、defaultsの値が設定される。
	 * </p>
	 */
	@ModuleParam(required = false)
	private MapParam defaults;

	/**
	 * カラム設定用モジュールパラメータ.
	 * 
	 * <p>
	 * 下記のSQLが返すリストを渡されることを想定:<br/>
	 * aec20/table/selectColumnSettingsByFormCd.sql
	 * </p>
	 */
	@ModuleParam(required = false)
	private MapListParam columnSettings;

	/**
	 * 入力値のみ使用指定用モジュールパラメータ.
	 * 
	 * <p>
	 * 「1」が渡された場合、inputsで値が入力されていない場合でもdefaultsの値を使用しない。<br/>
	 * 「1」以外が渡された場合はinputsで値が入力されていない場合はdefaultsの値を使用する。<br/>
	 * 入力エラー等で入力画面に返ってきた時に「1」を使用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String useInputsOnly;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		ModuleResult result = new ModuleResult();

		List<Map<String, Object>> columnSettingList = null;
		if (getColumnSettings() != null) {
			columnSettingList = getColumnSettings().getParamList();
		} else {
			columnSettingList = new ArrayList<>();
		}

		Map<String, Object> columnSettingMap = columnSettingList.stream()
				.collect(Collectors.toMap(s -> StringUtil
						.defaultString(s.get("form_field_cd"), ""), s -> s));

		Map<String, Object> inputParamMap = null;
		if (getInputs() != null) {
			inputParamMap = getInputs().getParamMap();
		} else {
			inputParamMap = new HashMap<>();
		}

		Map<String, Object> defaultParamMap = null;
		if (getDefaults() != null) {
			defaultParamMap = getDefaults().getParamMap();
		} else {
			defaultParamMap = new HashMap<>();
		}

		Map<String, Object> preparedInputs = prepareParams(inputParamMap,
				columnSettingMap, context.getDto());
		Map<String, Object> preparedDefaults = prepareParams(defaultParamMap,
				columnSettingMap, context.getDto());

		List<String> noInputKeys = extractNoInputKeys(inputParamMap);

		Map<String, Object> fieldMap = createInputMap(preparedInputs,
				preparedDefaults, isUseInputsOnly());

		boolean hasInput = hasInput(preparedInputs, preparedDefaults,
				noInputKeys);

		Map<String, String> valueMap = extractFormValues(fieldMap);

		result.getReturnData().put("fields", fieldMap);
		result.getReturnData().put("values", valueMap);
		result.getReturnData().put("has_input", hasInput);

		return result;
	}

	/**
	 * 検索条件に入力があるかチェック
	 *
	 * @param srcParams
	 * @return
	 */
	protected boolean hasInput(	Map<String, Object> srcParams,
								Map<String, Object> defaultParams,
								List<String> noInputKeys) {
		boolean hasInput = false;
		boolean isSet = false;

		if (srcParams.containsKey("has_input")) {
			String value = StringUtil.defaultString(srcParams.get("has_input"),
					"");
			if (!StringUtils.isEmpty(value)) {
				hasInput = Boolean.parseBoolean(value);
				isSet = true;
			}
		}

		if (!isSet) {
			for (Map.Entry<String, Object> param : srcParams.entrySet()) {
				String key = param.getKey();
				if (!noInputKeys.contains(key)) {
					String value = StringUtil.defaultString(param.getValue(),
							"");
					if (!StringUtils.isEmpty(value)) {
						if (defaultParams != null
								&& defaultParams.get(key) != null) {
							if (defaultParams.containsKey(key)
									&& !defaultParams.get(key).equals(value)) {
								hasInput = true;
							}
						} else {
							hasInput = true;
						}
					}
				}
			}
		}
		return hasInput;
	}

	/**
	 * has_input用の入力として扱わないキーを抽出
	 *
	 */
	protected List<String> extractNoInputKeys(Map<String, Object> srcParams) {
		List<String> keys = new ArrayList<>();

		String notInput = null;
		for (Map.Entry<String, Object> param : srcParams.entrySet()) {
			notInput = StringUtil.defaultString(
					((Map) param.getValue()).getOrDefault("not_input", ""), "");
			if (notInput.equals("1")) {
				keys.add(param.getKey());
			}
		}

		return keys;
	}

	/**
	 * タイプ別でパラメータ値を取得してマップ化
	 *
	 * @param srcParams
	 * @param dto
	 * @return
	 */
	protected Map<String, Object> prepareParams(Map<String, Object> srcParams,
												Map<String, Object> columnSettingMap,
												ActionDto dto) {
		// パラメータデータを取得
		Map<String, Object> params = new LinkedHashMap<String, Object>();

		String type = null;
		Map<String, Object> column = null;
		String dataInputType = null;
		for (Map.Entry<String, Object> param : srcParams.entrySet()) {

			Object val = ((Map) param.getValue()).get("val");

			type = String.valueOf(((Map) param.getValue()).get("type"));
			if (type.equals("param_auto")) {
				column = (Map<String, Object>) columnSettingMap
						.getOrDefault(param.getKey(), null);
				if (column != null) {
					dataInputType = StringUtil.defaultString(
							column.getOrDefault("data_input_type", ""), "");
					if (Arrays.asList("2", "5").contains(dataInputType)) {
						type = "param_multi";
					} else if ("7".equals(dataInputType)) {
						type = "param_file";
						val = ((Map) param.getValue()).get("val") + ".json";
					} else {
						type = "param";
					}
				}
			}

			params.put(param.getKey(),
					ParamUtil.getParamValueByType(type, val, dto));
		}

		return params;
	}

	/**
	 * フォームのフィールドごとの値を抽出してマップ化して返す
	 *
	 * @param dataMap
	 * @return
	 */
	protected Map<String, String> extractFormValues(Map<String, Object> fieldMap) {

		Map<String, String> newDataMap = new LinkedHashMap<>();
		Object inputValue = null;
		String value = null;
		for (Map.Entry<String, Object> entry : fieldMap.entrySet()) {

			inputValue = entry.getValue();
			if (inputValue instanceof String[]) {
				value = String.join(",", ((String[]) inputValue));
			} else {
				value = StringUtil.defaultString(inputValue, "");
			}
			newDataMap.put(entry.getKey(), value);
		}

		return newDataMap;
	}

	/**
	 * パラメータを元に入力データマップを生成する
	 *
	 * @param inputs
	 * @param defaults
	 * @return
	 */
	protected Map<String, Object> createInputMap(	Map<String, Object> inputs,
													Map<String, Object> defaults,
													boolean isUseInputsOnly) {

		Set<String> nameSet = new HashSet<>();
		nameSet.addAll(inputs.keySet());
		nameSet.addAll(defaults.keySet());

		Map<String, Object> newInputMap = new HashMap<>();
		for (String name : nameSet) {
			boolean useInputValue = false;
			if (isUseInputsOnly) {
				useInputValue = true;
			} else if (inputs.containsKey(name)) {
				if (inputs.get(name) instanceof String && !StringUtils.isEmpty(
						StringUtil.defaultString(inputs.get(name), ""))) {
					useInputValue = true;
				} else if (inputs.get(name) instanceof String[]
						&& ((String[]) inputs.get(name)).length > 0) {
					useInputValue = true;
				} else if (inputs.get(name) instanceof List
						&& ((List) inputs.get(name)).size() > 0) {
					useInputValue = true;
				}
			}

			if (useInputValue) {
				newInputMap.put(name, inputs.getOrDefault(name, ""));
			} else if (defaults.containsKey(name)) {
				newInputMap.put(name, defaults.get(name));
			} else if (inputs.containsKey(name)) {
				newInputMap.put(name, inputs.get(name));
			}
		}

		return newInputMap;
	}

	public MapParam getInputs() {
		return inputs;
	}

	public void setInputs(MapParam inputs) {
		this.inputs = inputs;
	}

	public MapParam getDefaults() {
		if (defaults == null) {
			defaults = new MapParam();
			defaults.setParamMap(new HashMap<String, Object>());
		}
		return defaults;
	}

	public void setDefaults(MapParam defaults) {
		this.defaults = defaults;
	}

	public MapListParam getColumnSettings() {
		return columnSettings;
	}

	public void setColumnSettings(MapListParam columnSettings) {
		this.columnSettings = columnSettings;
	}

	public boolean isUseInputsOnly() {
		return getUseInputsOnly().equals("1");
	}

	public String getUseInputsOnly() {
		if (useInputsOnly == null || useInputsOnly == "") {
			useInputsOnly = "0";
		}
		return useInputsOnly;
	}

	public void setUseInputsOnly(String useInputsOnly) {
		this.useInputsOnly = useInputsOnly;
	}

}
