package jp.ill.photon.module.file;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.HashMap;
import java.util.Map;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.ParamUtil;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;

/**
 * 複数ファイルをTmp領域（System.getProperty("java.io.tmpdir")）にコピーするモジュール
 *
 * @author m_fukukawa
 *
 */
public class SaveFilesToTmpAreaModule implements PhotonModule {

	/* モジュールパラメータ */
	@ModuleParam(required = false)
	private Map<String, Object> fileInfoMap;

	@ModuleParam(required = false)
	private String processName;

	@ModuleParam(required = false)
	private String operator;

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(SaveFilesToTmpAreaModule.class);

	@SuppressWarnings("unchecked")
	@Override
	public ModuleResult execute(ModuleContext context) throws PhotonModuleException {

		ModuleResult result = new ModuleResult();
		ActionDto dto = context.getDto();
		Map<String, Object> ret = new HashMap<String, Object>();

		// Tmp領域のディレクトリ取得
		String tmpDir = getTmpDir();

		// fileInfoMapで指定されているファイルパラメータ先に存在するファイルを
		// Tmp領域にセットする

		/*
		 * fileInfoMapの中身想定
		 *
		 * "key1": {
		 *   "full_path": {
		 *     "type": "param_file",
		 *     "val": "key1"
		 *   },
		 *   "file_name": {
		 *     "type": "param_file",
		 *     "val": "key1.file.filename"
		 *   }
		 * },
		 * "key2": {
		 *   "full_path": {
		 *     "type": "param_file",
		 *     "val": "key2"
		 *   },
		 *   "file_name": {
		 *     "type": "param_file",
		 *     "val": "key2.file.filename"
		 *   }
		 * }
		 *
		 *
		 */

		if (!MapUtils.isEmpty(fileInfoMap)) {

			for (Map.Entry<String, Object> entry : fileInfoMap.entrySet()) {

				String key = entry.getKey();
				Map<String, Object> valMap = (Map<String, Object>) entry.getValue();

				String filePath = ParamUtil.getParamStrValueByType(valMap.get("full_path"), dto);
				String fileName = ParamUtil.getParamStrValueByType(valMap.get("file_name"), dto);

				// ファイル名を作成
				String destFileName = getFileName(key);

				// filePathからTmp領域 + destFileNameへコピー
				Path destFilePath = Paths.get(tmpDir, destFileName);
				if (!StringUtils.isEmpty(filePath) && destFilePath != null) {

					try {
						Files.copy(Paths.get(filePath), destFilePath, StandardCopyOption.REPLACE_EXISTING);
					} catch (IOException e) {
						logger.error("ファイルコピーに失敗しました："
								+ destFilePath.getFileName().toString(), e);
					}

					logger.info("ファイルをコピーしました："
							+ filePath + " >>> " + destFilePath.getFileName().toString());

				}

				if (!StringUtils.isEmpty(fileName)) {
					ret.put(key, new HashMap<String, Object>(){{
						put("file_name", fileName);
						put("full_path", (destFilePath == null)? null : destFilePath.toString());
					}});
				}

			}

		}

		// result.getReturnData().put("set", destFileNameList);
		result.getReturnData().put("file_map", ret);
		result.setResultCode("save");

		return result;

	}

	/**
	 *
	 * Tmp領域のパスの取得
	 *
	 * @return
	 *
	 */
	private String getTmpDir() {

		String ret = System.getProperty("java.io.tmpdir");
		if (ret.endsWith(File.separator)) {
			ret = ret.substring(0, ret.length() - File.separator.length());
		}

		return ret;

	}

	/**
	 *
	 * 保存先ファイル名の取得
	 *
	 * @return
	 *
	 */
	private String getFileName(String name) {

		// [処理名] + "__" + [操作者が特定できるID] + "__" + [対象のオブジェクト名]
		// 例）利用者側のカートから注文作成時、担当者IDが"101"の担当者が注文明細4の2行目のファイル項目をセットしたときの名称
		// 　　"cart_to_order_order__101__order_detail_exfield4[1]"

		return String.format("%s__%s__%s.tmp",
				StringUtils.defaultString(processName, ""),
				StringUtils.defaultString(operator, ""),
				name);

	}

	/**
	 * fileInfoMapを取得します。
	 * @return fileInfoMap
	 */
	public Map<String, Object> getFileInfoMap() {
		return fileInfoMap;
	}

	/**
	 * fileInfoMapを設定します。
	 * @param fileInfoMap
	 */
	public void setFileInfoMap(Map<String, Object> fileInfoMap) {
		this.fileInfoMap = fileInfoMap;
	}

	/**
	 * processNameを取得します。
	 * @return processName
	 */
	public String getProcessName() {
		return processName;
	}

	/**
	 * processNameを設定します。
	 * @param processName
	 */
	public void setProcessName(String processName) {
		this.processName = processName;
	}

	/**
	 * operatorを取得します。
	 * @return operator
	 */
	public String getOperator() {
		return operator;
	}

	/**
	 * operatorを設定します。
	 * @param operator
	 */
	public void setOperator(String operator) {
		this.operator = operator;
	}

}
