package jp.ill.photon.module.file;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.action.FileParam;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.MapListParam;
import jp.ill.photon.model.MapParam;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringUtil;
import jp.ill.photon.util.UtilTools;

/**
 * 複数ファイルを指定パスにコピーするモジュール
 *
 * @author h_tanaka
 *
 */
public class MultiFileUploadModule implements PhotonModule {

	/* モジュールパラメータ */
	@ModuleParam(required = true)
	private MapListParam fileInfoList;

	@ModuleParam(required = true)
	private String srcFilePathKey;

	@ModuleParam(required = true)
	private String destRootDir;

	@ModuleParam(required = true)
	private String destSubDir;

	@ModuleParam(required = true)
	private String fileNameTemplate;

	@ModuleParam(required = false)
	private MapParam fileNameValues;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		ModuleResult result = new ModuleResult();

		// ファイル名テンプレートを準備

		String template = getFileNameTemplate();

		Map<String, Object> fileNameValueMap = new HashMap<>();
		if (this.fileNameValues != null) {
			fileNameValueMap = getFileNameValues().getParamMap();
		}
		template = updateFileNameTemplate(template, fileNameValueMap,
				context.getDto());

		List<Map<String, Object>> fileInfoList = getFileInfoList()
				.getParamList();

		// 各ファイルをコピー処理
		Map<String, Object> savedFileMap = new HashMap<>();
		String srcFilePathString = null;
		String destFileName = null;
		Path srcFilePath = null;
		Path destFilePath = null;
		FileParam[] fileParams = null;
		List<String> destFileNameList = new ArrayList<>();
		boolean isIOError = false;
		for (Map<String, Object> fileInfo : fileInfoList) {
			srcFilePathString = StringUtil.defaultString(
					fileInfo.getOrDefault(getSrcFilePathKey(), ""), "");
			if (srcFilePathString == "") {
				continue;
			}
			srcFilePath = Paths.get(srcFilePathString);

			destFileName = new String(template);
			destFileName = updateFileNameTemplateWithFileInfo(destFileName,
					fileInfo);
			destFileName = destFileName.replace("{{_file_name_}}",
					srcFilePath.getFileName().toString());
			destFileName = destFileName.replace("{{_ext_}}", srcFilePath
					.getFileName().toString().replaceAll("^[^\\.]*\\.", ""));

			if (destFileName.equals("")) {
				continue;
			}
			destFileNameList.add(destFileName);

			// 保存先ディレクトリ
			String destDirectory = getDestRootDir();
			if(!UtilTools.isEmpty(getDestSubDir())){
				destDirectory += File.separator + getDestSubDir();
			}
			// 保存先フルパス
			destFilePath = Paths.get(destDirectory, destFileName);

			try {
				Files.createDirectories(destFilePath);
				Files.copy(srcFilePath, destFilePath,
						StandardCopyOption.REPLACE_EXISTING);
			} catch (IOException e) {
				logger.error("ファイルコピーに失敗しました："
						+ destFilePath.getFileName().toString(), e);
				isIOError = true;
			}
		}

		// カラムコード毎のファイル名マップを結果オブジェクトにセット
		result.getReturnData().put("saved_file_names", destFileNameList);

		return result;

	}

	/**
	 * ファイル名テンプレートをパラメータマップで更新
	 *
	 * @param template
	 * @param fileNameValueMap
	 * @return
	 */
	protected String updateFileNameTemplate(String template,
											Map<String, Object> fileNameValueMap,
											ActionDto dto) {
		if (fileNameValueMap == null || fileNameValueMap.size() == 0) {
			return template;
		}

		String updatedTemplate = template;
		Map<String, Object> param = null;
		String paramValue = null;
		for (Map.Entry<String, Object> entry : fileNameValueMap.entrySet()) {
			param = (Map<String, Object>) entry.getValue();
			paramValue = String.valueOf(ParamUtil.getParamValueByType(
					String.valueOf(param.get("type")), param.get("val"), dto));
			updatedTemplate = updatedTemplate
					.replace("{{" + entry.getKey() + "}}", paramValue);
		}

		return updatedTemplate;
	}

	protected String updateFileNameTemplateWithFileInfo(String template,
														Map<String, Object> fileInfo) {
		if (fileInfo == null || fileInfo.size() == 0) {
			return template;
		}

		String updatedTemplate = new String(template);
		String value = null;
		for (Map.Entry<String, Object> entry : fileInfo.entrySet()) {
			value = StringUtil.defaultString(entry.getValue(), "");
			if (value != "") {
				updatedTemplate = updatedTemplate
						.replace("{{info." + entry.getKey() + "}}", value);
			}
		}

		return updatedTemplate;
	}

	public MapListParam getFileInfoList() {
		return fileInfoList;
	}

	public void setFileInfoList(MapListParam fileInfoList) {
		this.fileInfoList = fileInfoList;
	}

	public String getFileNameTemplate() {
		return fileNameTemplate;
	}

	public void setFileNameTemplate(String fileNameTemplate) {
		this.fileNameTemplate = fileNameTemplate;
	}

	public MapParam getFileNameValues() {
		return fileNameValues;
	}

	public void setFileNameValues(MapParam fileNameValues) {
		this.fileNameValues = fileNameValues;
	}

	public String getSrcFilePathKey() {
		return srcFilePathKey;
	}

	public void setSrcFilePathKey(String srcFilePathKey) {
		this.srcFilePathKey = srcFilePathKey;
	}

	public String getDestRootDir() {
		return destRootDir;
	}

	public void setDestRootDir(String destRootDir) {
		this.destRootDir = destRootDir;
	}

	public String getDestSubDir() {
		return destSubDir;
	}

	public void setDestSubDir(String destSubDir) {
		this.destSubDir = destSubDir;
	}

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(MultiFileUploadModule.class);
}
