package jp.ill.photon.module.file;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.action.ActionDispatcher;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.MetaObjectDao;
import jp.ill.photon.dao.MetaObjectDaoImpl;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.ErrorLog;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.ZipUtil;

/**
 * ZIPファイルを指定フォルダに解凍するモジュール
 *
 * @author h_tanaka
 *
 */
public class FileUnzipModule implements PhotonModule {

	@ModuleParam(required = false)
	private String srcDirPath;

	@ModuleParam(required = true)
	private String srcFilePath;

	@ModuleParam(required = true)
	private String destDirPath;

	@ModuleParam(required = false)
	private String destSubDirName;

	@ModuleParam(required = false)
	private String charset;

	@ModuleParam(required = false)
	private String jobId;

	@ModuleParam(required = false)
	private String scheduleId;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		ModuleResult result = new ModuleResult();

		ActionDto dto = context.getDto();
		String keyName = "error_field";
		String tenantId = dto.getTenantId();

		// 元ファイル存在チェック
		String srcDirPath = getSrcDirPath();
		Path srcPath = null;
		if (srcDirPath == null || srcDirPath.isEmpty()) {
			srcPath = Paths.get(getSrcFilePath());
		} else {
			srcPath = Paths.get(getSrcDirPath(), getSrcFilePath());
		}
		if (!Files.exists(srcPath)) {
			String errMes = "コピー元ファイルが存在しません：" + srcPath.toString();
			logger.info(errMes);
			writeErrorLog(keyName, errMes, dto);
			setErrorStatus(tenantId, jobId, scheduleId);
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.setResultCode("error_nosrc");
			return result;
		}

		// コピー先フォルダ存在チェック
		Path destPath = Paths.get(getDestDirPath());

		if (!Files.exists(destPath)) {
			String errMes = "コピー先フォルダが存在しません：" + destPath.toString();
			logger.info(errMes);
			writeErrorLog(keyName, errMes, dto);
			setErrorStatus(tenantId, jobId, scheduleId);
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.setResultCode("error_nodest");
			return result;
		}

		// サブディレクトリが存在する場合、ここで結合する
		String subDirName = getDestSubDirName();
		if (!StringUtils.isEmpty(subDirName)) {
			destPath = Paths.get(destPath.toString(), subDirName);
		}

		List<File> fileList = new ArrayList<>();
		try {
			fileList = ZipUtil.unZip(srcPath.toFile(), destPath.toFile(),
					getCharset());
		} catch (Exception e) {
			String errMes = "ファイルの解凍に失敗しました：" + srcPath.toString();
			logger.error(errMes, e);
			writeErrorLog(keyName, errMes, dto);
			setErrorStatus(tenantId, jobId, scheduleId);
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.setResultCode("error_io");
		}

		result.getReturnData().put("file_num", fileList.size());
		result.getReturnData().put("dest_dir", destPath.toString());

		return result;
	}

	/***
	 *
	 * エラーログを保存
	 *
	 * @param keyName
	 * @param errMes
	 * @param dto
	 */
	protected void writeErrorLog(String keyName, String errMes, ActionDto dto) {

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		List<ErrorLog> errorLogs = new ArrayList<>();

		ErrorLog log = new ErrorLog();
		log.setErrorTypeCd("0");
		log.setTenantId(dto.getTenantId());
		log.setAppId(dto.getAppId());
		log.setActionId(dto.getActionId());
		log.setScheduleId(getScheduleId());
		log.setRowNo(1);
		log.setKeyName(keyName);
		log.setMesg(errMes);
		log.setStrErrorInfo("{}");
		errorLogs.add(log);

		MetaObjectDao dao = new MetaObjectDaoImpl();
		tm.required(() -> {
			return dao.insertErrorLogs(errorLogs);
		});

	}

	/***
	 *
	 * ステータスを更新
	 *
	 * @param tenantId
	 * @param jobId
	 * @param scheduleId
	 */
	protected void setErrorStatus(	String tenantId,
									String jobId,
									String scheduleId) {

		Map<String, Object> initParameters = new HashMap<>();

		Map<String, Object> parameters = new HashMap<>();
		parameters.put("job_id", new String[] { jobId });
		parameters.put("schedule_id", new String[] { scheduleId });
		parameters.put("job_stat", new String[] { "9" });

		ActionDispatcher dispatcher = ActionDispatcher.getInstance();
		try {
			dispatcher.dispatch(tenantId, "batch", "/update_job_status",
					initParameters, parameters, null, null);
		} catch (Exception e) {
			logger.error("エラー終了ステータス更新エラー", e);
		}

	}

	public String getSrcDirPath() {
		return srcDirPath;
	}

	public void setSrcDirPath(String srcDirPath) {
		this.srcDirPath = srcDirPath;
	}

	public String getSrcFilePath() {
		return srcFilePath;
	}

	public void setSrcFilePath(String srcFilePath) {
		this.srcFilePath = srcFilePath;
	}

	public String getDestDirPath() {
		return destDirPath;
	}

	public void setDestDirPath(String destDirPath) {
		this.destDirPath = destDirPath;
	}

	public String getDestSubDirName() {
		return destSubDirName;
	}

	public void setDestSubDirName(String destSubDirName) {
		this.destSubDirName = destSubDirName;
	}

	public String getCharset() {
		if (charset == null || charset.length() == 0) {
			charset = "Windows-31J";
		}
		return charset;
	}

	public void setCharset(String charset) {
		this.charset = charset;
	}

	public String getScheduleId() {
		return scheduleId;
	}

	public void setScheduleId(String scheduleId) {
		this.scheduleId = scheduleId;
	}

	public String getJobId() {
		return jobId;
	}

	public void setJobId(String jobId) {
		this.jobId = jobId;
	}

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(FileUnzipModule.class);

}
