package jp.ill.photon.module.file;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.MapParam;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;

/**
 * JSONファイルを指定パスに出力するモジュール
 * 
 * @author h_tanaka
 *
 */
public class FileSpitJsonMapModule implements PhotonModule {

	@ModuleParam(required = true)
	private MapParam content;

	@ModuleParam(required = true)
	private String fileDirPath;

	@ModuleParam(required = true)
	private String fileName;

	@ModuleParam(required = false)
	private String encoding;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		// フォルダ存在チェック
		Path path = Paths.get(getFileDirPath());
		if (!Files.exists(path)) {
			throw new PhotonModuleException("出力先が存在しません:" + path.toString(),
					null);
		}

		String encoding = getEncoding();
		if (encoding == null || encoding.isEmpty()) {
			encoding = "UTF-8";
		}

		ObjectMapper mapper = new ObjectMapper();
		String json = "";
		try {
			json = mapper.writerWithDefaultPrettyPrinter()
					.writeValueAsString(getContent().getParamMap());
		} catch (JsonProcessingException e1) {
			throw new PhotonModuleException("JSON変換に失敗しました:" + path.toString(),
					e1);
		}

		String fileName = getFileName() + ".json";
		Path filePath = Paths.get(getFileDirPath(), fileName);

		// ファイルコンテンツを一括読み込み
		try {
			Files.write(filePath, json.getBytes(Charset.forName(encoding)));
		} catch (IOException e) {
			throw new PhotonModuleException("ファイルの出力に失敗しました:" + path.toString(),
					e);
		}

		// ファイルコンテンツを返す
		ModuleResult result = new ModuleResult();
		result.getReturnData().put("file_name", fileName);
		result.getReturnData().put("file_dir_path", getFileDirPath());
		result.getReturnData().put("file_path", filePath.toString());
		result.getReturnData().put("encoding", getEncoding());

		return result;
	}

	public MapParam getContent() {
		return content;
	}

	public void setContent(MapParam content) {
		this.content = content;
	}

	public String getFileDirPath() {
		return fileDirPath;
	}

	public void setFileDirPath(String fileDirPath) {
		this.fileDirPath = fileDirPath;
	}

	public String getEncoding() {
		return encoding;
	}

	public String getFileName() {
		return fileName;
	}

	public void setFileName(String fileName) {
		this.fileName = fileName;
	}

	public void setEncoding(String encoding) {
		this.encoding = encoding;
	}

}
