package jp.ill.photon.module.file;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.stream.Collectors;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;

/**
 * [file_slurp] ファイル一括読込モジュール.
 * 
 * <p>
 * 指定されたパスのファイルを読込み、テキストデータとしてDTOに追加する。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 * <dl>
 * <dt>contents</dt>
 * <dd>ファイルから読み込んだテキストデータ</dd>
 * </dl>
 * </p>
 * 
 * @author h_tanaka
 *
 */
public class FileSlurpModule implements PhotonModule {

	/**
	 * 対象ファイルのフォルダパス指定用モジュールパラメータ.
	 * 
	 */
	@ModuleParam(required = true)
	private String fileDirPath;

	/**
	 * 対象ファイルのパス指定用モジュールパラメータ.
	 * 
	 * <p>
	 * fileDirPathからの相対パスを指定する。
	 * </p>
	 */
	@ModuleParam(required = true)
	private String filePath;

	/**
	 * 対象ファイルの文字コード指定用モジュールパラメータ.
	 * 
	 * <p>
	 * 指定しなかった場合は「UTF-8」で読み込む。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String encoding;

	/**
	 * 対象ファイルの改行コード指定用モジュールパラメータ.
	 * 
	 * <p>
	 * 指定しなかった場合は「\n」で読み込む。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String lineBreak;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		// ファイル存在チェック
		Path path = Paths.get(getFileDirPath(), getFilePath());
		if (!Files.exists(path)) {
			throw new PhotonModuleException("ファイルが存在しません:" + path.toString(),
					null);
		}

		String encoding = getEncoding();
		if (encoding == null || encoding.isEmpty()) {
			encoding = "UTF-8";
		}

		String lineBreak = getLineBreak();
		if (lineBreak == null || lineBreak.isEmpty()) {
			lineBreak = "\n";
		}

		// ファイルコンテンツを一括読み込み
		String contents;
		try {
			contents = Files.readAllLines(path, Charset.forName(encoding))
					.stream().collect(Collectors.joining(lineBreak));
		} catch (IOException e) {
			throw new PhotonModuleException("ファイルの読込に失敗しました:" + path.toString(),
					e);
		}

		// ファイルコンテンツを返す
		ModuleResult result = new ModuleResult();
		result.getReturnData().put("contents", contents);

		return result;
	}

	public String getFileDirPath() {
		return fileDirPath;
	}

	public void setFileDirPath(String fileDirPath) {
		this.fileDirPath = fileDirPath;
	}

	public String getFilePath() {
		return filePath;
	}

	public void setFilePath(String filePath) {
		this.filePath = filePath;
	}

	public String getEncoding() {
		return encoding;
	}

	public void setEncoding(String encoding) {
		this.encoding = encoding;
	}

	public String getLineBreak() {
		return lineBreak;
	}

	public void setLineBreak(String lineBreak) {
		this.lineBreak = lineBreak;
	}

}
