package jp.ill.photon.module.file;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;

/**
 * 柔軟なファイルコピーモジュール。コピー元ファイルをパターン指定して複数コピー可能。
 * 
 * @author h_tanaka
 *
 */
public class FilePatternCopyModule implements PhotonModule {

	@ModuleParam(required = true)
	private String srcDirPath;

	@ModuleParam(required = true)
	private String srcFilePattern;

	@ModuleParam(required = true)
	private String destDirPath;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		ModuleResult result = new ModuleResult();
		result.getReturnData().put("file_path", "");

		// コピー元フォルダ取得
		Path srcPath = Paths.get(getSrcDirPath());
		if (!Files.exists(srcPath)) {
			logger.info("コピー元フォルダが存在しません：" + srcPath.toString());
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.setResultCode("error_nosrc");
			return result;
		}

		// コピー先フォルダ取得
		Path destPath = Paths.get(getDestDirPath());
		if (!Files.exists(destPath)) {
			logger.info("コピー先フォルダが存在しません：" + destPath.toString());
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.setResultCode("error_nodest");
			return result;
		}

		// コピー処理
		try {
			List<Path> fileList = null;
			try (Stream<Path> stream = Files.list(srcPath)) {
				fileList = stream.filter(
						path -> path.toString().matches(getSrcFilePattern()))
						.collect(Collectors.toList());
			}

			if (fileList != null && fileList.size() > 0) {
				for (Path filePath : fileList) {
					Files.copy(filePath,
							destPath.resolve(filePath.getFileName()),
							StandardCopyOption.REPLACE_EXISTING);
				}
			}
		} catch (IOException e) {
			logger.info("ファイルコピー失敗しました：" + destPath.toString());
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.setResultCode("error_io");
		}

		// コピー結果を返す
		result.getReturnData().put("file_path", destPath.toString());

		return result;
	}

	public String getSrcDirPath() {
		return srcDirPath;
	}

	public void setSrcDirPath(String srcDirPath) {
		this.srcDirPath = srcDirPath;
	}

	public String getSrcFilePattern() {
		return srcFilePattern;
	}

	public void setSrcFilePattern(String srcFilePattern) {
		this.srcFilePattern = srcFilePattern;
	}

	public String getDestDirPath() {
		return destDirPath;
	}

	public void setDestDirPath(String destDirPath) {
		this.destDirPath = destDirPath;
	}

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(FilePatternCopyModule.class);
}
