package jp.ill.photon.module.file;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.UtilTools;

public class FilePathModule implements PhotonModule {

	@ModuleParam(required = true)
	private String rootDirPath;

	@ModuleParam(required = false)
	private Map<String, Object> subDirPaths;

	@ModuleParam(required = false)
	private String fileName;
	@ModuleParam(required=false)
	private String encodedFileName;

	/** メッセージID（0件エラー） */
    public static final String FILE_NOT_FOUND = "FileNotFound";

	/** ログ用変数 */
    protected final LogUtil logger = new LogUtil(FileCopyModule.class);

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		Map<String, Object> subDirList = ParamUtil.getParamValueByType(subDirPaths, context);
		ModuleResult result = new ModuleResult();

		int i = 0;
		String[] subdir = new String[subDirList.size()];
		for (Entry<String, Object> e : subDirList.entrySet()) {
			subdir[i] = (String) e.getValue();
			i++;
		}
		Path srcPath = Paths.get(rootDirPath, subdir);

		if (!Files.exists(srcPath)) {
			logger.info("指定されたディレクトリが存在しません：" + srcPath.toString());
            result.setResultType(ModuleResult.ResultTypes.ERROR);
            result.setResultCode("error_nosrc");

            List<ActionDtoMessage> errorMessages = new ArrayList<ActionDtoMessage>();
            ActionDtoMessage msg = new ActionDtoMessage();
            msg.setMessageId(FILE_NOT_FOUND);
            errorMessages.add(msg);
            result.getMessages().put("error_page", errorMessages);

            return result;
		}

		String fname = "";
		if (!UtilTools.isEmpty(fileName)) {
			fname = fileName;
		} else if (!UtilTools.isEmpty(encodedFileName)) {
			try {
				fname = URLDecoder.decode(encodedFileName, "UTF-8");
			} catch (UnsupportedEncodingException ex) {
				throw new PhotonModuleException("Unexpected error.", ex);
			}
		}

		// ファイル存在チェック
		Path filePath = srcPath.resolve(fname);
		if (!Files.exists(filePath)) {
		    logger.info("指定されたファイルが存在しません：" + filePath.toString());
		    result.setResultType(ModuleResult.ResultTypes.ERROR);
            result.setResultCode("error_nosrc");

            List<ActionDtoMessage> errorMessages = new ArrayList<ActionDtoMessage>();
            ActionDtoMessage msg = new ActionDtoMessage();
            msg.setMessageId(FILE_NOT_FOUND);
            errorMessages.add(msg);
            result.getMessages().put("error_page", errorMessages);

            return result;
		}

		// 結果を返す

		result.getReturnData().put("dirname", srcPath.toString());
		result.getReturnData().put("filename", fname);
		result.getReturnData().put("filepath", filePath.toString());
		return result;
	}

	public String getRootDirPath() {
		return rootDirPath;
	}

	public void setRootDirPath(String rootDirPath) {
		this.rootDirPath = rootDirPath;
	}

	public Map<String, Object> getSubDirPaths() {
		return subDirPaths;
	}

	public void setSubDirPaths(Map<String, Object> subDirPaths) {
		this.subDirPaths = subDirPaths;
	}

	public String getFileName() {
		return fileName;
	}

	public void setFileName(String fileName) {
		this.fileName = fileName;
	}

	public String getEncodedFileName() {
		return encodedFileName;
	}

	public void setEncodedFileName(String encodedFileName) {
		this.encodedFileName = encodedFileName;
	}
}
