package jp.ill.photon.module.file;

import java.io.File;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.CheckUtil;
import jp.ill.photon.util.LogUtil;

/**
 *
 * ファイル、ディレクトリを削除します。<br />
 * 削除対象ファイル配置ディレクトリの指定がない場合は、絶対パスが入力されてるとして処理します。<br />
 * <br />
 * 入力: <br/>
 * dir_path: 削除対象基底ディレクトリ<br/>
 * file_name: 削除対象ファイル名<br/>
 * file_name_list: 削除対象ファイル名(複数 file_nameと同時に指定した場合はマージする)<br/>
 * dir_delete_flg: 指定対象がディレクトリの場合、削除するかしないか(true: 削除する false: 削除しない(default))<br/>
 * <br/>
 * 戻り値: <br />
 * result: 結果(String)<br/>
 * success_log_mess: 成功メッセージ(List{@literal<String>})<br/>
 * faild_log_mess: 失敗メッセージ(List{@literal<String>})<br/>
 * notice_log_mess: 注意メッセージ(List{@literal<String>})<br/>
 *
 */
public class FileDeleteModule implements PhotonModule {

	/** 削除対象基底ディレクトリ */
	@ModuleParam(required = false)
	private String dirPath;

	/** 削除対象ファイル名 */
	@ModuleParam(required = false)
	private String fileName;

	/** 削除対象ファイル名(複数／リスト) */
	@ModuleParam(required = false)
	private List<String> fileNameList;

	/** 削除対象ファイル名(複数／マップ) */
	@ModuleParam(required = false)
	private Map<String, String> fileNameMap;

	/** 削除対象ファイル名(複数／配列) */
	@ModuleParam(required = false)
	private String[] fileNameArray;

	/** 削除対象 0:ファイル 1:ディレクトリ 左記以外:区別なし */
	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "static", transferVal = "")
	private String dirDeleteFlg;

	/* dirDeleteFLgの値 */
	private final String DIR_FILE_FLG_FILE = "0";
	private final String DIR_FILE_FLG_DIRECTORY = "1";

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(FileDeleteModule.class);

	@Override
	public ModuleResult execute(ModuleContext context) throws PhotonModuleException {

		// 対象ファイル名をマージ
		List<String> targetFiles = new ArrayList<String>();
		if (fileNameList != null) {
			for (String tmpFileName : fileNameList) {
				if (!CheckUtil.isEmpty(tmpFileName)) {
					targetFiles.add(tmpFileName);

				}
			}
		}
		if (fileNameMap != null) {
			for (String key : fileNameMap.keySet()) {
				if (!CheckUtil.isEmpty(fileNameMap.get(key))) {
					targetFiles.add((String)fileNameMap.get(key));

				}
			}
		}
		if (fileNameArray != null) {
			for (int i = 0; i < fileNameArray.length; i++) {
				if (!CheckUtil.isEmpty(fileNameArray[i])) {
					targetFiles.add(fileNameArray[i]);

				}
			}
		}

		if (!CheckUtil.isEmpty(fileName) && !targetFiles.contains(fileName)) {
			targetFiles.add(fileName);

		}

		/** 返却用メッセージ */
		List<String> successLogMess = new ArrayList<String>();
		List<String> faildLogMess = new ArrayList<String>();
		List<String> noticeLogMess = new ArrayList<String>();

		// 削除処理
		for (String deleteFileName : targetFiles) {
			String deleteAbsolutePath = "";
			if (CheckUtil.isEmpty(dirPath)) {
				deleteAbsolutePath = deleteFileName;

			} else {
				deleteAbsolutePath = dirPath + File.separator + deleteFileName;

			}
			try {
				File file = new File(deleteAbsolutePath);

				logger.info("削除対象" + ( (file.isDirectory())? "ディレクトリ" : "ファイル" ) + "パス：" + deleteAbsolutePath);

				if (!file.exists()) {
					noticeLogMess.add(deleteAbsolutePath + " は存在しません。");

				} else if (file.isDirectory() && !DIR_FILE_FLG_FILE.equals(dirDeleteFlg)) {
					// ディレクトリの中身を削除
					boolean ret = deleteDirectory(file.getAbsolutePath());
					if (ret) {
						// ディレクトリを削除
						ret = file.delete();
					}
					if (ret) {
						successLogMess.add(deleteAbsolutePath + " を削除しました。");
					} else {
						faildLogMess.add(deleteAbsolutePath + " の削除に失敗しました。");
					}

				} else if (file.isFile() && !DIR_FILE_FLG_DIRECTORY.equals(dirDeleteFlg)) {
					if (file.delete()) {
						successLogMess.add(deleteAbsolutePath + " を削除しました。");

					} else {
						faildLogMess.add(deleteAbsolutePath + " の削除に失敗しました。");

					}
				}
			} catch (Exception e) {
				faildLogMess.add("システムエラーが発生しました。処理を中断します。");
				faildLogMess.add(deleteAbsolutePath + " の削除に失敗しました。");
				break;
			}
		}

		// 結果を返す
		ModuleResult result = new ModuleResult();
		result.getReturnData().put("result", faildLogMess.size() == 0 ? "true" : "false");
		result.getReturnData().put("success_log_mess", successLogMess);
		result.getReturnData().put("faild_log_mess", faildLogMess);
		result.getReturnData().put("notice_log_mess", noticeLogMess);

		return result;
	}

	/**
	 * ディレクトリの中身を削除
	 */
	private Boolean deleteDirectory(String path) {
		Boolean rtnBool = true;
		// ディレクトリの中身を削除
		File[] dirContents = Paths.get(path).toFile().listFiles();
		for (File file : dirContents) {
			if (file.isDirectory()) {
				if (deleteDirectory(file.getAbsolutePath())) {
					// ディレクトリの子要素 削除OK
					if (!file.delete()) {
						// ディレクトリ 削除NG
						rtnBool = false;
					}

				} else {
					// ディレクトリの子要素 削除NG
					rtnBool = false;
				}

			} else if (file.isFile()) {
				if (!file.delete()) {
					// 削除NG
					rtnBool = false;
				}
			}
		}
		return rtnBool;
	}

	/**
	 * 削除対象基底ディレクトリを取得します。
	 * @return 削除対象基底ディレクトリ
	 */
	public String getDirPath() {
	    return dirPath;
	}

	/**
	 * 削除対象基底ディレクトリを設定します。
	 * @param dirPath 削除対象基底ディレクトリ
	 */
	public void setDirPath(String dirPath) {
	    this.dirPath = dirPath;
	}

	/**
	 * 削除対象ファイル名を取得します。
	 * @return 削除対象ファイル名
	 */
	public String getFileName() {
	    return fileName;
	}

	/**
	 * 削除対象ファイル名を設定します。
	 * @param fileName 削除対象ファイル名
	 */
	public void setFileName(String fileName) {
	    this.fileName = fileName;
	}

	/**
	 * 削除対象ファイル名(複数／リスト)を取得します。
	 * @return 削除対象ファイル名(複数／リスト)
	 */
	public List<String> getFileNameList() {
	    return fileNameList;
	}

	/**
	 * 削除対象ファイル名(複数／リスト)を設定します。
	 * @param fileNameList 削除対象ファイル名(複数／リスト)
	 */
	public void setFileNameList(List<String> fileNameList) {
	    this.fileNameList = fileNameList;
	}

	/**
	 * 削除対象ファイル名(複数／マップ)を取得します。
	 * @return 削除対象ファイル名(複数／マップ)
	 */
	public Map<String, String> getFileNameMap() {
	    return fileNameMap;
	}

	/**
	 * 削除対象ファイル名(複数／マップ)を設定します。
	 * @param fileNameMap 削除対象ファイル名(複数／マップ)
	 */
	public void setFileNameMap(Map<String, String> fileNameMap) {
	    this.fileNameMap = fileNameMap;
	}

	/**
	 * 削除対象ファイル名(複数／配列)を取得します。
	 * @return 削除対象ファイル名(複数／配列)
	 */
	public String[] getFileNameArray() {
	    return fileNameArray;
	}

	/**
	 * 削除対象ファイル名(複数／配列)を設定します。
	 * @param fileNameArray 削除対象ファイル名(複数／配列)
	 */
	public void setFileNameArray(String[] fileNameArray) {
	    this.fileNameArray = fileNameArray;
	}

	/**
	 * 削除対象 0:ファイル 1:ディレクトリ 左記以外:区別なしを設定します。
	 * @param dirDeleteFlg 削除対象 0:ファイル 1:ディレクトリ 左記以外:区別なし
	 */
	public void setDirDeleteFlg(String dirDeleteFlg) {
	    this.dirDeleteFlg = dirDeleteFlg;
	}
}
