package jp.ill.photon.module.file;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;

/**
 * 厳密なファイルコピーモジュール。コピー元ファイルが存在しない場合はコピーせずエラー結果を返す。
 * 
 * @author h_tanaka
 *
 */
public class FileCopyModule implements PhotonModule {

	@ModuleParam(required = false)
	private String srcDirPath;

	@ModuleParam(required = true)
	private String srcFilePath;

	@ModuleParam(required = false)
	private String destDirPath;

	@ModuleParam(required = true)
	private String destFilePath;

	@ModuleParam(required = false)
	private String overwriteMode;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		ModuleResult result = new ModuleResult();
		result.getReturnData().put("file_path", "");

		// 元ファイル存在チェック
		String srcDirPath = getSrcDirPath();
		Path srcPath = null;
		if (srcDirPath == null || srcDirPath.isEmpty()) {
			srcPath = Paths.get(getSrcFilePath());
		} else {
			srcPath = Paths.get(getSrcDirPath(), getSrcFilePath());
		}
		if (!Files.exists(srcPath)) {
			logger.info("コピー元ファイルが存在しません：" + srcPath.toString());
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.setResultCode("error_nosrc");
			return result;
		}

		// コピー先ファイル存在チェック
		String destDirPath = getDestDirPath();
		Path destPath = null;
		if (destDirPath == null || destDirPath.isEmpty()) {
			destPath = Paths.get(getDestFilePath());
		} else {
			destPath = Paths.get(getDestDirPath(), getDestFilePath());
		}

		if (!isOverwrite() && Files.exists(destPath)) {
			logger.info("上書き禁止モードでコピー先ファイルが存在したため、コピーを中止しました："
					+ destPath.toString());
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.setResultCode("error_overwrite");
		} else {
			try {
				Files.copy(srcPath, destPath,
						StandardCopyOption.REPLACE_EXISTING);
			} catch (IOException e) {
				logger.error("ファイルコピー失敗しました：" + destPath.toString(), e);
				result.setResultType(ModuleResult.ResultTypes.ERROR);
				result.setResultCode("error_io");
			}
		}

		// コピー結果を返す
		result.getReturnData().put("file_path", destPath.toString());

		return result;
	}

	public String getSrcDirPath() {
		return srcDirPath;
	}

	public void setSrcDirPath(String srcDirPath) {
		this.srcDirPath = srcDirPath;
	}

	public String getSrcFilePath() {
		return srcFilePath;
	}

	public void setSrcFilePath(String srcFilePath) {
		this.srcFilePath = srcFilePath;
	}

	public String getDestDirPath() {
		return destDirPath;
	}

	public void setDestDirPath(String destDirPath) {
		this.destDirPath = destDirPath;
	}

	public String getDestFilePath() {
		return destFilePath;
	}

	public void setDestFilePath(String destFilePath) {
		this.destFilePath = destFilePath;
	}

	public String getOverwriteMode() {
		return overwriteMode;
	}

	public void setOverwriteMode(String overwriteMode) {
		this.overwriteMode = overwriteMode;
	}

	protected boolean isOverwrite() {
		boolean isOverwrite = true;
		if (getOverwriteMode() != null && !getOverwriteMode().isEmpty()
				&& !getOverwriteMode().equals("overwrite")) {
			isOverwrite = false;
		}

		return isOverwrite;
	}

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(FileCopyModule.class);
}
