package jp.ill.photon.module.file;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.action.ActionParamMap;
import jp.ill.photon.action.FileParam;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.MapParam;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.ParamUtil;

/**
 * カラム定義を元にアップロードファイルを保存するモジュール
 *
 * @author h_tanaka
 *
 */
public class FieldFileUploadModule implements PhotonModule {

	/* モジュールパラメータ */
	@ModuleParam(required = true)
	private MapParam fieldSettings;

	@ModuleParam(required = false)
	private MapParam fieldParamMap;

	@ModuleParam(required = true)
	private String fileNameTemplate;

	@ModuleParam(required = false)
	private MapParam fileNameValues;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		ModuleResult result = new ModuleResult();

		// パラメータ名とカラムコードの関連マップ準備
		Map<String, Map<String, String>> fieldSettingMap = createFieldSettingMap(
				getFieldSettings(), getFieldParamMap(), context.getDto());

		// ファイル名テンプレートを準備

		String template = getFileNameTemplate();

		Map<String, Object> fileNameValueMap = new HashMap<>();
		if (this.fileNameValues != null) {
			fileNameValueMap = getFileNameValues().getParamMap();
		}

		template = updateFileNameTemplate(template, fileNameValueMap,
				context.getDto());

		// 各ファイルをコピー処理

		ActionParamMap paramMap = context.getDto().getParams();
		Map<String, Object> savedFileMap = new HashMap<>();
		Map<String, Object> orgFileMap = new HashMap<>();
		Map<String, String> setting = null;
		String destFileName = "noname.txt";
		Path srcFilePath = null;
		Path destFilePath = null;
		FileParam[] fileParams = null;
		List<String> destFileNameList = null;
		List<String> destOrgFileNameList = null;
		boolean isIOError = false;
		for (Map.Entry<String, Map<String, String>> entry : fieldSettingMap
				.entrySet()) {
			setting = (Map<String, String>) entry.getValue();

			// パラメータからアップロードファイル情報を取得
			fileParams = paramMap.getFileParams(setting.get("param_name"));
			if (fileParams.length == 0) {
				continue;
			}

			destFileNameList = new ArrayList<>();
			destOrgFileNameList = new ArrayList<>();
			for (FileParam f : fileParams) {
				if (f.getFileName().equals("")) {
					continue;
				}
				destFileName = template;
				destFileName = destFileName.replace("{{_col_cd_}}",
						setting.get("col_cd"));
				destFileName = destFileName.replace("{{_file_name_}}",
						f.getFileName());
				destFileName = destFileName.replace("{{_ext_}}",
						f.getFileName().replaceAll("^[^\\.]*\\.", ""));

				srcFilePath = Paths.get(f.getFile().getAbsolutePath());
				destFilePath = Paths.get(setting.get("store_path"),
						destFileName);
				destFileNameList.add(destFileName);
				destOrgFileNameList.add(f.getFileName());
				try {
					Files.copy(srcFilePath, destFilePath,
							StandardCopyOption.REPLACE_EXISTING);
				} catch (IOException e) {
					logger.error(
							"ファイルコピーに失敗しました：" + f.getFile().getAbsolutePath(),
							e);
					isIOError = true;
				}
			}

			savedFileMap.put(setting.get("col_cd"), destFileNameList);
			orgFileMap.put(setting.get("col_cd"), destOrgFileNameList);
		}

		// カラムコード毎のファイル名マップを結果オブジェクトにセット
		result.getReturnData().put("saved_file_map", savedFileMap);
		result.getReturnData().put("org_file_name_map", orgFileMap);

		return result;
	}

	/**
	 * パラメータ名とカラムコードの設定マップ作成
	 *
	 * @param fieldSettingsParam
	 * @param fieldParamMapParam
	 * @return
	 */
	protected Map<String, Map<String, String>> createFieldSettingMap(	MapParam fieldSettingsParam,
																		MapParam fieldParamMapParam,
																		ActionDto dto) {

		Map<String, Object> common = (Map<String, Object>) dto.getDataMap()
				.get("common");
		Map<String, Object> systemSettingMap = (Map<String, Object>) common
				.get("systemsetting");

		Map<String, Object> fieldParamMap = new HashMap<>();
		if (fieldParamMapParam != null) {
			Map<String, Object> param = null;
			String paramValue = null;
			for (Map.Entry<String, Object> entry : fieldParamMapParam
					.getParamMap().entrySet()) {
				param = (Map<String, Object>) entry.getValue();
				paramValue = String.valueOf(ParamUtil.getParamValueByType(
						String.valueOf(param.get("type")), param.get("val"),
						dto));
				fieldParamMap.put(entry.getKey(), paramValue);
			}
		}

		// カラム設定データ準備

		Map<String, Object> fieldSettings = (Map<String, Object>) fieldSettingsParam
				.getParamMap();
		Map<String, Map<String, String>> fieldSettingMap = new HashMap<>();
		Map<String, Object> fieldSetting = null;
		Map<String, Object> softConst = null;
		Map<String, String> setting = null;
		String storePathSystemSettingId = null;
		Map<String, Object> storePathSetting = null;
		String storePath = null;
		if (fieldSettings != null) {
			for (Map.Entry<String, Object> entry : fieldSettings.entrySet()) {
				fieldSetting = (Map<String, Object>) entry.getValue();
				setting = new HashMap<>();

				setting.put("col_cd", entry.getKey());

				// パラメータ名設定
				setting.put("param_name",
						String.valueOf(fieldParamMap.getOrDefault(
								entry.getKey(), fieldSetting.get("col_cd"))));

				// 保存先ディレクトリ設定
				storePath = "";
				softConst = (Map<String, Object>) fieldSetting
						.getOrDefault("soft_const", null);
				if (softConst != null) {
					storePathSystemSettingId = String
							.valueOf(softConst.getOrDefault(
									"file_store_path_system_setting_id", ""));
					if (storePathSystemSettingId.length() > 0) {
						storePathSetting = (Map<String, Object>) systemSettingMap
								.getOrDefault(storePathSystemSettingId, null);
						if (storePathSetting != null) {
							storePath = String.valueOf(
									storePathSetting.getOrDefault("note", ""));
						}
					}
				}

				if (storePath.equals("")) {
					logger.info("col_cd: " + entry.getKey()
							+ " file_store_path_system_setting_idから保存先パスを取得できませんでした："
							+ storePathSystemSettingId);
				}
				setting.put("store_path", storePath);

				fieldSettingMap.put(entry.getKey(), setting);
			}
		}

		return fieldSettingMap;
	}

	/**
	 * ファイル名テンプレートをパラメータマップで更新
	 *
	 * @param template
	 * @param fileNameValueMap
	 * @return
	 */
	protected String updateFileNameTemplate(String template,
											Map<String, Object> fileNameValueMap,
											ActionDto dto) {
		if (fileNameValueMap == null || fileNameValueMap.size() == 0) {
			return template;
		}

		String updatedTemplate = template;
		Map<String, Object> param = null;
		String paramValue = null;
		for (Map.Entry<String, Object> entry : fileNameValueMap.entrySet()) {
			param = (Map<String, Object>) entry.getValue();
			paramValue = String.valueOf(ParamUtil.getParamValueByType(
					String.valueOf(param.get("type")), param.get("val"), dto));
			updatedTemplate = updatedTemplate
					.replace("{{" + entry.getKey() + "}}", paramValue);
		}

		return updatedTemplate;
	}

	public MapParam getFieldSettings() {
		return fieldSettings;
	}

	public void setFieldSettings(MapParam fieldSettings) {
		this.fieldSettings = fieldSettings;
	}

	public MapParam getFieldParamMap() {
		return fieldParamMap;
	}

	public void setFieldParamMap(MapParam fieldParamMap) {
		this.fieldParamMap = fieldParamMap;
	}

	public String getFileNameTemplate() {
		return fileNameTemplate;
	}

	public void setFileNameTemplate(String fileNameTemplate) {
		this.fileNameTemplate = fileNameTemplate;
	}

	public MapParam getFileNameValues() {
		return fileNameValues;
	}

	public void setFileNameValues(MapParam fileNameValues) {
		this.fileNameValues = fileNameValues;
	}

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(FieldFileUploadModule.class);
}
