package jp.ill.photon.module.file;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.MetaObjectDao;
import jp.ill.photon.dao.MetaObjectDaoImpl;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.model.ErrorLog;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.module.validation.SingleValidator;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.MessageUtil;

import org.apache.commons.lang3.StringUtils;
import org.seasar.doma.jdbc.tx.TransactionManager;

/**
 * 対象フォルダ内のファイルをバリデートしてエラーログにエラー内容を出力するモジュール
 *
 * @author h_tanaka
 *
 */
public class BulkFileValidateModule implements PhotonModule {

	@ModuleParam(required = false)
	private String tenantId;

	@ModuleParam(required = false)
	private String scheduleId;

	@ModuleParam(required = true)
	private String validationSetId;

	@ModuleParam(required = true)
	private String srcDirPath;

	@ModuleParam(required = true)
	private String srcFilePattern;

	@ModuleParam(required = false)
	private String countInEachRow;

	private int errorCount = 0;
	private int lineIndex = 0;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		ModuleResult result = new ModuleResult();

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		SingleValidator validator = new SingleValidator(getTenantId(),
				getValidationSetId(), "bulk_file_validate");

		// コピー元フォルダ取得
		Path srcPath = Paths.get(getSrcDirPath());
		if (!Files.exists(srcPath)) {
			logger.info("コピー元フォルダが存在しません：" + srcPath.toString());
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.setResultCode("error_nosrc");
			return result;
		}

		long count = 0;
		try {
			count = Files.list(srcPath).count();
			Files.list(srcPath).forEach(new Consumer<Path>() {
				@Override
				public void accept(Path t) {
					// バリデーション処理
					Map<String, Object> errors = validator.validate("file_name",
							"ファイル名", t.getFileName().toString(),
							context.getDto());
					if (!errors.isEmpty()) {
						lineIndex++;
						// エラーログを追加
						writeErrorLogs(tm, t.getFileName().toString(),
								errors, context.getDto());
						return;
					}
				}
			});
		} catch (IOException e) {
			logger.info("ファイル取得失敗しました");
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.setResultCode("error_io");
		}

		// 結果オブジェクト生成
		result.getReturnData().put("file_count", count);
		result.getReturnData().put("error_count", errorCount);

		return result;
	}

	/**
	 * エラーログを保存
	 *
	 * @param errors
	 */
	protected void writeErrorLogs(	TransactionManager tm,
									String fileName,
									Map<String, Object> errors,
									ActionDto dto) {
		List<ErrorLog> errorLogs = new ArrayList<>();

		ErrorLog log = null;
		MessageUtil.buildMessage(errors, dto);
		List<ActionDtoMessage> messageList = (List<ActionDtoMessage>) errors
				.get("error_field");
		for (ActionDtoMessage message : messageList) {
			log = new ErrorLog();
			log.setErrorTypeCd("0");
			log.setTenantId(dto.getTenantId());
			log.setAppId(dto.getAppId());
			log.setActionId(dto.getActionId());
			log.setScheduleId(getScheduleId());
			// log.setRowNo(lineIndex + 1);
			log.setRowNo( (StringUtils.isEmpty(countInEachRow))? ++errorCount : lineIndex );
			// log.setKeyName(message.getFormFieldCd());
			log.setKeyName("error_field");
			log.setMesg(fileName + ": " + message.getMessage());
			log.setStrErrorInfo("{}");
			errorLogs.add(log);
		}

		MetaObjectDao dao = new MetaObjectDaoImpl();
		tm.required(() -> {
			return dao.insertErrorLogs(errorLogs);
		});
	}

	public String getScheduleId() {
		return scheduleId;
	}

	public void setScheduleId(String scheduleId) {
		this.scheduleId = scheduleId;
	}

	public String getValidationSetId() {
		return validationSetId;
	}

	public void setValidationSetId(String validationSetId) {
		this.validationSetId = validationSetId;
	}

	public String getSrcDirPath() {
		return srcDirPath;
	}

	public void setSrcDirPath(String srcDirPath) {
		this.srcDirPath = srcDirPath;
	}

	public String getSrcFilePattern() {
		return srcFilePattern;
	}

	public void setSrcFilePattern(String srcFilePattern) {
		this.srcFilePattern = srcFilePattern;
	}

	public String getTenantId() {
		return tenantId;
	}

	public void setTenantId(String tenantId) {
		this.tenantId = tenantId;
	}

	public String getCountInEachRow() {
		return countInEachRow;
	}

	public void setCountInEachRow(String countInEachRow) {
		this.countInEachRow = countInEachRow;
	}

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(BulkFileValidateModule.class);

}
