package jp.ill.photon.module.estimate;

import java.io.File;
import java.math.BigDecimal;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.seasar.doma.jdbc.tx.TransactionManager;
import org.springframework.util.CollectionUtils;

import jp.ill.photon.action.ActionDispatcher;
import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.AoSetting;
import jp.ill.photon.model.CartItem;
import jp.ill.photon.model.Delivery;
import jp.ill.photon.model.Estimate;
import jp.ill.photon.model.EstimateDetail;
import jp.ill.photon.model.Item;
import jp.ill.photon.model.OrderSummary;
import jp.ill.photon.model.SystemSetting;
import jp.ill.photon.model.SystemSetting.CaseNumDiv;
import jp.ill.photon.model.User;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.CheckUtil;
import jp.ill.photon.util.DateUtil;
import jp.ill.photon.util.EditTableUtil;
import jp.ill.photon.util.FileUtil;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringUtil;

public class EstimateCreateModule implements PhotonModule {

    @ModuleParam(required = true, paramGroup=true)
    private Estimate estimate = new Estimate();

    @ModuleParam(required = true, paramGroup=true)
    private Delivery delivery = new Delivery();

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "got_session.userForm.user_cd")
    private String userCd;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "got_session_charge.userChargeForm.user_charge_id")
    private int userChargeId;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.caseNumDiv.note")
    private String caseNumDiv;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType="dto", transferVal="common.aladdinsetting")
    private AoSetting aoSetting;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType="dto", transferVal="common.systemsetting")
    private SystemSetting systemSetting;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "cart.list")
    private List<Map<String, Object>> cartList;

    @ModuleParam(required = false)
    @DefaultParamSetting(transferType = "dto", transferVal = "delivery_fee.delivery_fee")
    private Map<String, Object> deliveryFee;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.estimateEndTimeDate.note")
    private String estimateEndTimeDate;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "user.first")
    private Map<String, Object> userForm;

    @ModuleParam(required = false)
    private List<Map<String, Object>> orderExtSettingMapList;

    @ModuleParam(required = false)
    private List<Map<String, Object>> orderDetailExtSettingMapList;

    @ModuleParam(required = false)
    private Map<String, Object> orderExtValueMap;

    @ModuleParam(required = false)
    private List<Map<String, Object>> orderDetailExtValueMapList;

    @ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.webOrderCampaignItemCd.note")
    private String webOrderCampaignItemCd;

    @ModuleParam(required = false)
    private Map<String, Object> webCampaignData;

    public Map<String, Object> getUserForm() {
        return userForm;
    }

    public void setUserForm(Map<String, Object> userForm) {
        this.userForm = userForm;
    }

    public Map<String, Object> getDeliveryFee() {
        return deliveryFee;
    }

    public void setDeliveryFee(Map<String, Object> deliveryFee) {
        this.deliveryFee = deliveryFee;
    }

    public String getEstimateEndTimeDate() {
        return estimateEndTimeDate;
    }

    public void setEstimateEndTimeDate(String estimateEndTimeDate) {
        this.estimateEndTimeDate = estimateEndTimeDate;
    }

    public Estimate getEstimate() {
        return estimate;
    }

    public void setEstimate(Estimate estimate) {
        this.estimate = estimate;
    }

    public Delivery getDelivery() {
        return delivery;
    }

    public void setDelivery(Delivery delivery) {
        this.delivery = delivery;
    }

    public String getUserCd() {
        return userCd;
    }

    public void setUserCd(String userCd) {
        this.userCd = userCd;
    }

    public int getUserChargeId() {
        return userChargeId;
    }

    public void setUserChargeId(int userChargeId) {
        this.userChargeId = userChargeId;
    }

    public String getCaseNumDiv() {
        return caseNumDiv;
    }

    public void setCaseNumDiv(String caseNumDiv) {
        this.caseNumDiv = caseNumDiv;
    }

    public AoSetting getAoSetting() {
        return aoSetting;
    }

    public void setAoSetting(AoSetting aoSetting) {
        this.aoSetting = aoSetting;
    }

    public SystemSetting getSystemSetting() {
        return systemSetting;
    }

    public void setSystemSetting(SystemSetting systemSetting) {
        this.systemSetting = systemSetting;
    }

    public List<Map<String, Object>> getCartList() {
        return cartList;
    }

    public void setCartList(List<Map<String, Object>> cartList) {
        this.cartList = cartList;
    }

    public List<Map<String, Object>> getOrderExtSettingMapList() {
        return orderExtSettingMapList;
    }

    public void setOrderExtSettingMapList(List<Map<String, Object>> orderExtSettingMapList) {
        this.orderExtSettingMapList = orderExtSettingMapList;
    }

    public List<Map<String, Object>> getOrderDetailExtSettingMapList() {
        return orderDetailExtSettingMapList;
    }

    public void setOrderDetailExtSettingMapList(List<Map<String, Object>> orderDetailExtSettingMapList) {
        this.orderDetailExtSettingMapList = orderDetailExtSettingMapList;
    }

    public Map<String, Object> getOrderExtValueMap() {
        return orderExtValueMap;
    }

    public void setOrderExtValueMap(Map<String, Object> orderExtValueMap) {
        this.orderExtValueMap = orderExtValueMap;
    }

    public List<Map<String, Object>> getOrderDetailExtValueMapList() {
        return orderDetailExtValueMapList;
    }

    public void setOrderDetailExtValueMapList(List<Map<String, Object>> orderDetailExtValueMapList) {
        this.orderDetailExtValueMapList = orderDetailExtValueMapList;
    }

    public String getWebOrderCampaignItemCd() {
        return webOrderCampaignItemCd;
    }

    public void setWebOrderCampaignItemCd(String webOrderCampaignItemCd) {
        this.webOrderCampaignItemCd = webOrderCampaignItemCd;
    }

    public Map<String, Object> getWebCampaignData() {
        return webCampaignData;
    }

    public void setWebCampaignData(Map<String, Object> webCampaignData) {
        this.webCampaignData = webCampaignData;
    }

    // Module内で使用する変数
    /** ログ用変数 */
    protected final LogUtil logger = new LogUtil(ActionDispatcher.class);

    /** DTO */
    ActionDto dto;

    /** テナントID */
    String tenantId;

    public ModuleResult execute(ModuleContext context)
            throws PhotonModuleException {
        ModuleResult moduleResult = new ModuleResult();

        dto = context.getDto();
        tenantId = dto.getTenantId();

        TransactionManager tm = DomaConfig.singleton().getTransactionManager();
        JsonDataDao dao = new JsonDataDaoImpl();

        User user = User.getUser(dto, userCd);

        // 登録日時
        Timestamp nowDate = new Timestamp(System.currentTimeMillis());

        int result = tm.required(() -> {

            // EstimateNoの採番と採番テーブルの更新
            List<Map<String, Object>> seqMap = dao.getSeqSfNextVal(tenantId, "estimate", "estimate_no");
            if (seqMap == null || seqMap.isEmpty()) {
                return -1;
            }
            Integer orderNoRow = Integer.valueOf(String.valueOf(seqMap.get(0).get("seq")));
            String estimateNo = String.format("%08d", orderNoRow);

            int recordNo = 0;

            OrderSummary summary = new OrderSummary();
            List<EstimateDetail> estimateDetails = new ArrayList<EstimateDetail>();

            // パラメータから来たカート情報をマップにセットしておく
            Map<String, CartItem> paramCartMap = new HashMap<String, CartItem>();
            for (Map<String, Object> paramCart : cartList) {
                CartItem item = CartItem.valueOf(paramCart);
                paramCartMap.put(item.getCartId().toString(), item);
            }

            // パラメータのカートIDごとに処理
            for (int i = 0; i < estimate.getCartId().length; i++) {

                String cartId = estimate.getCartId()[i];

                // レコードNoをカウントアップ
                recordNo++;

                // カート情報をDBから取得して、該当カートレコードをロック
                CartItem cart = null;
                List<Map<String, Object>> cartMap = dao.getCartForOrder(tenantId, cartId, userCd, String.valueOf(userChargeId));
                if (cartMap == null || cartMap.isEmpty()) {
                    if (paramCartMap.get(cartId) == null) {
                        return -1;
                    } else {
                        // キャンペーンの場合
                        cart = paramCartMap.get(cartId);
                    }
                } else {
                    cart = CartItem.valueOf(cartMap.get(0));
                }

                Item item = summary.sumOrderDetail(context, dao, cart, userCd, aoSetting, null, null, userForm, cart.getItemCd().equals(webOrderCampaignItemCd));

                // ケース数入力が管理画面などからOFFに変更されており、かつケース数1以上の注文が含まれていれば
                // その注文は削除してスキップ
                if (CaseNumDiv.OFF.equals(caseNumDiv)
                        && cart.getOrderCaseQuantity().compareTo(
                                BigDecimal.ZERO) != 0) {
                    dao.deleteCartItem(tenantId, cartId, userCd, String.valueOf(userChargeId));
                    continue;
                }

                EstimateDetail detail = new EstimateDetail(estimateNo, recordNo, cart, item, user, aoSetting, userForm);
                estimateDetails.add(detail);

                // カート削除
                dao.deleteCartItem(tenantId, cartId, userCd, String.valueOf(userChargeId));
            }

            // 配送料
            if (deliveryFee != null) {
                CartItem deliveryFeeItem = CartItem.valueOf(deliveryFee);
                Item item = summary.sumOrderDetail(context, dao, deliveryFeeItem, userCd, aoSetting, null, null, userForm, true);
                EstimateDetail detail = new EstimateDetail(estimateNo, ++recordNo, deliveryFeeItem, item, user, aoSetting, userForm);
                estimateDetails.add(detail);
            }

            // 見積明細登録
            for (EstimateDetail detail : estimateDetails) {
                String detailMapStr = JsonUtil.mapToJsonSnakeCaseKey(detail);
                Map<String, Object> detailMap = JsonUtil.jsonToMap(detailMapStr);
                try {
                    dao.insertEditTableData(
                            tenantId,
                            "estimatedetail",
                            user.getUserCd(),
                            detailMapStr);
                    logger.info(String.format("登録した見積明細に拡張項目の値をUPDATEする　見積No：%s、行No：%s", estimateNo, Integer.toString(detail.getRecordNo())));
                    Map<String, Object> afterData = dao.selectEditTableDataFromJson(tenantId, "estimatedetail", detailMapStr);
                    if (!MapUtils.isEmpty(afterData)) {
                        String guid = StringUtil.defaultString(afterData.get("guid"), "");
                        if (!StringUtils.isEmpty(guid)) {
                            detailMap.put("guid", guid);
                        }
                    }
                    // 追加した見積明細データに拡張項目をUPDATE
                    updateEstimateDetailExt(dao, estimateNo, detail.getRecordNo(), JsonUtil.mapToJson(detailMap));
                } catch (Exception ex) {
                    ex.printStackTrace();
                }
            }

            // 見積見出登録
            SimpleDateFormat df = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
            estimate.setEstimateNo(estimateNo);
            estimate.setUserCd(user.getUserCd());
            estimate.setEstimateMakeUserChargeId(String.valueOf(userChargeId));
            estimate.setDelivery(delivery);
            estimate.setWholesalePrice(new BigDecimal(summary.getSumWholesalePrice()));
            estimate.setTotalOrderQuantity(new BigDecimal(summary.getTotalQuantity()));
            estimate.setChumonFlg("0");
            estimate.setEstimateDate(df.format(nowDate));
            estimate.setEstimateEndTime(df.format(getEstimateEndTime(nowDate)));

            String estimateMapStr = JsonUtil.mapToJsonSnakeCaseKey(estimate);
            Map<String, Object> estimateMap = JsonUtil.jsonToMap(estimateMapStr);
            try {
                dao.insertEditTableData(
                        tenantId,
                        "estimate",
                        user.getUserCd(),
                        estimateMapStr);
                // 追加した見積見出データに拡張項目をUPDATE
                logger.info(String.format("登録した見積見出に拡張項目の値をUPDATEする　見積No：%s", estimateNo));
                Map<String, Object> afterData = dao.selectEditTableDataFromJson(tenantId, "estimate", estimateMapStr);
                if (!MapUtils.isEmpty(afterData)) {
                    String guid = StringUtil.defaultString(afterData.get("guid"), "");
                    if (!StringUtils.isEmpty(guid)) {
                        estimateMap.put("guid", guid);
                    }
                }
                updateEstimateExt(dao, estimateNo, JsonUtil.mapToJson(estimateMap));
                moduleResult.getReturnData().put("estimate", estimate);
                moduleResult.getReturnData().put("estimate_no", estimateNo);
            } catch (Exception ex) {
                ex.printStackTrace();
            }

            return 0;
        });

        return moduleResult;
    }

    /***
     *
     * 見積有効期限取得
     *
     * @param nowDate
     * @param campaignDetail
     * @return
     */
    private Timestamp getEstimateEndTime(Timestamp nowDate) {
        if (!CheckUtil.isEmpty(estimateEndTimeDate)) {
            int amount = Integer.parseInt(estimateEndTimeDate);
            Calendar cal = Calendar.getInstance();
            cal.setTimeInMillis(nowDate.getTime());
            cal.add(Calendar.DAY_OF_MONTH, amount);
            Timestamp userSiteDispEndTime = new Timestamp(
                    cal.getTimeInMillis());

            // userSiteDispEndTimeとwebCampaignData.end_dateを比較
            // 短い方に合わせる
            if (webCampaignData != null) {
                Timestamp endDate = DateUtil.toSqlTimestamp((String) webCampaignData.get("end_date"));
                if (userSiteDispEndTime.after(endDate)) {
                    userSiteDispEndTime = endDate;
                }
            }

            return userSiteDispEndTime;
        } else {
            // ここには来ないはず
            return nowDate;
        }
    }

    /***
    *
    * 見積見出の拡張項目で入力された値を更新
    *
    * @param dao
    * @param estimateNo
    * @param data
    */
   private void updateEstimateExt(JsonDataDao dao, String estimateNo, String data) {

       if (CollectionUtils.isEmpty(orderExtSettingMapList)) {
           logger.info("　見積見出拡張項目設定データ（order_ext_setting_map_list）にデータが入っていないため、見積見出拡張項目の更新は行わない");
           return;
       }

       logger.info("　見積見出拡張項目設定データ（order_ext_setting_map_list）にデータが入っている");

       Map<String, Object> updateMap = new HashMap<String, Object>();

       for (Map<String, Object> orderExtSetting : orderExtSettingMapList) {

           String colCd = (String) orderExtSetting.get("col_cd");
           String dataInputType = (String) orderExtSetting.get("data_input_type");
           String value = "";

           // 対応する値を取得
           Map<String, Object> orderExtValues = null;
           if (orderExtValueMap != null) {

               logger.info("　見積見出拡張項目値マップ（order_ext_value_map）にデータが入っている");

               orderExtValues = new HashMap<String, Object>();
               orderExtValues.put("value",
                       ParamUtil.getParamObjectValueByType(orderExtValueMap.get(colCd), dto));
               if (EditTableUtil.DataInputType.FILE.equals(dataInputType)) {
                   orderExtValues.put("saved_ext_file_name",
                           ParamUtil.getParamObjectValueByType(
                                   orderExtValueMap.get(colCd + ".saved_ext_file_name"), dto));
                   orderExtValues.put("saved_ext_full_path",
                           ParamUtil.getParamObjectValueByType(
                                   orderExtValueMap.get(colCd + ".saved_ext_full_path"), dto));
               } else if (EditTableUtil.DataInputType.CHECKBOX.equals(dataInputType) ||
                       EditTableUtil.DataInputType.LISTBOX.equals(dataInputType)) {
                   orderExtValues.put("separator",
                           ParamUtil.getParamObjectValueByType(
                                   orderExtValueMap.get(colCd + ".separator"), dto));
               }
               if (orderExtValues != null) {
                   value = getStrValue(orderExtValues);
               }
           }

           // --------------------------------
           // ファイルアップロード
           // --------------------------------
           if (EditTableUtil.DataInputType.FILE.equals(dataInputType)) {

               // ファイルを登録できたら、そのファイル名を値として保存すること
               logger.info(String.format("　見積見出拡張項目ファイルアップロード　カラムコード：%s", colCd));
               String fileName = uploadFile(dao, "estimate", data, orderExtSetting, orderExtValues);
               if (!StringUtils.isEmpty(fileName)) {
                   value = fileName;
               }

           }

           // updateのためのパラメータ
           updateMap.put(colCd, value);

       }

       logger.info(String.format("　更新対象の値セット：%s ", updateMap));
       if (!MapUtils.isEmpty(updateMap)) {
           // 更新
           updateMap.put("estimate_no", estimateNo);
           dao.updateEditTableData(tenantId, "estimate", userCd, JsonUtil.mapToJson(updateMap));
           ;
       }

   }

   /***
    *
    * 見積明細の拡張項目で入力された値を更新
    *
    * @param dao
    * @param estimateNo
    * @param recordNo
    * @param data
    */
   private void updateEstimateDetailExt(JsonDataDao dao, String estimateNo, int recordNo, String data) {

       Map<String, Object> updateMap = new HashMap<String, Object>();

       if (CollectionUtils.isEmpty(orderDetailExtSettingMapList)) {
           logger.info("　見積明細拡張項目設定データ（order_detail_ext_setting_map_list）がデータが入っていないため、見積明細拡張項目の更新は行わない");
           return;
       }

       logger.info("　見積明細拡張項目設定データ（order_detail_ext_setting_map_list）にデータが入っている");

       for (Map<String, Object> orderDetailExtSetting : orderDetailExtSettingMapList) {

           String colCd = (String) orderDetailExtSetting.get("col_cd");
           String dataInputType = (String) orderDetailExtSetting.get("data_input_type");
           String value = "";

           // 対応する値を取得
           Map<String, Object> orderDetailExtValues = null;
           if (orderDetailExtValueMapList != null) {

               logger.info("　見積明細拡張項目値マップリスト（order_detail_ext_value_map_list）にデータが入っている");

               for (Map<String, Object> item : orderDetailExtValueMapList) {
                   orderDetailExtValues = new HashMap<String, Object>();
                   String idx = ParamUtil.getParamStrValueByType(item.get("idx"), dto);
                   if (item.containsKey(colCd) && Integer.toString(recordNo - 1).equals(idx)) {
                       orderDetailExtValues.put("value",
                               ParamUtil.getParamObjectValueByType(item.get(colCd), dto));
                       if (EditTableUtil.DataInputType.FILE.equals(dataInputType)) {
                           orderDetailExtValues.put("saved_ext_file_name",
                                   ParamUtil.getParamObjectValueByType(
                                           item.get(colCd + ".saved_ext_file_name"), dto));
                           orderDetailExtValues.put("saved_ext_full_path",
                                   ParamUtil.getParamObjectValueByType(
                                           item.get(colCd + ".saved_ext_full_path"), dto));
                       } else if (EditTableUtil.DataInputType.CHECKBOX.equals(dataInputType) ||
                               EditTableUtil.DataInputType.LISTBOX.equals(dataInputType)) {
                           orderDetailExtValues.put("separator",
                                   ParamUtil.getParamObjectValueByType(
                                           item.get(colCd + ".separator"), dto));
                       }
                       break;
                   }
               }
               if (orderDetailExtValues != null) {
                   value = getStrValue(orderDetailExtValues);
               }

               // --------------------------------
               // ファイルアップロード
               // --------------------------------
               if (EditTableUtil.DataInputType.FILE.equals(dataInputType)) {

                   // ファイルを登録できたら、そのファイル名を値として保存すること
                   logger.info(String.format("　見積明細拡張項目ファイルアップロード　カラムコード：%s", colCd));
                   value = uploadFile(dao, "estimatedetail", data, orderDetailExtSetting, orderDetailExtValues);

               }

               // updateのためのパラメータ
               updateMap.put(colCd, value);

           }

       }

       logger.info(String.format("　更新対象の値セット：%s ", updateMap));
       if (!MapUtils.isEmpty(updateMap)) {
           // 更新
           updateMap.put("estimate_no", estimateNo);
           updateMap.put("record_no", Integer.toString(recordNo));
           dao.updateEditTableData(tenantId, "estimatedetail", userCd, JsonUtil.mapToJson(updateMap));
           ;
       }

   }

   /***
   *
   * 値を取得
   *
   * @param mp
   * @return
   */
  private String getStrValue(Map<String, Object> mp) {
      String value = null;
      Object paramValue = mp.get("value");
      String separator = (mp.get("separator") == null)? "," : (String) mp.get("separator");
      if (paramValue != null) {
          if (paramValue.getClass() == String[].class) {
              value = String.join(separator, (String[]) paramValue);
          } else {
              value = (String) mp.get("value");
          }
      }
      return value;
  }

   /***
   *
   * ファイルアップロード
   *
   *
   * @param dao            DAOオブジェクト
   * @param editTableCd    テーブルコード
   * @param data           見出、明細から呼び出されるデータ
   * @param extSetting     設定
   * @param extValue       値
   * @return
   */
  private String uploadFile(JsonDataDao dao, String editTableCd,
                              String data, Map<String, Object> extSetting, Map<String, Object> extValue ) {

      // --------------------------------
      // ファイル保存先基準ディレクトリの取得
      // --------------------------------
      logger.info("　　ファイル保存先基準ディレクトリの取得");
      String baseDir = "";
      try {
          String fileStorePathSystemSettingId = (String) extSetting.get("file_store_path_system_setting_id");
          String saveBaseDirPath = null;
          if (!CheckUtil.isEmpty(fileStorePathSystemSettingId)) {
              saveBaseDirPath = systemSetting.get(fileStorePathSystemSettingId);
          }
          if (CheckUtil.isEmpty(saveBaseDirPath)) {
              saveBaseDirPath = systemSetting.get("exItemFileDir_" + editTableCd);
          }
          if (CheckUtil.isEmpty(saveBaseDirPath)) {
              saveBaseDirPath = null;
          }
          baseDir = saveBaseDirPath;
      } catch (Exception e) {
      }
      logger.info(String.format("　　→ファイル保存先基準ディレクトリ：%s", baseDir));

      // 保存先ディレクトリの作成
      String saveDirPath =  baseDir;

      // ファイル名作成
      String fileName = "";
      try {

          // 拡張項目定義からファイル名を取得
          String fileNameSelectQuery = (String) extSetting.get("file_name_select_query");
          // ファイル名作成クエリが取得できた場合
          if (!CheckUtil.isEmpty(fileNameSelectQuery)) {

              // 条件の作成＆SQL実行
              // 取得したサブディレクトリにはBase64でエンコードすること
              Map<String, Object> ret = dao.selectEditTableDataFieldFromJson(tenantId, editTableCd, data, fileNameSelectQuery);
              if (!MapUtils.isEmpty(ret)) {
                  fileName = (String) ret.get("data");
              }

          }

      } catch (Exception e) {
      }

      FileUtil fileUtil = new FileUtil();

      if (StringUtils.isEmpty(fileName)) {
          fileName = (String) extValue.get("saved_ext_file_name");
      } else {
          // 拡張子を付与
          fileName = fileName + "." + fileUtil.getSuffix((String) extValue.get("saved_ext_file_name"));
      }

      if (!StringUtils.isEmpty(fileName) && !StringUtils.isEmpty(saveDirPath) && !StringUtils.isEmpty((String) extValue.get("saved_ext_full_path"))) {

          String guid = (String) JsonUtil.jsonToMap(data).get("guid");
          String colCd = (String) extSetting.get("col_cd");
          // --------------------------------
          // ファイルパス作成
          // --------------------------------
          // ファイル名部分は難読化する
          String suffix = fileUtil.getSuffix(fileName);
          // String convertedFileName = String.format("%s_%s_%s", guid, editTableCd, colCd) + "." + suffix;
          String convertedFileName = FileUtil.getAttachmentFileName(guid, editTableCd, colCd, suffix);
          String saveFilePath = saveDirPath + File.separator + convertedFileName;

          // --------------------------------
          // ファイル保存
          // --------------------------------
          File file = null;
          try {
              file = new File((String) extValue.get("saved_ext_full_path"));
              fileUtil.moveFile(file.getPath(), saveFilePath);
              logger.info(String.format("　　ファイルを保存しました　保存先：%s", saveFilePath));
          } catch (Exception e) {
              fileName = null;
          }

          // --------------------------------
          // ファイル保存テーブルにデータ登録
          // --------------------------------
          Map<String, Object> attachmentMap = new HashMap<String, Object>();
          String saveBaseDirSysId = StringUtils.defaultIfEmpty((String) extSetting.get("file_store_path_system_setting_id"), "exItemFileDir_" + editTableCd);
          String refBaseDirSysId = StringUtils.defaultIfEmpty((String) extSetting.get("file_context_path_system_setting_id"), "exItemFilePath_" + editTableCd);
          attachmentMap.put("edit_table_cd", editTableCd);
          attachmentMap.put("col_cd", colCd);
          attachmentMap.put("ref_guid", guid);
          attachmentMap.put("save_base_dir_sys_id",saveBaseDirSysId);
          attachmentMap.put("ref_base_dir_sys_id", refBaseDirSysId);
          attachmentMap.put("file_name", convertedFileName);
          attachmentMap.put("org_file_name", fileName);
          String mapStr = JsonUtil.mapToJson(attachmentMap);
          dao.deleteEditTableData(tenantId, "attachment", mapStr);
          dao.insertEditTableData(tenantId, "attachment", userCd, mapStr);
          logger.info(String.format("　　ファイル保存テーブルに保存しました　保存したデータ：%s", mapStr));

      }
      logger.info(String.format("　　ファイル名：%s", fileName));
      return fileName;

  }



}
